#include "dock.h"


// +++++++++++++++++++++++++++++++++++++++++
AG_Conformer_Search::AG_Conformer_Search()
{
    vdwA = NULL;
    vdwB = NULL;
}


// +++++++++++++++++++++++++++++++++++++++++
AG_Conformer_Search::~AG_Conformer_Search()
{
    delete[]vdwA;
    delete[]vdwB;
}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::input_parameters(Parameter_Reader & parm)
{
    string          tmp;

    // cout << "\nAnchor & Grow Parameters" << endl;
    // cout <<
    // "------------------------------------------------------------------------------------------" 
    // << endl;

    anchor_size = atoi(parm.query_param("min_anchor_size", "40").c_str());
    if (anchor_size <= 0) {
        cout <<
            "ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
            << endl;
        exit(0);
    }

    cluster = (parm.query_param("pruning_use_clustering", "yes", "yes no") == "yes") ? true : false;
    if(cluster){
        num_anchor_poses =
            atoi(parm.query_param("pruning_max_orients", "100").c_str());
        if (num_anchor_poses <= 0) {
            cout <<
                "ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
        nc = atoi(parm.query_param("pruning_clustering_cutoff", "100").c_str());
        if (nc <= 0) {
        	cout <<
            	"ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
            	<< endl;
        	exit(0);
    	}
        
        anchor_score_cutoff = 1000.0;
        num_growth_poses = INT_MAX;
        growth_cutoff = false;

    } else { 
        num_anchor_poses =
            atoi(parm.query_param("pruning_max_orients", "100").c_str());
        if (num_anchor_poses <= 0) {
            cout <<
                "ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
        anchor_score_cutoff = atof(parm.query_param("pruning_orient_score_cutoff", "25.0").c_str());
        num_growth_poses = atoi(parm.query_param("pruning_max_conformers", "75").c_str());
        growth_score_cutoff = atof(parm.query_param("pruning_conformer_score_cutoff", "25.0").c_str());
    }

    use_internal_energy =
        (parm.query_param("use_internal_energy", "yes", "yes no") ==
         "yes") ? true : false;

    if (use_internal_energy) {
        ie_att_exp =
            atoi(parm.query_param("internal_energy_att_exp", "6").c_str());
        if (ie_att_exp <= 0) {
            cout <<
                "ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
        ie_rep_exp =
            atoi(parm.query_param("internal_energy_rep_exp", "12").c_str());
        if (ie_rep_exp <= 0) {
            cout <<
                "ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
        ie_diel =
            atof(parm.query_param("internal_energy_dielectric", "4.0").c_str());
        if (ie_diel <= 0.0) {
            cout <<
                "ERROR:  Parameter must be a float greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }

    }

    use_clash_penalty =
        (parm.query_param("use_clash_overlap", "no", "yes no") ==
         "yes") ? true : false;

    if (use_clash_penalty) {
        clash_penalty = atof(parm.query_param("clash_overlap", "0.5").c_str());
        if (clash_penalty <= 0.0) {
            cout <<
                "ERROR:  Parameter must be an float greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
    }
}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::initialize()
{


    cout << "Initializing Conformer Generator Routines..." << endl;


}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::prepare_molecule(DOCKMol & mol)
{
    copy_molecule(orig, mol);

    bond_list.clear();
    atom_seg_ids.clear();
    bond_seg_ids.clear();
    anchors.clear();
    layers.clear();
    pruned_confs.clear();
    orig_segments.clear();
    layer_segments.clear();

    current_anchor = 0;

    identify_rigid_segments(mol);
    id_anchor_segments();

    anchor_positions.clear();
    anchor_positions.reserve(1000);

    delete[]vdwA;
    vdwA = NULL;
    delete[]vdwB;
    vdwB = NULL;

    vdwA = new float[mol.num_atoms];
    vdwB = new float[mol.num_atoms];

    // calculate vdwA and vdwB terms
    if (use_internal_energy) {
        for (int i = 0; i < mol.num_atoms; i++) {
            vdwA[i] =
                sqrt(mol.amber_at_well_depth[i] *
                     (ie_att_exp / (ie_rep_exp - ie_att_exp)) *
                     pow((2 * mol.amber_at_radius[i]), ie_rep_exp));
            vdwB[i] =
                sqrt(mol.amber_at_well_depth[i] *
                     (ie_rep_exp / (ie_rep_exp - ie_att_exp)) *
                     pow((2 * mol.amber_at_radius[i]), ie_att_exp));
        }
    }
}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::identify_rigid_segments(DOCKMol & mol)
{
    int             i,
                    j,
                    max_central,
                    central;
    int             a1,
                    a2,
                    a3,
                    a4;
    int             t1,
                    t2,
                    nbr;
    vector < int   >nbrs;

    BREADTH_SEARCH  bfs;

    // Identify rigid segments & which segment each atom belongs to
    for (i = 0; i < orig.num_atoms; i++)
        atom_seg_ids.push_back(-1);

    for (i = 0; i < orig.num_bonds; i++)
        bond_seg_ids.push_back(-1);

    extend_segments(0, 0, mol);

    for (i = 0; i < orig.num_bonds; i++) {

        t1 = orig.bonds_origin_atom[i];
        t2 = orig.bonds_target_atom[i];

        if (atom_seg_ids[t1] == atom_seg_ids[t2]) {
            bond_seg_ids[i] = atom_seg_ids[t1];
            orig_segments[bond_seg_ids[i]].bonds.push_back(i);
        }

    }

    // ID the inter-segment rot-bonds
    for (i = 0; i < bond_list.size(); i++) {

        a2 = orig.bonds_origin_atom[bond_list[i].bond_num];
        a3 = orig.bonds_target_atom[bond_list[i].bond_num];

        max_central = -1;
        nbrs = mol.get_atom_neighbors(a2);

        for (j = 0; j < nbrs.size(); j++) {
            nbr = nbrs[j];
            if (nbr != a3) {
                central = bfs.get_search_radius(mol, nbr, a2);
                if (central > max_central) {
                    a1 = nbr;
                    max_central = central;
                }
            }
        }

        max_central = -1;
        nbrs = mol.get_atom_neighbors(a3);

        for (j = 0; j < nbrs.size(); j++) {
            nbr = nbrs[j];
            if (nbr != a2) {
                central = bfs.get_search_radius(mol, nbr, a3);
                if (central > max_central) {
                    a4 = nbr;
                    max_central = central;
                }
            }
        }

        bond_list[i].atom1 = a1;
        bond_list[i].atom2 = a2;
        bond_list[i].atom3 = a3;
        bond_list[i].atom4 = a4;
        bond_list[i].seg1 = atom_seg_ids[bond_list[i].atom2];   // -1
        bond_list[i].seg2 = atom_seg_ids[bond_list[i].atom3];   // -1
        bond_list[i].initial_angle =
            orig.get_torsion(bond_list[i].atom1, bond_list[i].atom2,
                             bond_list[i].atom3, bond_list[i].atom4);
    }

}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::extend_segments(int atom_num, int segment, DOCKMol & mol)
{
    int             nbr;
    int             bond;
    int             i;
    vector < int   >nbrs;

    SEGMENT         tmp_segment;
    ROT_BOND        tmp_bond;

    if (atom_num == 0)
        orig_segments.push_back(tmp_segment);

    atom_seg_ids[atom_num] = segment;
    orig_segments[segment].atoms.push_back(atom_num);

    if (orig.amber_at_heavy_flag[atom_num])
        orig_segments[segment].num_hvy_atoms++;

    nbrs = orig.get_atom_neighbors(atom_num);

    for (i = 0; i < nbrs.size(); i++) {

        nbr = nbrs[i];
        bond = orig.get_bond(atom_num, nbr);

        if (atom_seg_ids[nbr] == -1) {

            if (mol.bond_is_rotor(bond)) {

                bond_list.push_back(tmp_bond);
                bond_list[bond_list.size() - 1].bond_num = bond;

                orig_segments.push_back(tmp_segment);

                orig_segments[segment].neighbors.push_back(orig_segments.
                                                           size() - 1);
                orig_segments[segment].neighbor_bonds.push_back(bond_list.
                                                                size() - 1);
                orig_segments[segment].neighbor_atoms.push_back(nbr);

                orig_segments[orig_segments.size() -
                              1].neighbors.push_back(segment);
                orig_segments[orig_segments.size() -
                              1].neighbor_bonds.push_back(bond_list.size() - 1);
                orig_segments[orig_segments.size() -
                              1].neighbor_atoms.push_back(atom_num);

                extend_segments(nbr, orig_segments.size() - 1, mol);

            } else {

                extend_segments(nbr, segment, mol);

            }

        }

    }

    
}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::id_anchor_segments()
{
    int             i;
    INTPair         tmp;

    for (i = 0; i < orig_segments.size(); i++) {
        tmp.first = orig_segments[i].num_hvy_atoms + orig_segments[i].neighbors.size(); // problem 
                                                                                        // with 
                                                                                        // anchor 
                                                                                        // defs 
                                                                                        // this 
                                                                                        // way
        tmp.second = i;
        anchors.push_back(tmp);
    }

    sort(anchors.begin(), anchors.end());
    reverse(anchors.begin(), anchors.end());

}


// +++++++++++++++++++++++++++++++++++++++++
bool
AG_Conformer_Search::next_anchor(DOCKMol & mol)
{
    DOCKMol         tmp_mol,
                    blank_mol;
    CONFORMER       tmp_conf;
    vector < CONFORMER > conf_vec;
    int             i;



    // If there are no structs in anchor_confs, generate a new anchor and
    // expand it
    if (anchor_confs.size() == 0) {

        // if there are no more anchors, or you have reached the end of the
        // list
        if ((anchors.size() == 0) || (current_anchor == anchors.size())
            || ((anchors[current_anchor].first < anchor_size)
                && (current_anchor > 0)))
            return false;

        copy_molecule(tmp_mol, orig);

        // clear bond directionality during minization
        bond_tors_vectors.clear();
        bond_tors_vectors.resize(mol.num_bonds, -1);

        // reset the assigned atoms list
        assigned_atoms.clear();
        assigned_atoms.resize(mol.num_atoms, false);

        // clear and resize the layer_segment list to be the same size as the
        // orig_segments list
        layer_segments.clear();
        layer_segments.resize(orig_segments.size());

        layers.clear();
        extend_layers(anchors[current_anchor].second,
                      anchors[current_anchor].second, 0);

        // cout << "@@@\t" << orig_segments.size() << "\t" << layers.size() <<
        // "\t" << layer_segments.size() << endl;

        for (i = 0; i < layers[0].segments.size(); i++) {
            activate_layer_segment(tmp_mol, 0, i);
        }

        current_anchor++;

        // copy the anchor to the anchor_conf list
        anchor_confs.clear();
        anchor_confs.resize(anchor_confs.size() + 1);
        copy_molecule(anchor_confs[anchor_confs.size() - 1], tmp_mol);

    }

/**
	int i,j,k;
	for(i=0;i<layers.size();i++) {
		for(j=0;j<layers[i].segments.size();j++) {
			for(k=0;k<layer_segments[layers[i].segments[j]].atoms.size();k++) {
				cout << i << "\t" << j << "\t" << layer_segments[layers[i].segments[j]].atoms[k]+1 << endl;
			}
		}
	}
**/

    // copy last mol from anchor_conf to mol
    copy_molecule(mol, anchor_confs[anchor_confs.size() - 1]);
    anchor_confs.pop_back();

    if (anchor_confs.size() > 0)
        last_conformer = false;
    else
        last_conformer = true;

    return true;


}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::extend_layers(int previous_segment, int current_segment,
                                   int current_layer)
{
    int             i;
    INTPair         tmp_pair;
    LAYER           tmp_layer;
    LAYER_SEGMENT   tmp_segment;

    if (current_layer >= layers.size())
        layers.push_back(tmp_layer);

    layers[current_layer].segments.push_back(current_segment);
    layers[current_layer].num_segments = layers[current_layer].segments.size();

    // loop over the atoms in each orig_segment and add them to the
    // layer_segment
    // if the atom has not already been assigned to a previous segment
    for (i = 0; i < orig_segments[current_segment].atoms.size(); i++) {
        if (!assigned_atoms[orig_segments[current_segment].atoms[i]]) {
            layer_segments[current_segment].atoms.
                push_back(orig_segments[current_segment].atoms[i]);
            assigned_atoms[orig_segments[current_segment].atoms[i]] = true;
        }
    }

    // copy the bonds from orig_segment to layer_segment
    layer_segments[current_segment].bonds =
        orig_segments[current_segment].bonds;

    for (i = 0; i < orig_segments[current_segment].neighbors.size(); i++) {

        if (orig_segments[current_segment].neighbors[i] != previous_segment) {
            // if this bond leads to the next layer, then add the leading atom
            // and bond to the current segment

            layer_segments[current_segment].bonds.
                push_back(bond_list
                          [orig_segments[current_segment].neighbor_bonds[i]].
                          bond_num);

            if (!assigned_atoms
                [orig.
                 bonds_origin_atom[bond_list
                                   [orig_segments[current_segment].
                                    neighbor_bonds[i]].bond_num]]) {
                layer_segments[current_segment].atoms.push_back(orig.
                                                                bonds_origin_atom
                                                                [bond_list
                                                                 [orig_segments
                                                                  [current_segment].
                                                                  neighbor_bonds
                                                                  [i]].
                                                                 bond_num]);
                assigned_atoms[orig.
                               bonds_origin_atom[bond_list
                                                 [orig_segments
                                                  [current_segment].
                                                  neighbor_bonds[i]].
                                                 bond_num]] = true;

                // set bond direction vector
                bond_tors_vectors[bond_list
                                  [orig_segments[current_segment].
                                   neighbor_bonds[i]].bond_num] =
                    orig.
                    bonds_target_atom[bond_list
                                      [orig_segments[current_segment].
                                       neighbor_bonds[i]].bond_num];
            }

            if (!assigned_atoms
                [orig.
                 bonds_target_atom[bond_list
                                   [orig_segments[current_segment].
                                    neighbor_bonds[i]].bond_num]]) {
                layer_segments[current_segment].atoms.push_back(orig.
                                                                bonds_target_atom
                                                                [bond_list
                                                                 [orig_segments
                                                                  [current_segment].
                                                                  neighbor_bonds
                                                                  [i]].
                                                                 bond_num]);
                assigned_atoms[orig.
                               bonds_target_atom[bond_list
                                                 [orig_segments
                                                  [current_segment].
                                                  neighbor_bonds[i]].
                                                 bond_num]] = true;

                // set bond direction vector
                bond_tors_vectors[bond_list
                                  [orig_segments[current_segment].
                                   neighbor_bonds[i]].bond_num] =
                    orig.
                    bonds_origin_atom[bond_list
                                      [orig_segments[current_segment].
                                       neighbor_bonds[i]].bond_num];
            }

            extend_layers(current_segment,
                          orig_segments[current_segment].neighbors[i],
                          current_layer + 1);

        } else {
            // if this bond comes from the previous layer, then add it as the
            // rot_bond in the segment

            layer_segments[current_segment].rot_bond =
                orig_segments[current_segment].neighbor_bonds[i];
            layer_segments[current_segment].origin_segment = previous_segment;
        }
    }

}


// +++++++++++++++++++++++++++++++++++++++++
bool
AG_Conformer_Search::submit_anchor_orientation(DOCKMol & mol, bool more_orients)
{
    SCOREMol        tmp_mol;
    float           mol_score;
    int             insert_point;
    int             i;


    mol_score = mol.current_score;

    // Loop over existing list of anchors
    insert_point = -1;
    for (i = 0; i < anchor_positions.size(); i++) {

        // finds the first instance of a lower score than the mol, and selects
        // that as the insert point
        if (insert_point == -1) {
            if (anchor_positions[i].first > mol_score)
                insert_point = i;
        }

    }

    //insert mol into list, and pop last member off
    if (insert_point == -1) {
        // if insert_point = -1 (mol to be appended to end of list)
        // only append mol if list is less than nc
        if (anchor_positions.size() < num_anchor_poses) {
        	anchor_positions.push_back(tmp_mol);
                copy_molecule(anchor_positions[anchor_positions.size() - 1].
                              second, mol);
                anchor_positions[anchor_positions.size() - 1].first = mol_score;
	}

    } else {
        // if molecule is to be inserted within the list

        // if list is smaller than limit, append a blank mol on the end
        if (anchor_positions.size() < num_anchor_poses) {
            anchor_positions.push_back(tmp_mol);
        }
        // start at end of list, and shift all mols one spot to the end of
        // the list, until the insert point is reached
        for (i = anchor_positions.size() - 1; i > insert_point; i--) {
            copy_molecule(anchor_positions[i].second,
                              anchor_positions[i - 1].second);
            anchor_positions[i].first = anchor_positions[i - 1].first;
        }

        // copy mol into insert point
        copy_molecule(anchor_positions[insert_point].second, mol);
        anchor_positions[insert_point].first = mol_score;
   
    }

    // enter into loop if no more orients...
    if (more_orients)
        return false;
    else if (last_conformer) {
        sort(anchor_positions.begin(), anchor_positions.end(), less_than_pair);
        return true;
    } else
        return false;



}


// +++++++++++++++++++++++++++++++++++++++++
float
AG_Conformer_Search::calc_layer_rmsd(CONFORMER & a, CONFORMER & b)
{
    int             i,
                    j,
                    k;
    int             atom;
    float           rmsd;
    int             heavy_total;

    rmsd = 0.0;
    heavy_total = 0;

    for (i = 0; i < layers.size(); i++) {
        for (j = 0; j < layers[i].segments.size(); j++) {
            for (k = 0; k < layer_segments[layers[i].segments[j]].atoms.size();
                 k++) {

                atom = layer_segments[layers[i].segments[j]].atoms[k];

                if (a.structure.atom_active_flags[atom]
                    && a.structure.amber_at_heavy_flag[atom]) {

                    rmsd +=
                        (pow((a.structure.x[atom] - b.structure.x[atom]), 2) +
                         pow((a.structure.y[atom] - b.structure.y[atom]),
                             2) + pow((a.structure.z[atom] -
                                       b.structure.z[atom]), 2)) * (i + 1);

                    heavy_total += i + 1;
                }
            }
        }
    }

    if (heavy_total > 0)
        rmsd = sqrt(rmsd / (float) heavy_total);
    else
        rmsd = 0.0;

    return rmsd;
}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::grow_periphery(Master_Score & score,
                                    Simplex_Minimizer & simplex, Bump_Filter & bump)
{
    int             i,
                    j,
                    k,
                    l;

    CONFORMER       tmp_conf;
    SCOREMol        tmp_scoremol;

    vector < CONFORMER > seeds, exp_seeds;
    float           rmsd;

    // initialize
    bool valid_orient = false;
    int num_layers = layers.size();
    float RMSD_CUTOFF = (float) nc;

    // /////////////////////////// ANCHORS //////////////////////////////
    // add anchors to seed list

    for (i = 0; i < anchor_positions.size(); i++) {

        // compute the score for the molecule
        if (score.use_primary_score) {
            valid_orient =
                score.compute_primary_score(anchor_positions[i].second);
        } else {
            valid_orient = true;
        }

        if (valid_orient) {
            exp_seeds.push_back(tmp_conf);
            exp_seeds[exp_seeds.size() - 1].layer_num = 1;
            exp_seeds[exp_seeds.size() - 1].score =
                anchor_positions[i].second.current_score;
            exp_seeds[exp_seeds.size() - 1].used = false;
            copy_molecule(exp_seeds[exp_seeds.size() - 1].structure,
                          anchor_positions[i].second);
        }
    }

    // Pruning section
    seeds.clear();
    sort(exp_seeds.begin(), exp_seeds.end(), conformer_less_than);

    // filter confs by scores
    for (j = 0; j < exp_seeds.size(); j++) {
        if (exp_seeds[j].score > anchor_score_cutoff)
            exp_seeds[j].used = true;
    }

    // remove confs that fail the rank/rmsd test
    if(cluster){
    	for (j = 0; j < exp_seeds.size(); j++) {
       	    if (!exp_seeds[j].used) {
            	for (k = j + 1; k < exp_seeds.size(); k++) {
                    if (!exp_seeds[k].used) {

                    	rmsd = calc_layer_rmsd(exp_seeds[j], exp_seeds[k]);
                    	rmsd = MAX(rmsd, 0.001);

                    	if ((float) k / rmsd > RMSD_CUTOFF) {
                        	exp_seeds[k].used = true;
                    	}

                    }	
            	}
            }
        }
    }

    // End pruning section

    // copy pruned confs to seed list
    for (j = 0; j < exp_seeds.size(); j++) {
        if (!exp_seeds[j].used)
            seeds.push_back(exp_seeds[j]);
    }
    // /////////////////////////// END ANCHORS //////////////////////////

    // /////////////////////////// GROWTH ///////////////////////////////
    // loop over layers

    for (i = 1; i < num_layers; i++) {  // 

        // loop over rot bonds in segment
        for (l = 0; l < layers[i].segments.size(); l++) {

            exp_seeds.clear();

            // loop over seed conformers
            for (j = 0; j < seeds.size(); j++) {

                // generate and loop over new confs
                segment_torsion_drive(seeds[j], l, exp_seeds);

                // score/minimize the new confs
                for (k = 0; k < exp_seeds.size(); k++) {

                    if (segment_clash_check(exp_seeds[k].structure, i, l)) {    // i+1
	                
                        if(bump.check_growth_bumps(exp_seeds[k].structure)) {

                             simplex.minimize_flexible_growth(exp_seeds[k].structure,
                                                         score,
                                                         bond_tors_vectors);

                            // compute the score for the molecule
                            if (score.use_primary_score)
                                valid_orient =
                                    score.compute_primary_score(exp_seeds[k].
                                                            structure);
                            else
                                valid_orient = true;


                            if (valid_orient)
                                exp_seeds[k].score = exp_seeds[k].structure.current_score + segment_internal_energy(exp_seeds[k].structure, i, l);  // i+1
                            else {
                                exp_seeds[k].used = true;
                            }
                        } else {

                            exp_seeds[k].used = true;

                        }

                    } else {

                        exp_seeds[k].used = true;

                    }

                }

            }
            // cout << endl;

            // Pruning section
            seeds.clear();
            sort(exp_seeds.begin(), exp_seeds.end(), conformer_less_than);

            // remove confs with very bad scores
           for(j=0;j<exp_seeds.size();j++) { 
	        if(j<num_growth_poses){
			if(growth_cutoff){
	  	            if(exp_seeds[j].score > growth_score_cutoff){
			        exp_seeds[j].used = true; 
                	    }
			}
                }else{
                    exp_seeds[j].used = true;
                }
           } 

	    
            // remove confs that fail the rank/rmsd test
	    if(cluster){
                for (j = 0; j < exp_seeds.size(); j++) {
                    if (!exp_seeds[j].used) {
                        for (k = j + 1; k < exp_seeds.size(); k++) {
                            if (!exp_seeds[k].used) {

                                rmsd = calc_layer_rmsd(exp_seeds[j], exp_seeds[k]);
                                rmsd = MAX(rmsd, 0.001);

                                if ((float) k / rmsd > RMSD_CUTOFF) {
                                    exp_seeds[k].used = true;
                                }

                            }
                        }
                    }
                }
             }
	     
            // End pruning section

            // copy pruned confs to seed list
            for (j = 0; j < exp_seeds.size(); j++) {
                if (!exp_seeds[j].used) {
                    seeds.push_back(exp_seeds[j]);
                    // score.compute_primary_score(exp_seeds[j].structure);
                }
            }

        }                       // End loop over segments

    }
    // ///////////////////////// END GROWTH /////////////////////////////

    // copy mols to pruned confs
    pruned_confs.clear();
    for (i = 0; i < seeds.size(); i++) {
        pruned_confs.push_back(tmp_scoremol);
        pruned_confs[i].first = seeds[i].score;
        copy_molecule(pruned_confs[i].second, seeds[i].structure);
    }

}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::segment_torsion_drive(CONFORMER & conf, int current_bond,
                                           vector < CONFORMER > &return_list)
{
    int             num_torsions,
                    bond;
    CONFORMER       new_conf,
                    tmp_conf;
    float           new_angle;
    int             seg_id;

    // bond =
    // layer_segments[layers[conf.layer_num-1].segments[current_bond]].rot_bond;
    bond = layer_segments[layers[conf.layer_num].segments[current_bond]].rot_bond;      // -1

    num_torsions =
        conf.structure.amber_bt_torsion_total[bond_list[bond].bond_num];

    for (int i = 0; i < num_torsions; i++) {

        copy_molecule(new_conf.structure, conf.structure);
        new_conf.layer_num = conf.layer_num;
        new_conf.score = conf.score;
        new_conf.used = conf.used;

        new_angle =
            new_conf.structure.amber_bt_torsions[bond_list[bond].bond_num][i];
        new_angle = (PI / 180) * new_angle;

        if (bond_list[bond].seg1 == layer_segments[layers[new_conf.layer_num].segments[current_bond]].origin_segment) { // -1
            new_conf.structure.set_torsion(bond_list[bond].atom1,
                                           bond_list[bond].atom2,
                                           bond_list[bond].atom3,
                                           bond_list[bond].atom4, new_angle);
            seg_id = bond_list[bond].seg2;
        }

        if (bond_list[bond].seg2 == layer_segments[layers[new_conf.layer_num].segments[current_bond]].origin_segment) { // -1
            new_conf.structure.set_torsion(bond_list[bond].atom4,
                                           bond_list[bond].atom3,
                                           bond_list[bond].atom2,
                                           bond_list[bond].atom1, new_angle);
            seg_id = bond_list[bond].seg1;
        }

        if (layer_segments[layers[new_conf.layer_num].segments[current_bond]].origin_segment != bond_list[bond].seg1)   // -1
            if (layer_segments[layers[new_conf.layer_num].segments[current_bond]].origin_segment != bond_list[bond].seg2)       // -1
                cout << "Layer growth error!" << endl;

        activate_layer_segment(new_conf.structure, new_conf.layer_num, current_bond);   // -1

        // Write_Mol2(new_conf.structure,
        // cout);////////////////////////////////////////////////////////////////

        if (current_bond == layers[new_conf.layer_num].segments.size() - 1) {   // -1
            new_conf.layer_num++;
        }

        return_list.push_back(tmp_conf);        // ///////////////////////////////////////////////////////////////
        return_list[return_list.size() - 1].layer_num = new_conf.layer_num;
        return_list[return_list.size() - 1].score = new_conf.score;
        return_list[return_list.size() - 1].used = new_conf.used;
        copy_molecule(return_list[return_list.size() - 1].structure,
                      new_conf.structure);

    }

}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::activate_layer_segment(DOCKMol & mol, int layer_num,
                                            int curr_seg)
{
    int             i,
                    j,
                    k;
    int             seg_num;

    reset_active_lists(mol);

    // activate all layers but last
    for (i = 0; i < layer_num; i++) {
        for (j = 0; j < layers[i].segments.size(); j++) {
            for (k = 0; k < layer_segments[layers[i].segments[j]].atoms.size();
                 k++) {
                mol.atom_active_flags[layer_segments[layers[i].segments[j]].
                                      atoms[k]] = true;
            }

            for (k = 0; k < layer_segments[layers[i].segments[j]].bonds.size();
                 k++) {
                mol.bond_active_flags[layer_segments[layers[i].segments[j]].
                                      bonds[k]] = true;
            }
        }
    }

    // activate last layer up to current segment
    for (i = 0; i <= curr_seg; i++) {

        seg_num = layers[layer_num].segments[i];

        for (j = 0; j < layer_segments[seg_num].atoms.size(); j++)
            mol.atom_active_flags[layer_segments[seg_num].atoms[j]] = true;

        for (j = 0; j < layer_segments[seg_num].bonds.size(); j++)
            mol.bond_active_flags[layer_segments[seg_num].bonds[j]] = true;

    }


    mol.num_active_atoms = 0;
    mol.num_active_bonds = 0;

    for (i = 0; i < mol.num_atoms; i++)
        if (mol.atom_active_flags[i])
            mol.num_active_atoms++;

    for (i = 0; i < mol.num_bonds; i++)
        if (mol.bond_active_flags[i])
            mol.num_active_bonds++;

}


// +++++++++++++++++++++++++++++++++++++++++
void
AG_Conformer_Search::reset_active_lists(DOCKMol & mol)
{
    int             i;

    for (i = 0; i < mol.num_atoms; i++)
        mol.atom_active_flags[i] = 0;

    for (i = 0; i < mol.num_bonds; i++)
        mol.bond_active_flags[i] = 0;

    mol.num_active_atoms = 0;
    mol.num_active_bonds = 0;

}


// +++++++++++++++++++++++++++++++++++++++++
bool
AG_Conformer_Search::segment_clash_check(DOCKMol & conf, int layer_num,
                                         int current_bond)
{
    bool            skip_flag;
    int             a1,
                    a2;
    float           distance,
                    ref;

    int             i,
                    j,
                    k,
                    l;
    int             seg_num,
                    new_seg;

    // perform clash check
    skip_flag = false;

    if (use_clash_penalty) {

        // ID current segment
        seg_num = layers[layer_num].segments[current_bond];

        // loop over atoms in newest segment
        for (i = 0; i < layer_segments[seg_num].atoms.size(); i++) {
            a1 = layer_segments[seg_num].atoms[i];

            // loop from last layer to all inner layers
            for (j = layer_num; j >= 0; j--) {

                // loop over segments in current layer
                for (k =
                     ((j ==
                       layer_num) ? (current_bond -
                                     1) : (layers[j].segments.size() - 1));
                     k >= 0; k--) {

                    new_seg = layers[j].segments[k];

                    // loop over atoms in this segment
                    for (l = 0; l < layer_segments[new_seg].atoms.size(); l++) {
                        a2 = layer_segments[new_seg].atoms[l];

                        if ((conf.get_bond(a1, a2) == -1)
                            && (!conf.atoms_are_one_three(a1, a2))) {

                            distance =
                                pow((conf.x[a1] - conf.x[a2]),
                                    2) + pow((conf.y[a1] - conf.y[a2]),
                                             2) + pow((conf.z[a1] - conf.z[a2]),
                                                      2);
                            ref =
                                clash_penalty * (conf.amber_at_radius[a1] +
                                                 conf.amber_at_radius[a2]);

                            if ((!conf.atom_active_flags[a1])
                                || (!conf.atom_active_flags[a2]))
                                cout << "Layer ERROR!!!!" << endl;

                            if (distance < ref * ref) {
                                skip_flag = true;
                            }

                        }

                    }

                }

            }

        }

    }
    // END CLASH CHECK

    return !skip_flag;
}


// +++++++++++++++++++++++++++++++++++++++++
float
AG_Conformer_Search::segment_internal_energy(DOCKMol & mol, int layer_num,
                                             int current_bond)
{
    // int m, n, o, p, l1;
    int             a1,
                    a2;
    double          distance;

    int             i,
                    j,
                    k,
                    l;
    int             seg_num,
                    new_seg;

    float           total_energy,
                    total_vdw,
                    total_es;

    total_energy = total_vdw = total_es = 0.0;

    // perform internal energy calculation
    if (use_internal_energy) {

        // ID current segment
        seg_num = layers[layer_num].segments[current_bond];

        // loop over atoms in newest segment
        for (i = 0; i < layer_segments[seg_num].atoms.size(); i++) {
            a1 = layer_segments[seg_num].atoms[i];

            // loop from last layer to all inner layers
            for (j = layer_num; j >= 0; j--) {

                // loop over segments in current layer
                for (k =
                     ((j ==
                       layer_num) ? (current_bond -
                                     1) : (layers[j].segments.size() - 1));
                     k >= 0; k--) {

                    new_seg = layers[j].segments[k];

                    // loop over atoms in this segment
                    for (l = 0; l < layer_segments[new_seg].atoms.size(); l++) {
                        a2 = layer_segments[new_seg].atoms[l];

                        if ((mol.get_bond(a1, a2) == -1)
                            && (!mol.atoms_are_one_three(a1, a2))
                            && (!mol.atoms_are_one_four(a1, a2))) {

                            distance =
                                pow((mol.x[a1] - mol.x[a2]),
                                    2) + pow((mol.y[a1] - mol.y[a2]),
                                             2) + pow((mol.z[a1] - mol.z[a2]),
                                                      2);

                            if (distance < 1.0)
                                distance = 1.0;

                            distance = sqrt(distance);

                            total_vdw +=
                                (vdwA[a1] * vdwA[a2]) / pow(distance,
                                                            (ie_rep_exp)) -
                                (vdwB[a1] * vdwB[a2]) / pow(distance,
                                                            (ie_att_exp));
                            total_es +=
                                (mol.charges[a1] * mol.charges[a2] * ie_diel) /
                                distance;

                        }
                    }
                }
            }
        }

        // total_energy = total_vdw + total_es;
        // cout << "@@@\t" << total_energy << endl;

    }
    // END INTERNAL ENERGY

    total_energy = total_vdw + total_es;

    return total_energy;

}


// +++++++++++++++++++++++++++++++++++++++++
bool
AG_Conformer_Search::next_conformer(DOCKMol & mol)
{



    if (pruned_confs.size() > 0) {
        copy_molecule(mol, pruned_confs[pruned_confs.size() - 1].second);
        pruned_confs.pop_back();
        return true;
    } else {
        return false;
    }



}


// +++++++++++++++++++++++++++++++++++++++++
// +++++++++++++++++++++++++++++++++++++++++
// +++++++++++++++++++++++++++++++++++++++++


// +++++++++++++++++++++++++++++++++++++++++
void
Master_Conformer_Search::input_parameters(Parameter_Reader & parm)
{
    cout << "\nFlexible Ligand Parameters" << endl;
    cout <<
        "------------------------------------------------------------------------------------------"
        << endl;

    use_internal_energy = false;

    flexible_ligand =
        parm.query_param("flexible_ligand", "yes", "yes no") == "yes";

    method = 0;                 // assume rigid ligand

    if (flexible_ligand) {

        // present anchor and grow options
        // bool ag_search = false;
        // ag_search = (parm.query_param("ag_conf_search", "yes", "yes no") ==
        // "yes") ? true : false;
        // if(ag_search){
        method = 1;
        c_ag_conf.input_parameters(parm);
        use_internal_energy = c_ag_conf.use_internal_energy;
        // }

                /**** Currently Disabled ****
		// present the HDB option
		if(method == 0) {
			c_hdb_conf.input_parameters(parm);

			if(c_hdb_conf.flexible_ligand)
				method = 2;
		}
		****************************/

        // if no method selected, fail
        // if(method == 0) {
        // cout << "Conf Search Error:\tno search method selected." << endl;
        // exit(0);
        // }

    }
}


// +++++++++++++++++++++++++++++++++++++++++
void
Master_Conformer_Search::initialize()
{

    switch (method) {

    case 0:
        break;

    case 1:
        c_ag_conf.initialize();
        break;

        // case 2:
        // c_hdb_conf.initialize();
        // break;

    }

}


// +++++++++++++++++++++++++++++++++++++++++
void
Master_Conformer_Search::prepare_molecule(DOCKMol & mol)
{

    switch (method) {

    case 0:
        more_anchors = true;
        last_conformer = true;
        break;

    case 1:
        c_ag_conf.prepare_molecule(mol);
        break;

        // case 2:
        // c_hdb_conf.prepare_molecule(mol);
        // break;

    }

}


// +++++++++++++++++++++++++++++++++++++++++
bool
Master_Conformer_Search::next_anchor(DOCKMol & mol)
{
    bool            return_val;

    switch (method) {

    case 0:
        if (more_anchors) {
            more_anchors = false;
            return_val = true;
        } else
            return_val = false;
        break;

    case 1:
        return_val = c_ag_conf.next_anchor(mol);
        break;

        // case 2:
        // return_val = c_hdb_conf.next_anchor(mol);
        // break;

    default:
        throw;
        break;
    }

    return return_val;
}


// +++++++++++++++++++++++++++++++++++++++++
bool
Master_Conformer_Search::submit_anchor_orientation(DOCKMol & mol,
                                                   bool more_orients)
{
    bool            return_val;

    switch (method) {

    case 0:
        if (more_orients) {
            more_anchors = true;
            last_conformer = true;
            return_val = true;
        } else if (last_conformer) {
            more_anchors = true;
            last_conformer = false;
            return_val = true;
        } else
            return_val = false;
        break;

    case 1:
        return_val =
            c_ag_conf.submit_anchor_orientation(mol, more_orients);
        break;

        // case 2:
        // return_val = c_hdb_conf.submit_anchor_orientation(mol, more_orients, 
        // score);
        // break;

    default:
        throw;
        break;
    }

    return return_val;
}


// +++++++++++++++++++++++++++++++++++++++++
void
Master_Conformer_Search::grow_periphery(Master_Score & score,
                                        Simplex_Minimizer & simplex, Bump_Filter & bump)
{

    switch (method) {

    case 0:
        break;

    case 1:
        c_ag_conf.grow_periphery(score, simplex, bump);
        break;

        // case 2:
        // c_hdb_conf.grow_periphery(score, bump);
        // break;

    }

}


// +++++++++++++++++++++++++++++++++++++++++
bool
Master_Conformer_Search::next_conformer(DOCKMol & mol)
{
    bool            return_val;

    switch (method) {

    case 0:
        if (more_anchors) {
            more_anchors = false;
            last_conformer = true;
            return_val = true;
        } else
            return_val = false;
        break;

    case 1:
        return_val = c_ag_conf.next_conformer(mol);
        break;

        // case 2:
        // return_val = c_hdb_conf.next_conformer(mol);
        // break;

    default:
        throw;
        break;
    }

    return return_val;
}


// +++++++++++++++++++++++++++++++++++++++++
// +++++++++++++++++++++++++++++++++++++++++
// +++++++++++++++++++++++++++++++++++++++++


#ifdef HDB_IS_NOT_YET_COMPLETELY_IMPLEMENTED
// +++++++++++++++++++++++++++++++++++++++++
void HDB_Conformer_Search::input_parameters(Parameter_Reader &parm) {
	string	tmp;

	cout << "\nHierarchy DB Search Parameters" << endl;
	cout << "------------------------------------------------------------------------------------------" << endl;

	tmp = parm.query_param("hdb_conf_search","no","yes no");

	if(tmp == "yes")
		flexible_ligand = true;
	else
		flexible_ligand = false;

}


// +++++++++++++++++++++++++++++++++++++++++
void HDB_Conformer_Search::initialize() {

}


// +++++++++++++++++++++++++++++++++++++++++
void HDB_Conformer_Search::prepare_molecule(DOCKMol &mol) {
	int				total_atoms;
	int				i, j, k, index;
	string			line1, line2;
	int				current_level, current_value, current_count;

	INTVec			conf_state;

	if(flexible_ligand) {

		// copy molecule to total_mol
		copy_molecule(total_mol, mol);

		// read the hierarchy headers
		line1 = mol.mol_data.substr(0, mol.mol_data.find("\n"));
		family_line = line1;
		line2 = mol.mol_data.substr((mol.mol_data.find("\n")+1), mol.mol_data.size());

		line1 = line2.substr(0, line2.find("\n"));
		name_line = line1;
		line2 = line2.substr((line2.find("\n")+1), line2.size());

		branch_lines.clear();
		level_lines.clear();

		while(line2 != "") {
			line1 = line2.substr(0, line2.find("\n"));
			branch_lines.push_back(line1);
			line2 = line2.substr((line2.find("\n")+1), line2.size());

			line1 = line2.substr(0, line2.find("\n"));
			level_lines.push_back(line1);
			line2 = line2.substr((line2.find("\n")+1), line2.size());
		}

		// extract values from hierarchy strings
		sscanf(family_line.c_str(), "%*s %d %d %d %d", &clunum, &nmol, &nbr, &fbr);

		name = name_line.substr(0,47);
		refcode = name_line.substr(47, 56);

		branches.resize(branch_lines.size());

		for(i=0;i<branch_lines.size();i++) {

			branches[i].level_counts.clear();
			branches[i].level_values.clear();

			sscanf(branch_lines[i].c_str(), "%d %d %d %d %f %d %f %d %d", &branches[i].beatm, &branches[i].bmatm, &branches[i].bnhvy, &branches[i].bnhyd, &branches[i].solvat, &branches[i].confnum, &branches[i].apol, &branches[i].bi, &branches[i].iconf);

			line2 = level_lines[i];

			while(line2 != "") {
				line1 = line2.substr(0, line2.find(","));
				line2 = line2.substr((line2.find(",")+1), line2.size());

				sscanf(line1.c_str(), "%d %d", &level_count, &level_value);

				branches[i].level_counts.push_back(level_count);
				branches[i].level_values.push_back(level_value);
			}

		}

		// initialize search structures
		atoms.resize(mol.num_atoms);

		// loop over all branches
		index = 0;

		for(i=0;i<branches.size();i++) {

			// init counters
			current_level = 0;
			current_count = 0;

			conf_state.clear();
			conf_state.resize(branches[i].level_values.size(), -1);
			conf_state[0] = 0;

			// loop over all atoms in each branch
			for(j=0;j<branches[i].beatm;j++) {

				// init branch & level data structs
				atoms[index].branch_num = i;
				atoms[index].level_confs.resize(branches[i].level_counts.size(), -1);

				// read atom_data text info into atom_info class
				sscanf(mol.atom_data[index].c_str(), "%d %d %f %d %d %f %f", &(atoms[index].level), &(atoms[index].vdwtype), &(atoms[index].charge), &(atoms[index].flagat), &(atoms[index].lcolor), &(atoms[index].polsolv), &(atoms[index].apolsolv));

				// scan for level changes
				if(j > 0) {

					// if level has changed
					if(atoms[index].level != atoms[index-1].level) {

						current_count = 0;
						current_level = 0;

						for(k=0;k<branches[i].level_values.size();k++) {
							if(atoms[index].level == branches[i].level_values[k]) {
								current_level = k;
								break;
							}
						}

						// increment state vector at new level
						conf_state[current_level]++;

						// reset higher levels
						for(k=current_level+1;k<conf_state.size();k++)
							conf_state[k] = -1;

					}

				} // end scan for level changes

				// scan for multiple confs in same level
				if(current_count >= branches[i].level_counts[current_level]) {
					conf_state[current_level]++;
					current_count = 0;
				}

				atoms[index].level_confs = conf_state;

				current_count++;
				index++;

			}

		}

		// init rest of data structs
		anchor_positions.clear();
		anchor_positions.reserve(1000);

	}

	return_anchor_value = true;
	return_periph_value = true;

}


// +++++++++++++++++++++++++++++++++++++++++
bool HDB_Conformer_Search::next_anchor(DOCKMol &mol) {
	int		i;

	// set flags to end next_anchor loop
	if(return_anchor_value == true) {
		return_anchor_value = false;
	} else
		return false;

	if(flexible_ligand) {

		// copy molecule from total_mol to mol
		copy_molecule(mol, total_mol);

		// activate just the anchor atoms
		mol.num_active_atoms = 0;

		for(i=0;i<mol.num_atoms;i++)
			mol.atom_active_flags[i] = false;

		for(i=0;i<branches[0].bmatm;i++) {
			mol.atom_active_flags[i] = true;
			mol.num_active_atoms++;
		}

	}

	return true;

}


// +++++++++++++++++++++++++++++++++++++++++
bool HDB_Conformer_Search::submit_anchor_orientation(DOCKMol &mol, bool more_orients, Master_Score &score) {
  SCOREMol	tmp_mol;
  float		mol_score;
  int		insert_point;
  bool		rmsd_exclude_flag;
  bool		valid_orient = false;
  int		i;

  if(!flexible_ligand) {

	  return true;

  } else {

	  // compute the score for the molecule
	  if(score.use_primary_score) {
		  valid_orient = score.compute_primary_score(mol);
		  mol_score = mol.current_score;
	  } else {
		  valid_orient = true;
		  mol_score = 0.0;
	  }

	  if(!valid_orient)
		return false;

	  // Loop over existing list of anchors
	  insert_point = -1;
	  rmsd_exclude_flag = false;

	  // code to simply process all anchor positions
	  tmp_mol.first = mol_score;
	  copy_molecule(tmp_mol.second, mol);
	  anchor_positions.push_back(tmp_mol);

/*****************
	  for(i=0;i<anchor_positions.size();i++) {

		  // finds the first instance of a lower score than the mol, and selects that as the insert point
		  if(insert_point == -1) {
			  if(anchor_positions[i].first > mol_score)
				  insert_point = i;
		  }

	  }

	  // If rmsd_exclude_flag = false, then insert mol into list, and pop last member off
	  if(!rmsd_exclude_flag) {

		  if(insert_point == -1) {
			  // if insert_point = -1 (mol to be appended to end of list)
			  // only append mol if list is less than nc
			  if(anchor_positions.size() < num_anchor_poses) {
				  anchor_positions.push_back(tmp_mol);
				  copy_molecule(anchor_positions[anchor_positions.size()-1].second, mol);
				  anchor_positions[anchor_positions.size()-1].first = mol_score;
			  }

		  } else {
			  // if molecule is to be inserted within the list

			  // if list is smaller than limit, append a blank mol on the end
			  if(anchor_positions.size() < num_anchor_poses) {
				  anchor_positions.push_back(tmp_mol);
			  }

			  // start at end of list, and shift all mols one spot to the end of the list, until the insert point is reached
			  for(i=anchor_positions.size()-1;i>insert_point;i--) {
				  copy_molecule(anchor_positions[i].second, anchor_positions[i-1].second);
				  anchor_positions[i].first = anchor_positions[i-1].first;
			  }

			  // copy mol into insert point
			  copy_molecule(anchor_positions[insert_point].second, mol);
			  anchor_positions[insert_point].first = mol_score;

		  }

	  }
***************/

	  // enter into loop if no more orients...
	  if(more_orients) {
		  return false;
	  } else {
		  sort(anchor_positions.begin(), anchor_positions.end(), less_than_pair);
		  return true;
	  }

  }

}


// +++++++++++++++++++++++++++++++++++++++++
void HDB_Conformer_Search::deactivate_all_branches(DOCKMol &mol) {
	int		i, j, offset;

	offset = branches[0].beatm;

	for(i=0;i<offset;i++)
		mol.atom_active_flags[i] = true;

	for(i=offset;i<total_mol.num_atoms;i++)
		mol.atom_active_flags[i] = false;

	mol.num_active_atoms = offset;

}


// +++++++++++++++++++++++++++++++++++++++++
void HDB_Conformer_Search::deactivate_molecule(DOCKMol &mol) {
	int		i;

	for(i=0;i<total_mol.num_atoms;i++)
		mol.atom_active_flags[i] = false;

	mol.num_active_atoms = 0;

}


// +++++++++++++++++++++++++++++++++++++++++
void HDB_Conformer_Search::activate_anchor(DOCKMol &mol) {
	int		i;

	for(i=0;i<branches[0].beatm;i++) {
		if(!mol.atom_active_flags[i]) {
			mol.atom_active_flags[i] = true;
			mol.num_active_atoms++;
		}
	}

}


// +++++++++++++++++++++++++++++++++++++++++
void HDB_Conformer_Search::deactivate_branch(DOCKMol &mol, int branch_num) {
	int		i;
	int		offset;

	offset = 0;

	for(i=0;i<branch_num;i++)
		offset += branches[i].beatm;

	for(i=offset;i<(branches[branch_num].beatm + offset);i++)
		mol.atom_active_flags[i] = false;

	mol.num_active_atoms = 0;

	for(i=0;i<mol.num_atoms;i++)
		if(mol.atom_active_flags[i])
			mol.num_active_atoms++;

}


// +++++++++++++++++++++++++++++++++++++++++
bool HDB_Conformer_Search::return_next_branch_conformer(INTVec &conf, int branch_num, int &pos) {
	int			start_atom, end_atom;
	int			i, j;

	// ID the start and end atoms for the branch
	start_atom = 0;
	end_atom = 0;

	for(i=0;i<=branch_num;i++) {
		for(j=0;j<branches[i].beatm;j++) {
			if(i<branch_num)
				start_atom++;
			end_atom++;
		}
	}

	if(pos < start_atom)
		pos = start_atom;

	if(pos >= end_atom)
		return false;

	conf = atoms[pos].level_confs;

	for(i=pos;i<=end_atom;i++) {
		if(atoms[i].level_confs != conf) {
			pos = i;
			break;
		}
	}

	return true;

}


// +++++++++++++++++++++++++++++++++++++++++
void HDB_Conformer_Search::skip_branch_conformer(INTVec &conf, int branch_num, int &pos) {
	int			start_atom, end_atom;
	int			i, j;
	bool		match_flag;

	// ID the start and end atoms for the branch
	start_atom = 0;
	end_atom = 0;

	for(i=0;i<=branch_num;i++) {
		for(j=0;j<branches[i].beatm;j++) {
			if(i<branch_num)
				start_atom++;
			end_atom++;
		}
	}

	if(pos < start_atom)
		pos = start_atom;

	// loop over next atoms
	for(i=pos;i<end_atom;i++) {

		// assume a match...
		match_flag = true;

		// flag any mis-matches as false (allowing wildcards on the query conf descriptors
		for(j=0;j<atoms[i].level_confs.size();j++) {

			if(conf[j] != atoms[i].level_confs[j]) {
				if(conf[j] != -1)
					match_flag = false;
			}

		}

		// if its a mismatch
		if(!match_flag)
			break;
	}

	// assign the new pos value
	pos = i;

}


// +++++++++++++++++++++++++++++++++++++++++
bool HDB_Conformer_Search::activate_branch_conformer(DOCKMol &mol, int branch_num, INTVec &conf) {
	int			start_atom, end_atom;
	int			i, j;
	bool		match_flag;

	// ID the start and end atoms for the branch
	start_atom = 0;
	end_atom = 0;

	for(i=0;i<=branch_num;i++) {
		for(j=0;j<branches[i].beatm;j++) {
			if(i<branch_num)
				start_atom++;
			end_atom++;
		}
	}

	// loop over atoms
	for(i=start_atom;i<end_atom;i++) {

		// assume a match...
		match_flag = true;

		// flag any mis-matches as false (allowing wildcards on the atom conf descriptors
		for(j=0;j<atoms[i].level_confs.size();j++) {
			if(conf[j] != atoms[i].level_confs[j]) {
				if(atoms[i].level_confs[j] != -1)
					match_flag = false;
			}
		}

		// if match is true, activate the atoms
		if(match_flag) {
			mol.atom_active_flags[i] = true;
			mol.num_active_atoms++;
		}

	}

	return true;

}


// +++++++++++++++++++++++++++++++++++++++++
bool HDB_Conformer_Search::generate_mols_from_confs(HDB_MULTICONF &conf, DOCKMol &ret_mol) {
	int				i, j;
	bool			continue_confs;
	DOCKMol			mol, tmp_mol;

	// initialize molecule for conformer construction
	copy_molecule(mol, conf.mol);
	deactivate_molecule(mol);
	activate_anchor(mol);

	// iterate over the conf levels
	continue_confs = true;

	// create conformer
	deactivate_all_branches(mol);

	for(j=0;j<conf_state.size();j++) {
		activate_branch_conformer(mol, j+1, conf.branch_confs[j][conf_state[j]]);
	}

	copy_molecule(ret_mol, mol);

	// increment the conf state
	conf_state[0]++;

	// carry over the conf state "remainders"
	for(j=0;j<conf_state.size()-1;j++) {
		if(conf_state[j] >= conf_state_max[j]) {
			conf_state[j] = 0;
			conf_state[j+1]++;
		}
	}

	// exit if the last conf has been reached
	if(conf_state[conf_state.size()-1] >= conf_state_max[conf_state_max.size()-1])
		continue_confs = false;

	return continue_confs;
}


// +++++++++++++++++++++++++++++++++++++++++
void HDB_Conformer_Search::grow_periphery(Master_Score &score, Bump_Filter &bump) {
	int				i, j, k, l;
	int				idx;
	DOCKMol			mol, tmp_mol;
	INTVec			conf;
	int				pos;
	HDB_MULTICONF	pose_confs;
	bool			complete_confs;
	bool			valid_conformer = false;
	int				num_confs;


	if(flexible_ligand) {

		// clear the docked conformation list
		docked_confs.clear();

		for(idx=0;idx<anchor_positions.size();idx++) {

			// resize the pose confs to have the proper # of branches
			pose_confs.clear();
			pose_confs.branch_conf_scores.resize(branches.size()-1);
			pose_confs.branch_confs.resize(branches.size()-1);

			// copy last anchor position to mol for conf expansion
			copy_molecule(mol, anchor_positions[idx].second);

			pos = 0;

			// loop over the branches
			for(i=1;i<branches.size();i++) {

				// loop over the conformers in each branch
				while(return_next_branch_conformer(conf, i, pos)) {

					// activate only the atoms of the conf being explored
					deactivate_molecule(mol);
					activate_branch_conformer(mol, i, conf);

					// score the branch conformer
//					score.compute_primary_score(mol);

					// if the branch conf bumps (or scores badly) skip it, else store it
//					if(score.return_primary_score() > 1000) {
					if(!bump.check_bumps(mol, true)) {

						skip_branch_conformer(conf, i, pos);

					} else {

						// if the conformer is complete
						if(conf[conf.size()-1] != -1) {

							// score the branch conformer
							valid_conformer = score.compute_primary_score(mol);
							 
							if(valid_conformer){
								// store it too
								pose_confs.branch_confs[i-1].push_back(conf);
								pose_confs.branch_conf_scores[i-1].push_back(mol.current_score);
								copy_molecule(pose_confs.mol, mol);
							}else{
								skip_branch_conformer(conf, i, pos);
								
							}
						}

					}

				}

			}

			// check for incomplete conformational expansion
			complete_confs = true;
			for(i=0;i<pose_confs.branch_conf_scores.size();i++)
				if(pose_confs.branch_conf_scores[i].size() == 0)
					complete_confs = false;

			// store the conformers for this docked pose
			if(complete_confs)
				docked_confs.push_back(pose_confs);

		}

	}

	// delete anchor positions
	anchor_positions.clear();

	// init conf state structures for molecule reconstruction
	conf_state.clear();
	conf_state.resize(docked_confs[docked_confs.size()-1].branch_conf_scores.size(), 0);

	conf_state_max.clear();
	conf_state_max.resize(docked_confs[docked_confs.size()-1].branch_conf_scores.size());

	for(j=0;j<conf_state_max.size();j++)
		conf_state_max[j] = docked_confs[docked_confs.size()-1].branch_conf_scores[j].size();

	return_periph_value = true;
}


// +++++++++++++++++++++++++++++++++++++++++
bool HDB_Conformer_Search::next_conformer(DOCKMol &mol) {
	int			j;

	if(flexible_ligand) {

		// hack to return the last conf
		if(!return_periph_value)
			return false;

		// if last multiconf is done...
		if(!generate_mols_from_confs(docked_confs[docked_confs.size()-1], mol)) {

			// delete it
			docked_confs.pop_back();

			// if more mols, prepare them
			if(docked_confs.size() > 0) {

				// init conf state structs
				conf_state.clear();
				conf_state.resize(docked_confs[docked_confs.size()-1].branch_conf_scores.size(), 0);

				conf_state_max.clear();
				conf_state_max.resize(docked_confs[docked_confs.size()-1].branch_conf_scores.size());

				for(j=0;j<conf_state_max.size();j++)
					conf_state_max[j] = docked_confs[docked_confs.size()-1].branch_conf_scores[j].size();

			} else {

				// hack to return the last conf
				if(return_periph_value) {
					return_periph_value = false;
					return true;
				}

			}

		}

		return true;

	} else { // if no lig flexibility

		if(return_periph_value) {
			return_periph_value = false;
			return true;
		} else
			return false;

	}
}
#endif
