#include "dock.h"
#include "version.h"

/************************************************/
int
main(int argc, char **argv)
{

    // set the function that will be called if new fails.
    std::set_new_handler(dock_new_handler);

    // synchronize C++ stream io and C stdio.
    ios::sync_with_stdio();

    // DOCK Classes
    // DOCK does not use constructors, but each class has an initialize member.
    // Note that the compiler-generated default constructors effectively
    // do nothing; so that these objects should be considered uninitialized
    // until their initialize members are called below.
    Parameter_Reader c_parm;
    Library_File    c_library;
    Orient          c_orient;
    Master_Conformer_Search c_master_conf;
    Bump_Filter     c_bmp_score;
    Simplex_Minimizer c_simplex;
    Master_Score    c_master_score;
    AMBER_TYPER     c_typer;
    DOCKMol         mol;

    // Local vars
    ofstream        outfile;
    streambuf      *original_cout_buffer;
    char            fname[50];
    bool            USE_MPI;

    // Preprocessor macro MPI is controlled by the platform configuration
    // which is specified during installation.
#ifdef BUILD_DOCK_WITH_MPI
    USE_MPI = true;
#else
    USE_MPI = false;
#endif

    // initialize mpi and determine if proper number of processors has been
    // called
    if (USE_MPI)
        // mpi_init requires pointers to argc and argv.
        USE_MPI = c_library.initialize_mpi(&argc, &argv);

    // ///////////////// Direct the output to a file or stdout
    // ////////////////////
    if (check_commandline_argument(argv, argc, "-o") != -1) {
        if (USE_MPI) {
            if (c_library.rank > 0) {
                sprintf(fname, "%s.%d",
                        parse_commandline_argument(argv, argc, "-o").c_str(),
                        c_library.rank);
            } else {
                sprintf(fname, "%s",
                        parse_commandline_argument(argv, argc, "-o").c_str());
            }
        } else {
            sprintf(fname, "%s",
                    parse_commandline_argument(argv, argc, "-o").c_str());
        }
        // redirect non error output to the -o file.
        // C stdio.
        freopen( fname, "a", stdout );
        // C++ stream io.
        outfile.open(fname);
        original_cout_buffer = cout.rdbuf();
        cout.rdbuf(outfile.rdbuf());

    } else if (USE_MPI) {
        cout << "Error: DOCK must be run with the -o outfile option under MPI"
            << endl;
        c_library.finalize_mpi();
        exit(0);
    }
    // /////////////////////////////////////////////////////////////////////////
    // Begin timing
    double          start_time = wall_clock_seconds();

    print_header();

    // Read input parameters
    c_parm.initialize(argc, argv);
    c_library.input_parameters_input(c_parm);
    c_orient.input_parameters(c_parm);
    c_master_conf.input_parameters(c_parm);
    c_bmp_score.input_parameters(c_parm);
    c_master_score.input_parameters(c_parm);
    c_simplex.input_parameters(c_parm, c_master_conf.flexible_ligand,
                               c_master_score);
    mol.input_parameters(c_parm);
    if (c_master_conf.flexible_ligand || c_simplex.minimize_ligand
        || c_orient.orient_ligand || c_bmp_score.bump_filter || c_library.calc_rmsd)
        c_master_score.read_vdw = true;
    c_typer.input_parameters(c_parm, c_master_score.read_vdw,
                             c_orient.use_chemical_matching);
    c_library.input_parameters_output(c_parm, c_master_score, USE_MPI);

    if (!USE_MPI)
        c_parm.write_params();

    // Exit the program if parameterization fails //
    if (!c_parm.parameter_input_successful()) {
        if (USE_MPI)
            c_library.finalize_mpi();
        if (outfile.is_open()) {
            // return non error output to standard output.
            cout.rdbuf(original_cout_buffer);
            outfile.close();
            fclose(stdout);  // C stdio.
        }
        return 1;               // non-zero return indicates error
    }
    // //////////////////////////////////////////////

    // Initialization routines
    c_typer.initialize(c_master_score.read_vdw, c_master_score.read_gb_parm,
                       c_orient.use_chemical_matching);
    c_library.initialize(argc, argv, USE_MPI);
    c_orient.initialize(argc, argv);
    c_master_conf.initialize();
    c_bmp_score.initialize();
    c_master_score.initialize_all(c_typer, c_simplex.final_min_add_internal,
                                  argc, argv);
    c_simplex.initialize();

#ifdef BUILD_DOCK_WITH_MPI
    if ((USE_MPI) && (c_library.rank > 0)) {
        cout << "DOCK is currently running on ";
        char klient[ MPI_MAX_PROCESSOR_NAME ];
        int length;
        MPI_Get_processor_name(klient, &length);
        for ( int i = 0; i < length; ++i )
            cout << klient[i];
        cout << endl;
    }
#endif

    // Main loop
    //if ligand has been docked
    //      write out and read in next ligand
    //if entire docking is complete
    //      perform final analysis and write out
    //else
    //      read in ligand
    while (c_library.get_mol(mol, USE_MPI, c_master_score.amber, c_master_score, c_simplex)) {

        // keep track of time for individual molecules
        double          mol_start_time = wall_clock_seconds();

        //seed random number generator
        c_simplex.initialize();

        //parse ligand atoms into child lists
        mol.prepare_molecule();

        //label ligand atoms with proper vdw, bond, and chem
        //types
        c_typer.prepare_molecule(mol, c_master_score.read_vdw,
                                 c_orient.use_chemical_matching);
    
        //parse ligand into rigid and flexible portions
        c_master_conf.prepare_molecule(mol);

        //while there is still another anchor to be docked
        while (c_master_conf.next_anchor(mol)) {

            c_library.num_anchors++;

            //generate entire list of atom center-sphere center matches
            c_orient.match_ligand(mol);

            //transform the ligand to match
            while (c_orient.new_next_orientation(mol)) {

                //perform bump check on anchor and filter if fails
                if (c_bmp_score.check_anchor_bumps(mol, c_orient.more_orientations())) {
                    
                    //minimize anchor orientations
                    c_simplex.minimize_rigid_anchor(mol, c_master_score);
                    
                    //score orientation and write out 
                    if(c_library.submit_orientation(mol, c_master_score, c_orient.orient_ligand)){

                        //rank orientations but only keep number user cutoff
                        if(c_master_conf.
                            submit_anchor_orientation(mol,
                                                  c_orient.more_orientations())){ 

                            //prune anchors, then perform growth, minimization, and pruning 
                            //until molecule is fully grown
                            c_master_conf.grow_periphery(c_master_score, c_simplex, c_bmp_score);

                            //for the fully grown conformations, if there are conformations
                            //remaining
                            while (c_master_conf.next_conformer(mol)) {

                                    //minimize the final pose
                                    c_simplex.minimize_final_pose(mol,
                                                                  c_master_score, c_typer);

                                    //add best scoring pose to list for
                                    //ranking and further analysis
                                    c_library.submit_scored_pose(mol,
                                                                 c_master_score,
                                                                 c_simplex);

                            }       

                            //write out list of final conformations
                            c_library.submit_conformation(c_master_score);
                        }

                    }          
                    

                }               


            }  
                
            
        }                     
        //print error messages if orienting or growth has failed
        if (c_library.num_orients == 0) {
            double          mol_stop_time = wall_clock_seconds();
            cout << endl << "-----------------------------------" << endl;
            cout << "Molecule: " << mol.title << endl << endl;
            cout << " Elapsed time:\t" << mol_stop_time -
                mol_start_time << " seconds" << endl << endl;
            cout << " ERROR:  Could not find valid anchor orientation." << endl;
            cout << "         Confirm all spheres are inside grid box and " <<
                endl;
            cout << "         grid box is big enough to contain anchor. " <<
                endl;
        } else if (c_library.num_confs == 0) {
            double          mol_stop_time = wall_clock_seconds();
            cout << endl << "-----------------------------------" << endl;
            cout << "Molecule: " << mol.title << endl << endl;
            cout << " Elapsed time:\t" << mol_stop_time -
                mol_start_time << " seconds" << endl << endl;
            cout << " ERROR:  Could not complete growth. " << endl;
            cout << "         Confirm grid box is large enough to contain " <<
                endl;
            cout << "         ligand and try increasing max_orients." << endl;
        }
        // End individual molecule timing 
        if (c_library.num_orients > 0 && c_library.num_confs > 0) {
            double          mol_stop_time = wall_clock_seconds();
            cout << endl << "-----------------------------------" << endl;
            cout << "Molecule: " << mol.title << endl << endl;
            cout << " Elapsed time for docking:\t" << mol_stop_time -
                mol_start_time << " seconds" << endl << endl;
        }
		
    }                          


    //rescore library using secondary scoring
    if(c_master_score.use_secondary_score){
        
	    c_library.secondary_rescore_poses(c_master_score, c_simplex);

        c_library.submit_secondary_pose();		

    }

    if ((!USE_MPI) || (c_library.rank == 0))
        cout << endl << endl << (c_library.total_mols -
                                 c_library.
                                 initial_skip) << " Molecules Processed" <<
            endl;
    else
        cout << endl << endl << c_library.
            completed << " Molecules Processed" << endl;

    // End timing
    double          stop_time = wall_clock_seconds();
    cout << "Total elapsed time:\t" << stop_time -
        start_time << " seconds" << endl;

    if (USE_MPI)
        c_library.finalize_mpi();

    if (outfile.is_open()) {
        // return non error output to standard output.
        cout.rdbuf(original_cout_buffer);
        outfile.close();
        fclose(stdout);  // C stdio.
    }

    return 0;                   // zero return indicates success

}

/************************************************/
void
dock_new_handler()
{

    // Define the default behavior for the failure of new.

    cout << "Error: memory exhausted!" << endl
        << "  For Unix platforms increase the datasize, stacksize, and\n"
        << "  memoryuse using the limit, ulimit, or unlimit commands." << endl;

    // Throw so that recovery or special error notification can be performed.
    throw           bad_alloc();
}

/************************************************/
void
print_header()
{
    cout << endl << endl << endl;
    cout << "--------------------------------------" << endl;
    cout << DOCK_VERSION << endl << endl;
    cout << "Released " << DOCK_RELEASE_DATE << endl;
    cout << "Copyright UCSF" << endl;
    cout << "--------------------------------------" << endl << endl;
}

/************************************************/
double
wall_clock_seconds()
{
    time_t          t;
    if (static_cast < time_t > (-1) == time(&t)) {
        cout << "Error from time function!  Elapsed time is erroneous." << endl;
    }
    return static_cast < double >(t);
}
