#include <iomanip>
#include "dock.h"

/***********************************************************************/
bool
Read_Mol2(DOCKMol & mol, istream & ifs, bool read_color, bool read_solvation,
          bool read_amber)
{

    char            line[1000];
    int             count;
    int             i;
    int             n1;
    bool            atom_line;
    char            typ[100],
                    col[100];

    char            tmp1[100],
                    tmp2[100],
                    subst_name[100];
    int             natoms,
                    nbonds;
    string          l1,
                    l2,
                    l3,
                    l4,
                    l5,
                    l6;
    string          t1,
                    t2,
                    t3;         // kxr205
    float           f1,
                    f2,
                    f3,
                    f4,
                    f5;

    bool            found_solvation = false;
    bool            found_color = false;
    bool            found_amber = false;



    // init state vars
    atom_line = false;


    // read forward until the tripos molecule tag is reached
    for (;;) {
        if (!ifs.getline(line, 1000)) {
            mol.clear_molecule();
            // cout << endl << "ERROR: Ligand file empty.  Program will
            // terminate." << endl;
            return false;
        }

        if (!strncmp(line, "@<TRIPOS>MOLECULE", 17))
            break;
    }

    // loop over the header info
    for (count = 0;; count++) {

        if (!ifs.getline(line, 1000)) {
            mol.clear_molecule();
            cout << endl <<
                "ERROR:  Ligand file empty.  Program will terminate." << endl;
            return false;
        }

        if (!strncmp(line, "@<TRIPOS>ATOM", 13)) {
            atom_line = true;
            break;
        }
        // assign the first 5 header lines to the proper fields
        switch (count) {

        case 0:
            l1 = line;
            break;

        case 1:
            l2 = line;
            break;

        case 2:
            l3 = line;
            break;

        case 3:
            l4 = line;
            break;

        case 4:
            l5 = line;
            break;

        case 5:
            l6 = line;
            break;
        }

    }

    // if there are no atoms, throw error and return false
    if (!atom_line) {
        mol.clear_molecule();
        cout <<
            "ERROR: @<TRIPOS>ATOM indicator missing from ligand file.  Program will terminate."
            << endl;
        return false;
    }
    // get # of atoms and bonds from mol info line
    sscanf(l2.c_str(), "%d %d", &natoms, &nbonds);

    // initialize molecule vectors
    mol.allocate_arrays(natoms, nbonds);

    mol.title = l1;
    mol.mol_info_line = l2;
    mol.comment1 = l3;
    mol.comment2 = l4;
    mol.energy = l5;
    mol.comment3 = l6;

    // loop over atoms and read in atom info
    for (i = 0; i < mol.num_atoms; i++) {
        if (!ifs.getline(line, 1000)) {
            mol.clear_molecule();
            cout <<
                "ERROR:  Atom information missing from ligand file.  Program will terminate."
                << endl;
            return false;
        }

        sscanf(line, "%*s %s %f %f %f %s %*s %s %f", tmp1, &mol.x[i], &mol.y[i],
               &mol.z[i], tmp2, subst_name, &mol.charges[i]);
        mol.atom_names[i] = tmp1;
        mol.atom_types[i] = tmp2;
        mol.subst_names[i] = subst_name;

    }

    // skip down to the bond section
    for (;;) {
        if (!ifs.getline(line, 1000)) {
            mol.clear_molecule();
            cout <<
                "ERROR: @<TRIPOS>BOND indicator missing from ligand file.  Program will terminate."
                << endl;
            return false;
        }

        if (!strncmp(line, "@<TRIPOS>BOND", 13))
            break;
    }

    // loop over bonds and add them
    for (i = 0; i < mol.num_bonds; i++) {
        if (!ifs.getline(line, 1000)) {
            mol.clear_molecule();
            cout <<
                "ERROR: Bond information missing from ligand file.  Program will terminate."
                << endl;
            return false;
        }


        sscanf(line, "%*d %d %d %s", &mol.bonds_origin_atom[i],
               &mol.bonds_target_atom[i], tmp1);

        // adjust bond atom #'s to start at 0
        mol.bonds_origin_atom[i]--;
        mol.bonds_target_atom[i]--;

        mol.bond_types[i] = tmp1;

    }
    

    // ID ring atoms/bonds
    mol.id_ring_atoms_bonds();

    
    // Read Atom Color from a pre-colored mol2 file kxr 0206
    // skip to the color section

    if (read_color) {
        for (;;) {

            if (!ifs.getline(line, 1000)) {
                mol.clear_molecule();
                cout <<
                    "ERROR: @<TRIPOS>COLOR indicator missing from ligand file.  Program will terminate."
                    << endl;
                return false;
            }

            if (!strncmp(line, "@<TRIPOS>COLOR", 14)) {
                found_color = true;
                break;
            }


        }


        if (found_color) {

            for (i = 0; i < mol.num_atoms; i++) {

                if (!ifs.getline(line, 1000)) {
                    mol.clear_molecule();
                    cout <<
                        "ERROR: Coloring information missing from ligand file.  Program will terminate."
                        << endl;
                    return false;
                }

                sscanf(line, "%d %99s %99s", &n1, typ, col);
                mol.atom_color[i] = col;
            }
        }
    }
    // Read Total and atomic desolvation numbers kxr 
    // skip to the solvation info section
    if (read_solvation) {

        for (;;) {
            if (!ifs.getline(line, 1000)) {
                mol.clear_molecule();
                cout << endl <<
                    "ERROR: @<TRIPOS>SOLVATION indicator missing from ligand file.  Program will terminate."
                    << endl;
                return false;
            }
            // if(!found_solvation) {
            if (!strncmp(line, "@<TRIPOS>SOLVATION", 18)) {
                found_solvation = true;
                break;
            }                   // else { cout << endl << "ERROR: Desolvation
                                // input absent.  Program will terminate." <<
                                // endl; return false; } } else break;
        }

        if (found_solvation) {

            if (!ifs.getline(line, 1000)) {
                mol.clear_molecule();
                cout <<
                    "ERROR: Solvation information missing from ligand file.  Program will terminate."
                    << endl;
                return false;
            }

            mol.total_dsol = 0.0f;
            int             start_atm = 0;

            if (ifs.getline(line, 1000)) {
                sscanf(line, "%f %f %f %f %f", &f1, &f2, &f3, &f4, &f5);
                mol.total_dsol += f5;
                mol.atom_psol[start_atm] = f2;
                mol.atom_apsol[start_atm] = f4;
                start_atm = 1;
            }


            for (i = start_atm; i < mol.num_atoms; i++) {

                if (ifs.getline(line, 1000)) {
                    sscanf(line, "%f %f %f %f %f", &f1, &f2, &f3, &f4, &f5);
                    mol.total_dsol += f5;
                    mol.atom_psol[i] = f2;
                    mol.atom_apsol[i] = f4;
                }

            }

        }

    }

    if (read_amber) {
        for (;;) {
            // check if AMBER_SCORE_ID is even present
            if (!ifs.getline(line, 1000)) {
                mol.clear_molecule();
                cout << endl <<
                    "ERROR: @<TRIPOS>AMBER_SCORE_ID indicator missing from ligand file.  Program will terminate."
                    << endl;
                return false;
            }
            // identify that we are in the amber_score_id portion of the file
            if (!strncmp(line, "@<TRIPOS>AMBER_SCORE_ID", 23)) {
                found_amber = true;
                break;
            }
        }

        if (found_amber) {
            // check if there is any information in the AMBER_SCORE_ID field
            if (!ifs.getline(line, 1000)) {
                mol.clear_molecule();
                cout <<
                    "ERROR: No AMBER Score location information available.  Program will terminate."
                    << endl;
                return false;
            }

            sscanf(line, "%99s", tmp1);
            mol.amber_score_ligand_id = tmp1;
        }
    }


    return true;
}

/***********************************************************************/
bool
Write_Mol2(DOCKMol & mol, ostream & ofs)
{
    char            line[1000];
    int             i;
    vector < int   >renumber;
    int             current_atom,
                    current_bond;

    // init atom/bond renumbering data
    renumber.resize(mol.num_atoms + 1);
    current_atom = 1;
    current_bond = 1;

    // write out header information
    ofs << "@<TRIPOS>MOLECULE" << endl;

    if (mol.title.empty())
        ofs << "*****" << endl;
    else
        ofs << mol.title << endl;

    sprintf(line, " %d %d 1 0 0", mol.num_active_atoms, mol.num_active_bonds);
    ofs << line << endl;

    ofs << mol.comment1 << endl;
    ofs << mol.comment2 << endl;
    ofs << mol.energy << endl;
    ofs << mol.comment3 << endl;

    // write out atom lines
    ofs << "@<TRIPOS>ATOM" << endl;

    for (i = 0; i < mol.num_atoms; i++) {
        if (mol.atom_active_flags[i]) {

            sprintf(line,
                    "%7d%1s%-6s%12.4f%10.4f%10.4f%1s%-5s%4s%1s %-8s%10.4f",
                    current_atom, "", mol.atom_names[i].c_str(), mol.x[i],
                    mol.y[i], mol.z[i], "", mol.atom_types[i].c_str(), "1", "",
                    mol.subst_names[i].c_str(), mol.charges[i]);

            ofs << line << endl;

            renumber[i] = current_atom;
            current_atom++;
        }
    }

    // write out bond lines
    ofs << "@<TRIPOS>BOND" << endl;

    for (i = 0; i < mol.num_bonds; i++) {
        if (mol.bond_active_flags[i]) {

            sprintf(line, "%6d%6d%6d%3s%2s", current_bond,
                    renumber[mol.bonds_origin_atom[i]],
                    renumber[mol.bonds_target_atom[i]], "",
                    mol.bond_types[i].c_str());

            ofs << line << endl;
            current_bond++;
        }
    }

    ofs << "@<TRIPOS>SUBSTRUCTURE" << endl;
    sprintf(line,
            "     1 %-4s        1 TEMP              0 ****  ****    0 ROOT",
            mol.subst_names[0].c_str());
    ofs << line << endl << endl;


    return true;
}

/***********************************************************************/
void
copy_molecule(DOCKMol & target, const DOCKMol & original)
{
    int             i;

    target.allocate_arrays(original.num_atoms, original.num_bonds);

    // copy scalar data
    target.title = original.title;
    target.mol_info_line = original.mol_info_line;
    target.comment1 = original.comment1;
    target.comment2 = original.comment2;
    target.comment3 = original.comment3;
    target.energy = original.energy;

    target.mol_data = original.mol_data;

    target.num_atoms = original.num_atoms;
    target.num_bonds = original.num_bonds;

    target.num_active_atoms = original.num_active_atoms;
    target.num_active_bonds = original.num_active_bonds;
    target.score_text_data = original.score_text_data;

    target.amber_at_assigned = original.amber_at_assigned;
    target.amber_bt_assigned = original.amber_bt_assigned;
    target.chem_types_assigned = original.chem_types_assigned;

    target.current_score = original.current_score;
    target.current_data = original.current_data;
    target.primary_data = original.primary_data;

    target.amber_score_ligand_id = original.amber_score_ligand_id;
    target.total_dsol = original.total_dsol;

    // copy arrays
    for (i = 0; i < original.num_atoms; i++) {

        target.atom_data[i] = original.atom_data[i];

        target.x[i] = original.x[i];
        target.y[i] = original.y[i];
        target.z[i] = original.z[i];

        target.charges[i] = original.charges[i];
        target.atom_types[i] = original.atom_types[i];
        target.atom_names[i] = original.atom_names[i];
        target.subst_names[i] = original.subst_names[i];
        target.atom_color[i] = original.atom_color[i];  // kxr205
        target.atom_psol[i] = original.atom_psol[i];    // kxr205
        target.atom_apsol[i] = original.atom_apsol[i];  // kxr205

        target.atom_ring_flags[i] = original.atom_ring_flags[i];
        target.atom_active_flags[i] = original.atom_active_flags[i];

        target.amber_at_id[i] = original.amber_at_id[i];
        target.amber_at_radius[i] = original.amber_at_radius[i];
        target.amber_at_well_depth[i] = original.amber_at_well_depth[i];
        target.amber_at_heavy_flag[i] = original.amber_at_heavy_flag[i];
        target.amber_at_valence[i] = original.amber_at_valence[i];
        target.amber_at_bump_id[i] = original.amber_at_bump_id[i];

        target.chem_types[i] = original.chem_types[i];

        target.gb_hawkins_radius[i] = original.gb_hawkins_radius[i];
        target.gb_hawkins_scale[i] = original.gb_hawkins_scale[i];

        target.neighbor_list[i] = original.neighbor_list[i];
    }

    for (i = 0; i < original.num_bonds; i++) {

        target.bonds_origin_atom[i] = original.bonds_origin_atom[i];
        target.bonds_target_atom[i] = original.bonds_target_atom[i];
        target.bond_types[i] = original.bond_types[i];

        target.bond_ring_flags[i] = original.bond_ring_flags[i];
        target.bond_active_flags[i] = original.bond_active_flags[i];

        target.amber_bt_id[i] = original.amber_bt_id[i];
        target.amber_bt_minimize[i] = original.amber_bt_minimize[i];
        target.amber_bt_torsion_total[i] = original.amber_bt_torsion_total[i];
        target.amber_bt_torsions[i] = original.amber_bt_torsions[i];

    }

    // for(i=0;i<original.num_atoms*original.num_atoms;i++)
    // target.ie_neighbor_list[i] = original.ie_neighbor_list[i];

    for (i = 0; i < 2 * original.num_bonds; i++)
        target.atom_child_list[i] = original.atom_child_list[i];

}

/***********************************************************************/
void
copy_crds(DOCKMol & target, DOCKMol & original)
{
    int             i;

    for (i = 0; i < original.num_atoms; i++) {
        target.x[i] = original.x[i];
        target.y[i] = original.y[i];
        target.z[i] = original.z[i];
    }

}

/*************************************/
DOCKMol::DOCKMol()
{

    initialize();

}

/*************************************/
DOCKMol::DOCKMol(const DOCKMol & original)
{

    initialize();
    copy_molecule(*this, original);

}

/*************************************/
DOCKMol::~DOCKMol()
{

    clear_molecule();

}

/*************************************/
void
DOCKMol::operator=(const DOCKMol & original)
{

    copy_molecule(*this, original);

}


/*************************************/
void
DOCKMol::input_parameters(Parameter_Reader & parm)
{


    // read_solvation = (parm.query_param("read_mol_solvation","no","yes no")
    // == "yes")?true:false;

    // read_color = (parm.query_param("read_mol_color","no","yes no") ==
    // "yes")?true:false;


  /***
  string tmp;

  cout << "\nMolecule Parameters" << endl;
  cout << "------------------------------------------------------------------------------------------" << endl;

  tmp = parm.query_param("calc_internal_energy", "no", "yes no");
  if(tmp == "yes")
	  use_internal_energy = true;
  else
	  use_internal_energy = false;

  if(use_internal_energy) {
	  att_exp = atoi(parm.query_param("internal_energy_att_exp", "6").c_str());
	  rep_exp = atoi(parm.query_param("internal_energy_rep_exp", "12").c_str());
	  dielectric = atof(parm.query_param("internal_energy_dielectric", "4.0").c_str());

	  // internal energy hardcoded params
	  dielectric = 332.0 / dielectric;
  }
  ***/
    // use_internal_energy = false;

}

/*************************************/
void
DOCKMol::initialize()
{

    arrays_allocated = false;
    // use_internal_energy = false;
    clear_molecule();           // /////////////////////

}

/*************************************/
void
DOCKMol::allocate_arrays(int natoms, int nbonds)
{
    int             i;

    clear_molecule();

    num_atoms = natoms;
    num_bonds = nbonds;

    atom_data = new string[num_atoms];

    x = new float[num_atoms];
    y = new float[num_atoms];
    z = new float[num_atoms];

    charges = new float[num_atoms];
    atom_types = new string[num_atoms];
    atom_names = new string[num_atoms];
    subst_names = new string[num_atoms];
    atom_color = new string[num_atoms]; // kxr205
    atom_psol = new float[num_atoms];   // kxr205
    atom_apsol = new float[num_atoms];  // kxr205

    atom_ring_flags = new bool[num_atoms];
    bond_ring_flags = new bool[num_bonds];

    bonds_origin_atom = new int[num_bonds];
    bonds_target_atom = new int[num_bonds];
    bond_types = new string[num_bonds];

    atom_active_flags = new bool[num_atoms];
    bond_active_flags = new bool[num_bonds];

    num_active_atoms = num_atoms;
    num_active_bonds = num_bonds;

    amber_at_bump_id = new int[num_atoms];
    amber_at_heavy_flag = new int[num_atoms];
    amber_at_id = new int[num_atoms];
    amber_at_valence = new int[num_atoms];
    amber_at_radius = new float[num_atoms];
    amber_at_well_depth = new float[num_atoms];

    chem_types = new string[num_atoms];

    amber_bt_id = new int[num_bonds];
    amber_bt_minimize = new int[num_bonds];
    amber_bt_torsion_total = new int[num_bonds];
    amber_bt_torsions = new TOR_LIST[num_bonds];

    gb_hawkins_radius = new float[num_atoms];
    gb_hawkins_scale = new float[num_atoms];

    neighbor_list = new INTVec[num_atoms];
    // ie_neighbor_list = new bool[num_atoms*num_atoms];

    atom_child_list = new INTVec[num_bonds * 2];

    arrays_allocated = true;

    // assign init values
    for (i = 0; i < num_atoms; i++) {

        atom_data[i] = "";
        atom_names[i] = "";
        subst_names[i] = "";
        atom_types[i] = "";
        atom_color[i] = "null"; // kxr205
        atom_psol[i] = 0.0;     // kxr205
        atom_apsol[i] = 0.0;
        charges[i] = 0.0;

        atom_ring_flags[i] = false;
        atom_active_flags[i] = true;

        amber_at_bump_id[i] = 0;
        amber_at_heavy_flag[i] = 0;
        amber_at_id[i] = 0;
        amber_at_valence[i] = 0;
        amber_at_radius[i] = 0.0;
        amber_at_well_depth[i] = 0.0;

        chem_types[i] = "";

        gb_hawkins_radius[i] = 0.0;
        gb_hawkins_scale[i] = 0.0;

    }

    for (i = 0; i < num_bonds; i++) {

        bonds_origin_atom[i] = 0;
        bonds_target_atom[i] = 0;
        bond_types[i] = "";

        bond_ring_flags[i] = false;
        bond_active_flags[i] = true;

        amber_bt_id[i] = 0;
        amber_bt_minimize[i] = 0;
        amber_bt_torsion_total[i] = 0;
    }

    // test_child_list = new bool[2*num_bonds*num_atoms];

}

/*************************************/
void
DOCKMol::clear_molecule()
{
    int             i;

    amber_at_assigned = false;
    amber_bt_assigned = false;


    if (arrays_allocated) {

        for (i = 0; i < num_bonds; i++)
            amber_bt_torsions[i].clear();

        for (i = 0; i < num_atoms; i++)
            neighbor_list[i].clear();

        for (i = 0; i < 2 * num_bonds; i++)
            atom_child_list[i].clear();

        delete[]atom_data;
        delete[]x;
        delete[]y;
        delete[]z;

        delete[]charges;
        delete[]atom_types;
        delete[]atom_names;
        delete[]subst_names;
        delete[]atom_color;     // kxr205
        delete[]atom_psol;      // kxr205
        delete[]atom_apsol;

        delete[]bonds_origin_atom;
        delete[]bonds_target_atom;
        delete[]bond_types;

        delete[]atom_ring_flags;
        delete[]bond_ring_flags;

        delete[]atom_active_flags;
        delete[]bond_active_flags;

        delete[]amber_at_id;
        delete[]amber_at_radius;
        delete[]amber_at_well_depth;
        delete[]amber_at_heavy_flag;
        delete[]amber_at_valence;
        delete[]amber_at_bump_id;

        delete[]chem_types;

        delete[]amber_bt_id;
        delete[]amber_bt_minimize;
        delete[]amber_bt_torsion_total;
        delete[]amber_bt_torsions;

        delete[]gb_hawkins_radius;
        delete[]gb_hawkins_scale;

        delete[]neighbor_list;
        // delete [] ie_neighbor_list;
        // delete [] test_child_list;
        delete[]atom_child_list;

        arrays_allocated = false;
    }
    // clear MOL2 header info
    title = "";
    mol_info_line = "";
    comment1 = "";
    comment2 = "";
    comment3 = "";
    energy = "";
    mol_data = "";

    // clear scalar data
    num_atoms = 0;
    num_bonds = 0;
    num_active_atoms = 0;
    num_active_bonds = 0;
    score_text_data = "";
    current_score = 0.0;
    current_data = "ERROR: Conformation could not be scored by DOCK.\n";
    primary_data = "ERROR: Conformation could not be scored by DOCK.\n";
    amber_score_ligand_id = "";
    total_dsol = 0.0;

}

/*************************************/
vector < int   >
DOCKMol::get_atom_neighbors(int index)
{
    vector < int   >nbrs;
    int             i;

    nbrs.clear();

    // loop over bonds and find any that include the index atom
    for (i = 0; i < num_bonds; i++) {

        if (bonds_origin_atom[i] == index)
            nbrs.push_back(bonds_target_atom[i]);

        if (bonds_target_atom[i] == index)
            nbrs.push_back(bonds_origin_atom[i]);

    }

    return nbrs;
}

/*************************************/
vector < int   >
DOCKMol::get_bond_neighbors(int index)
{
    vector < int   >nbrs;
    int             i;

    nbrs.clear();

    // loop over bonds- id any that contain index atom
    for (i = 0; i < num_bonds; i++) {

        if ((bonds_origin_atom[i] == index) || (bonds_target_atom[i] == index))
            nbrs.push_back(i);

    }

    return nbrs;
}

/*************************************/
int
DOCKMol::get_bond(int a1, int a2)
{
    int             bond_id;
    int             i;

    bond_id = -1;

    for (i = 0; i < num_bonds; i++) {

        if ((bonds_origin_atom[i] == a1) && (bonds_target_atom[i] == a2))
            bond_id = i;

        if ((bonds_origin_atom[i] == a2) && (bonds_target_atom[i] == a1))
            bond_id = i;

    }

    return bond_id;
}

/*************************************/
void
DOCKMol::id_ring_atoms_bonds()
{
    vector < bool > atoms_visited, bonds_visited;
    vector < int   >atom_path,
                    bond_path;
    int             i;

    atoms_visited.clear();
    bonds_visited.clear();

    atoms_visited.resize(num_atoms, false);
    bonds_visited.resize(num_bonds, false);
    atom_path.clear();
    bond_path.clear();

    for (i = 0; i < num_atoms; i++)
        atoms_visited[i] = false;

    for (i = 0; i < num_bonds; i++)
        bonds_visited[i] = false;

    // loop over all atoms and find rings
    for (i = 0; i < num_atoms; i++)
        if (!atoms_visited[i])
            find_rings(atom_path, bond_path, atoms_visited, bonds_visited, i);

}

/*************************************/
void
DOCKMol::find_rings(vector < int >apath, vector < int >bpath,
                    vector < bool > &atoms, vector < bool > &bonds, int atnum)
{
    int             i,
                    j,
                    nbr_bond;
    vector < int   >nbrs;

    if (atoms[atnum]) {

        i = apath.size() - 1;
        j = bpath.size() - 1;

        while ((i >= 0) && (j >= 0)) {  // ///// changed from > to >= fixes the 
                                        // first atom bug
            atom_ring_flags[apath[i--]] = true;
            bond_ring_flags[bpath[j--]] = true;

            if (i == -1)        // added fix to address if i = 0
                break;
            else if (apath[i] == atnum)
                break;

        }

    } else {

        atoms[atnum] = true;
        nbrs = get_atom_neighbors(atnum);

        for (i = 0; i < nbrs.size(); i++) {

            nbr_bond = get_bond(nbrs[i], atnum);

            if (!bonds[nbr_bond]) {

                bonds[nbr_bond] = true;

                apath.push_back(nbrs[i]);
                bpath.push_back(nbr_bond);

                find_rings(apath, bpath, atoms, bonds, nbrs[i]);
                apath.pop_back();
                bpath.pop_back();
            }

        }

    }

}

/*************************************/
vector < int   >
DOCKMol::get_atom_children(int a1, int a2)
{
    vector < int   >children;
    bool           *visited;
    vector < int   >nbrs,
                    new_nbrs;
    int             i;

    if (get_bond(a1, a2) != -1) {
        visited = new bool[num_atoms];
        memset(visited, 0, num_atoms * sizeof(bool));
        // visited.resize(num_atoms, false);

        visited[a1] = true;
        visited[a2] = true;

        // nbrs = get_atom_neighbors(a2);
        nbrs = neighbor_list[a2];

        while (nbrs.size() > 0) {

            if (!visited[nbrs[nbrs.size() - 1]]) {

                children.push_back(nbrs[nbrs.size() - 1]);
                new_nbrs.clear();
                // new_nbrs = get_atom_neighbors(nbrs[nbrs.size()-1]);
                new_nbrs = neighbor_list[nbrs[nbrs.size() - 1]];
                visited[nbrs[nbrs.size() - 1]] = true;
                nbrs.pop_back();
                for (i = 0; i < new_nbrs.size(); i++)
                    nbrs.push_back(new_nbrs[i]);

            } else {
                nbrs.pop_back();
            }

        }

        delete[]visited;
    }

    return children;
}

/*************************************/
float
DOCKMol::get_torsion(int a1, int a2, int a3, int a4)
{
    float           torsion;
    DOCKVector      v1,
                    v2,
                    v3,
                    v4;

    v1.x = x[a1];
    v1.y = y[a1];
    v1.z = z[a1];

    v2.x = x[a2];
    v2.y = y[a2];
    v2.z = z[a2];

    v3.x = x[a3];
    v3.y = y[a3];
    v3.z = z[a3];

    v4.x = x[a4];
    v4.y = y[a4];
    v4.z = z[a4];

    torsion = get_torsion_angle(v1, v2, v3, v4);

    return torsion;
}

/*************************************/
void
DOCKMol::set_torsion(int a1, int a2, int a3, int a4, float angle)
{
    int             tor[4];
    vector < int   >atoms;
    float           v1x,
                    v1y,
                    v1z,
                    v2x,
                    v2y,
                    v2z,
                    v3x,
                    v3y,
                    v3z;
    float           c1x,
                    c1y,
                    c1z,
                    c2x,
                    c2y,
                    c2z,
                    c3x,
                    c3y,
                    c3z;
    float           c1mag,
                    c2mag,
                    radang,
                    costheta,
                    m[9];
    float           nx,
                    ny,
                    nz,
                    mag,
                    rotang,
                    sn,
                    cs,
                    t,
                    tx,
                    ty,
                    tz;
    int             i,
                    j,
                    idx;

    tor[0] = a1;
    tor[1] = a2;
    tor[2] = a3;
    tor[3] = a4;

    idx = get_bond(a2, a3);

    if (a2 < a3)
        idx = 2 * idx;
    else
        idx = 2 * idx + 1;

    atoms = atom_child_list[idx];
    // atoms = get_atom_children(a2, a3);
    // atoms = child_list[a2][a3]; //////////////////

    // calculate the torsion angle
    v1x = x[tor[0]] - x[tor[1]];
    v2x = x[tor[1]] - x[tor[2]];
    v1y = y[tor[0]] - y[tor[1]];
    v2y = y[tor[1]] - y[tor[2]];
    v1z = z[tor[0]] - z[tor[1]];
    v2z = z[tor[1]] - z[tor[2]];
    v3x = x[tor[2]] - x[tor[3]];
    v3y = y[tor[2]] - y[tor[3]];
    v3z = z[tor[2]] - z[tor[3]];

    c1x = v1y * v2z - v1z * v2y;
    c2x = v2y * v3z - v2z * v3y;
    c1y = -v1x * v2z + v1z * v2x;
    c2y = -v2x * v3z + v2z * v3x;
    c1z = v1x * v2y - v1y * v2x;
    c2z = v2x * v3y - v2y * v3x;
    c3x = c1y * c2z - c1z * c2y;
    c3y = -c1x * c2z + c1z * c2x;
    c3z = c1x * c2y - c1y * c2x;

    c1mag = pow(c1x, 2) + pow(c1y, 2) + pow(c1z, 2);
    c2mag = pow(c2x, 2) + pow(c2y, 2) + pow(c2z, 2);

    if (c1mag * c2mag < 0.01)
        costheta = 1.0;         // avoid div by zero error
    else
        costheta = (c1x * c2x + c1y * c2y + c1z * c2z) / (sqrt(c1mag * c2mag));

    if (costheta < -0.999999)
        costheta = -0.999999f;
    if (costheta > 0.999999)
        costheta = 0.999999f;

    if ((v2x * c3x + v2y * c3y + v2z * c3z) > 0.0)
        radang = -acos(costheta);
    else
        radang = acos(costheta);

    // 
    // now we have the torsion angle (radang) - set up the rot matrix
    // 

    // find the difference between current and requested
    rotang = angle - radang;

    sn = sin(rotang);
    cs = cos(rotang);
    t = 1 - cs;

    // normalize the rotation vector
    mag = sqrt(pow(v2x, 2) + pow(v2y, 2) + pow(v2z, 2));
    nx = v2x / mag;
    ny = v2y / mag;
    nz = v2z / mag;

    // set up the rotation matrix
    m[0] = t * nx * nx + cs;
    m[1] = t * nx * ny + sn * nz;
    m[2] = t * nx * nz - sn * ny;
    m[3] = t * nx * ny - sn * nz;
    m[4] = t * ny * ny + cs;
    m[5] = t * ny * nz + sn * nx;
    m[6] = t * nx * nz + sn * ny;
    m[7] = t * ny * nz - sn * nx;
    m[8] = t * nz * nz + cs;

    // 
    // now the matrix is set - time to rotate the atoms
    // 
    tx = x[tor[1]];
    ty = y[tor[1]];
    tz = z[tor[1]];

    for (i = 0; i < atoms.size(); i++) {
        j = atoms[i];

        // for(i=0;i<num_atoms;i++) {
        // j = i;

        // if(child_list[a2*num_atoms + i] == a3) { //////////////

        x[j] -= tx;
        y[j] -= ty;
        z[j] -= tz;

        nx = x[j] * m[0] + y[j] * m[1] + z[j] * m[2];
        ny = x[j] * m[3] + y[j] * m[4] + z[j] * m[5];
        nz = x[j] * m[6] + y[j] * m[7] + z[j] * m[8];

        x[j] = nx;
        y[j] = ny;
        z[j] = nz;
        x[j] += tx;
        y[j] += ty;
        z[j] += tz;

        // } /////////////////////////
    }

}

/*********************************************/
void
DOCKMol::translate_mol(const DOCKVector & vec)
{
    int             i;

    for (i = 0; i < num_atoms; i++) {
        x[i] += vec.x;
        y[i] += vec.y;
        z[i] += vec.z;
    }

}

/*********************************************/
void
DOCKMol::rotate_mol(float mat[9])
{
    int             i;
    float           nx,
                    ny,
                    nz;

    for (i = 0; i < num_atoms; i++) {
        nx = x[i];
        ny = y[i];
        nz = z[i];

        x[i] = mat[0] * nx + mat[1] * ny + mat[2] * nz;
        y[i] = mat[3] * nx + mat[4] * ny + mat[5] * nz;
        z[i] = mat[6] * nx + mat[7] * ny + mat[8] * nz;
    }

}

/*********************************************/
void
DOCKMol::rotate_mol(float mat[3][3])
{
    float           new_mat[9];
    int             i,
                    j,
                    k;

    k = 0;

    for (i = 0; i < 3; i++)
        for (j = 0; j < 3; j++)
            new_mat[k++] = mat[i][j];

    rotate_mol(new_mat);

}

/*********************************************/
bool
DOCKMol::atoms_are_one_three(int a1, int a2)
{
    vector < int   >nbrs1,
                    nbrs2;
    int             i,
                    j;

    nbrs1 = get_atom_neighbors(a1);
    nbrs2 = get_atom_neighbors(a2);

    for (i = 0; i < nbrs1.size(); i++) {
        for (j = 0; j < nbrs2.size(); j++) {
            if (nbrs1[i] == nbrs2[j])
                return true;
        }
    }

    return false;
}

/*********************************************/
bool
DOCKMol::atoms_are_one_four(int a1, int a2)
{
    vector < int   >nbrs1,
                    nbrs2;
    int             i,
                    j;

    nbrs1 = get_atom_neighbors(a1);
    nbrs2 = get_atom_neighbors(a2);

    for (i = 0; i < nbrs1.size(); i++) {
        for (j = 0; j < nbrs2.size(); j++) {
            if (get_bond(nbrs1[i], nbrs2[j]) != -1)
                return true;
        }
    }

    return false;
}

/*********************************************
float DOCKMol::compute_internal_energy() {
        int             i,j;
        float   total_energy;
        float   total_vdw;
        float   total_es;
        float   distance;

        total_energy = total_vdw = total_es = 0.0;

        if(use_internal_energy) {

                for(i=0;i<num_atoms;i++) {

                        for(j=i+1;j<num_atoms;j++) {

                                if(atom_active_flags[i] && atom_active_flags[j]) {

                                        if( (get_bond(i, j)==-1) && (!atoms_are_one_three(i, j)) && (!atoms_are_one_four(i, j)) ) {
//                                      if(!ie_neighbor_list[i*num_atoms + j]) {

                                                distance = pow((x[i]-x[j]),2) + pow((y[i]-y[j]),2) + pow((z[i]-z[j]),2);
//                                              distance = sqrt(distance);

                                                total_vdw += (vdwA[i]*vdwA[j])/pow(distance, (rep_exp)) - (vdwB[i]*vdwB[j])/pow(distance, (att_exp)); // removed /2 term
                                                total_es  += (charges[i] * charges[j] * dielectric) / pow(distance, 1/2);//sqrt(distance);

                                        }

                                }
                        }

                }

        }

        total_energy = total_vdw + total_es;

        return total_energy;
}
**/

/*************************************/
void
DOCKMol::prepare_molecule()
{
    int             i,
                    j,
                    idx;

    // pre-cache neighbor list
    for (i = 0; i < num_atoms; i++)
        neighbor_list[i] = get_atom_neighbors(i);

        /**
	// pre-cache ie_neighbor_list
	for(i=0;i<num_atoms;i++) {
		for(j=0;j<num_atoms;j++) {
			if((get_bond(i, j)==-1)&&(!atoms_are_one_three(i, j))&&(!atoms_are_one_four(i, j))&&(i!=j))
				ie_neighbor_list[i*num_atoms + j] = false;
			else
				ie_neighbor_list[i*num_atoms + j] = true;
		}
	}
	**/

    // pre-cache atom child list
    for (i = 0; i < num_atoms; i++) {
        for (j = 0; j < num_atoms; j++) {
            if (i != j) {
                idx = get_bond(i, j);
                if (idx != -1) {
                    if (i < j)
                        idx = 2 * idx;
                    else
                        idx = 2 * idx + 1;

                    atom_child_list[idx] = get_atom_children(i, j);
                }
            }
        }
    }


}

/*********************************************/
int
less_than_pair(SCOREMol a, SCOREMol b)
{
    return (a.first < b.first);
}

/************************************************/
int
BREADTH_SEARCH::get_search_radius(DOCKMol & mol, int root_atom, int avoid_atom)
{
    int             root;
    int             x,
                    max_radius;
    int             nbr_atom;
    int             i;
    vector < int   >atom_nbrs;

    atoms.clear();
    nbrs.clear();
    nbrs_next.clear();

    atoms.resize(mol.num_atoms, -1);
    nbrs.push_back(root_atom);
    atoms[root_atom] = 0;
    atoms[avoid_atom] = -2;

    while (nbrs.size() > 0) {
        root = nbrs[nbrs.size() - 1];
        nbrs.pop_back();

        // atom_nbrs = mol.get_atom_neighbors(root);
        atom_nbrs = mol.neighbor_list[root];

        for (i = 0; i < atom_nbrs.size(); i++) {
            nbr_atom = atom_nbrs[i];

            if (atoms[nbr_atom] == -1) {
                atoms[nbr_atom] = atoms[root] + 1;
                nbrs.push_back(nbr_atom);
            }
        }

        if ((nbrs.size() == 0) && (nbrs_next.size() > 0)) {
            nbrs = nbrs_next;
            nbrs_next.clear();
        }

    }

    max_radius = 0;
    for (x = 0; x < atoms.size(); x++) {
        if (atoms[x] > max_radius)
            max_radius = atoms[x];
    }

    return max_radius;
}

/******************************************************/
void
transform(DOCKMol & mol, float rmat[3][3], DOCKVector trans, DOCKVector com)
{
    int             i;
    float           temp1,
                    temp2,
                    temp3,
                    temp4,
                    temp5,
                    temp6;

    for (i = 0; i < mol.num_atoms; i++) {
        temp1 = mol.x[i] - com.x;
        temp2 = mol.y[i] - com.y;
        temp3 = mol.z[i] - com.z;

        temp4 = rmat[0][0] * temp1;
        temp5 = rmat[1][0] * temp2;
        temp6 = rmat[2][0] * temp3;

        mol.x[i] = temp4 + temp5 + temp6 + com.x + trans.x;

        temp4 = rmat[0][1] * temp1;
        temp5 = rmat[1][1] * temp2;
        temp6 = rmat[2][1] * temp3;

        mol.y[i] = temp4 + temp5 + temp6 + com.y + trans.y;

        temp4 = rmat[0][2] * temp1;
        temp5 = rmat[1][2] * temp2;
        temp6 = rmat[2][2] * temp3;

        mol.z[i] = temp4 + temp5 + temp6 + com.z + trans.z;

    }

}

// +++++++++++++++++++++++++++++++++++++++++
// Copy the Cartesian coordinates from the argument to this DOCKMol.
// The linear memory representation of argument xyz, starting from *xyz, is
// first atom x, first atom y, first atom z, second atom x, ...
void
DOCKMol::setxyz( const double * xyz )
{
    for (int i = 0; i < num_atoms; ++ i) {
        x[i] = xyz[ 3 * i + 0 ];
        y[i] = xyz[ 3 * i + 1 ];
        z[i] = xyz[ 3 * i + 2 ];
    }
}

// +++++++++++++++++++++++++++++++++++++++++
// Copy the charges from the argument to this DOCKMol
// multiplying them by the units conversion factor.
void
DOCKMol::setcharges(const double* chrgs, double units_factor )
{
    for (int i = 0; i < num_atoms; ++ i) {
        charges[i] = units_factor * chrgs[i];
    }
}

