#include "dock.h"

/************************************************/
string
Library_File::hdb_atom_type_converter(int numeric_type)
{
    string          type;

    switch (numeric_type) {

    case 1:
        type = "C.";
        break;

    case 5:
        type = "C.3";
        break;

    case 6:
        type = "H";
        break;

    case 7:
        type = "H";
        break;

    case 8:
        type = "N.";
        break;

    case 9:
        type = "N.4";
        break;

    case 10:
        type = "N.3";
        break;

    case 11:
        type = "O.";
        break;

    case 12:
        type = "O.3";
        break;

    case 13:
        type = "P.";            // possible degeneracy with P. or P2
        break;

    case 14:
        type = "S.";
        break;

    case 15:
        type = "F";
        break;

    case 16:
        type = "Cl";
        break;

    case 17:
        type = "Br";
        break;

    case 18:
        type = "I";
        break;

    case 19:
        type = "Na";            // possible degeneracy with Na or K
        break;

    case 20:
        type = "Li";            // possible degeneracy with Li, Al or B
        break;

    case 21:
        type = "Ca";
        break;

    case 24:
        type = "Si";
        break;

    case 25:
        type = "Du";
        break;

    default:
        type = "Du";
        cout << "Atom type conversion not found for atom type " << numeric_type
            << ".  Du type assumed." << endl;
        break;
    }

    return type;

}

/************************************************/
// bool Library_File::read_hierarchy_db(vector<DOCKMol> &mol_branches, ifstream 
// &ifs) {
bool
Library_File::read_hierarchy_db(DOCKMol & mol, ifstream & ifs)
{
    int             i,
                    j,
                    k,
                    l;
    char            line[1000];
    string          tmp;
    int             total_atoms;
    vector < DOCKMol > mol_branches;

    // family level data
    int             clunum,
                    nmol,
                    nbr,
                    fbr;

    // molecule level data
    string          name,
                    refcode;

    // branch level data
    int             beatm,
                    bmatm,
                    bnhvy,
                    bnhyd,
                    confnum,
                    bi,
                    iconf;
    float           solvat,
                    apol;
    int             int_color;  // kxr

    // new molecule data
    DOCKMol         new_mol;
    FLOATVec        charges;
    vector < string > types;
    vector < string > color;    // kxr
    vector < string > summary_lines;
    string          family_line,
                    name_line,
                    branch_line;
    INTVec          level_counts,
                    level_values,
                    level_offsets;
    int             current_level,
                    current_count,
                    index;
    INTVec          atom_levels;
    int             next_level,
                    last_level;

    mol_branches.clear();

    // if file has no lines, return false
    if (!ifs.getline(line, 1000))
        return false;

    do {

        if (!strncmp(line, "Family", 6)) {

            // read vars from Family line
            sscanf(line, "%*s %d %d %d %d", &clunum, &nmol, &nbr, &fbr);
            family_line = line;

            // hack to bypass nbr and fbr = 0 kxr
            if (fbr < 5) {
                // cout << "Error: records are zero. Skipping to next entry" << 
                // endl;
                // ;
                // }

                nbr++;
                fbr++;

                // loop over nmols
                for (i = 0; i < nmol; i++) {

                    // ////////////////////
                    // skip first MBR loop
                    // ////////////////////

                    // read in mol name & refcode
                    if (!ifs.getline(line, 1000)) {
                        cout << "Error: Incomplete hierarchy molecule record."
                            << endl;
                        return false;
                    }

                    name_line = line;
                    name = name_line.substr(0, 47);
                    refcode = name_line.substr(47, 9);

                    // /////////////////////
                    // skip the other MBR variables
                    // /////////////////////

                    // loop over branches
                    for (j = 0; j < fbr; j++) {

                        // read branch header line
                        if (!ifs.getline(line, 1000)) {
                            cout <<
                                "Error: Incomplete hierarchy molecule record."
                                << endl;
                            return false;
                        }

                        sscanf(line, "%d %d %d %d %f %d %f %d %d", &beatm,
                               &bmatm, &bnhvy, &bnhyd, &solvat, &confnum, &apol,
                               &bi, &iconf);
                        branch_line = line;

                        // create new DOCKMol to store each branch
                        new_mol.clear_molecule();
                        new_mol.allocate_arrays(beatm, 0);
                        // new_mol.title = name;
                        new_mol.title = refcode;

                        // assign heirarchy data to mol_data string
                        if (j == 0) {
                            new_mol.mol_data = family_line + "\n";
                            new_mol.mol_data += name_line + "\n";
                        }
                        new_mol.mol_data += branch_line;

                        // allocate heirarchy data structures
                        charges.resize(bmatm);
                        types.resize(bmatm);
                        color.resize(bmatm);
                        summary_lines.resize(bmatm);

                        level_counts.clear();
                        level_counts.push_back(0);
                        level_values.clear();
                        level_values.push_back(0);

                        atom_levels.resize(beatm, 0);

                        current_level = 0;
                        next_level = 0;
                        last_level = 0;

                        // read the info into infovec struct
                        for (k = 0; k < bmatm; k++) {
                            ifs.getline(line, 1000);
                            tmp = line;

                            charges[k] = atof(tmp.substr(5, 5).c_str()) * 0.001;
                            int_color = atoi(tmp.substr(3, 2).c_str());
                            types[k] = hdb_atom_type_converter(int_color);

                            // hard code ligand coloring for now kxr
                            switch (int_color) {

                            case 1:
                                color[k] = "positive";
                                break;

                            case 2:
                                color[k] = "negative";
                                break;

                            case 3:
                                color[k] = "acceptor";
                                break;

                            case 4:
                                color[k] = "donor";
                                break;

                            case 5:
                                color[k] = "ester_o";
                                break;

                            case 6:
                                color[k] = "amide_o";
                                break;

                            case 7:
                                color[k] = "neutral";
                                break;

                            default:
                                color[k] = "null";
                                break;
                            }



                            // cout << " color " << color[k] << endl;
                            summary_lines[k] = "";
                            summary_lines[k] += tmp.substr(0, 3) + "\n";        // level
                            summary_lines[k] += tmp.substr(3, 2) + "\n";        // vdwtype
                            sprintf(line, "%f\n", charges[k]);
                            summary_lines[k] += line;   // charge
                            summary_lines[k] += tmp.substr(10, 1) + "\n";       // flagat
                            summary_lines[k] += tmp.substr(11, 2) + "\n";       // lcolor
                            summary_lines[k] += tmp.substr(13, 9) + "\n";       // polsolv
                            summary_lines[k] += tmp.substr(22, 9) + "\n";       // apolsolv
                            summary_lines[k] += types[k] + "\n";        // converted 
                                                                        // type

                            next_level = atoi(tmp.substr(0, 3).c_str());

                            // count the number of atoms in each level
                            if (k > 0) {
                                if (next_level != last_level) { // !=, >
                                    level_counts.push_back(0);
                                    level_values.push_back(0);
                                    current_level++;
                                    last_level = next_level;
                                }
                            } else
                                last_level = next_level;

                            level_values[current_level] = next_level;
                            level_counts[current_level]++;
                        }

                        // calculate the level offsets
                        level_offsets.clear();
                        level_offsets.resize(level_counts.size(), 0);

                        for (k = 1; k < level_offsets.size(); k++) {
                            level_offsets[k] =
                                level_offsets[k - 1] + level_counts[k - 1];
                        }

                        // add the level count information to mol_data
                        new_mol.mol_data += "\n";

                        for (k = 0; k < level_counts.size(); k++) {
                            sprintf(line, "%d %d,", level_counts[k],
                                    level_values[k]);
                            new_mol.mol_data += line;
                        }

                        current_level = 0;
                        current_count = 0;

                        // read coords for all confs
                        for (k = 0; k < beatm; k++) {

                            // read one line
                            ifs.getline(line, 1000);
                            tmp = line;

                            // assign coordinates
                            new_mol.x[k] =
                                atoi(tmp.substr(3, 6).c_str()) * 0.001;
                            new_mol.y[k] =
                                atoi(tmp.substr(9, 6).c_str()) * 0.001;
                            new_mol.z[k] =
                                atoi(tmp.substr(15, 6).c_str()) * 0.001;

                            atom_levels[k] = atoi(tmp.substr(0, 3).c_str());

                            // cout << atom_levels[k] <<
                            // "\t";//////////////////////////////

                            // scan for level transitions
                            if (k > 0) {

                                // if level has increased
                                if (atom_levels[k] > atom_levels[k - 1]) {

                                    // cout <<
                                    // ".";////////////////////////////////////////

                                    current_level = 0;

                                    for (l = 0; l < level_values.size(); l++) {

                                        // cout << level_values[l] << "
                                        // ";//////////////

                                        if (atom_levels[k] == level_values[l]) {
                                            current_level = l;
                                            break;
                                        }
                                    }

                                    current_count = 0;
                                }
                                // if level has decreased
                                if (atom_levels[k] < atom_levels[k - 1]) {
                                    current_level = 0;

                                    for (l = 0; l < level_values.size(); l++) {
                                        if (atom_levels[k] == level_values[l]) {
                                            current_level = l;
                                            break;
                                        }
                                    }

                                    current_count = 0;
                                }

                                if (current_count >=
                                    level_counts[current_level])
                                    current_count = 0;
                            }
                            // cout << current_level <<
                            // endl;/////////////////////////////////

                            // apply vdw & charge values from info section to
                            // atom records
                            index =
                                level_offsets[current_level] + current_count;

                            new_mol.charges[k] = charges[index];
                            new_mol.atom_data[k] = summary_lines[index];
                            new_mol.atom_types[k] = types[index];
                            new_mol.atom_names[k] = types[index];
                            new_mol.subst_names[k] = types[index];
                            new_mol.atom_color[k] = color[index];
                            current_count++;

                        }

                        // assign branches of molecule to vector of mols
                        mol_branches.push_back(new_mol);

                    }

                }

                // merge molecules into one total mol

                // count all the atoms
                total_atoms = 0;
                for (i = 0; i < mol_branches.size(); i++)
                    total_atoms += mol_branches[i].num_atoms;

                // create new hybrid molecule with all branches
                mol.clear_molecule();
                mol.allocate_arrays(total_atoms, 0);

                index = 0;

                for (i = 0; i < mol_branches.size(); i++) {

                    for (j = 0; j < mol_branches[i].num_atoms; j++) {

                        // copy molecule atom data
                        mol.x[index] = mol_branches[i].x[j];
                        mol.y[index] = mol_branches[i].y[j];
                        mol.z[index] = mol_branches[i].z[j];

                        mol.charges[index] = mol_branches[i].charges[j];
                        mol.atom_types[index] = mol_branches[i].atom_types[j];
                        mol.atom_names[index] = mol_branches[i].atom_names[j];
                        mol.subst_names[index] = mol_branches[i].subst_names[j];
                        mol.atom_color[index] = mol_branches[i].atom_color[j];
                        // cout << " mol_atom_color " << mol.atom_color[index]
                        // << endl;

                        mol.atom_data[index] = mol_branches[i].atom_data[j];

                        // no bonds, hence no rings
                        mol.atom_ring_flags[index] = false;

                        // activate atoms
                        mol.atom_active_flags[index] = true;    // /////////////

                        index++;

                    }

                    mol.mol_data += mol_branches[i].mol_data;
                    mol.mol_data += "\n";

                }

                mol.num_active_atoms = total_atoms;

                mol_branches.clear();
                // return true if mol is found
                return true;
            }

        }

    } while (ifs.getline(line, 1000));

    // if no mol found, return false
    return false;

}

/************************************************/
void
Library_File::input_parameters_input(Parameter_Reader & parm)
{
    string          tmp;

    cout << "\nMolecule Library Input Parameters" << endl;
    cout <<
        "------------------------------------------------------------------------------------------"
        << endl;

    input_file_name = parm.query_param("ligand_atom_file", "database.mol2");
   
    max_mol_limit =
        (parm.query_param("limit_max_ligands", "no", "yes no") ==
         "yes") ? true : false;
    if (max_mol_limit) {
        max_mols = atoi(parm.query_param("max_ligands", "1000").c_str());
        if (max_mols <= 0) {
            cout <<
                "ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
    } else
        max_mols = -1;
    
    bool            skip_molecule;
    skip_molecule =
        (parm.query_param("skip_molecule", "no", "yes no") ==
         "yes") ? true : false;
    if (skip_molecule) {
        initial_skip = atoi(parm.query_param("initial_skip", "0").c_str());
        if (initial_skip <= 0) {
            cout <<
                "ERROR: Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
    } else {
        initial_skip = 0;
    }

    read_solvation =
        (parm.query_param("read_mol_solvation", "no", "yes no") ==
         "yes") ? true : false;
    // read_color = (parm.query_param("read_mol_color","no","yes no") ==
    // "yes")?true:false;
    read_color = false;         // silenced keyword for now

    calc_rmsd = (parm.query_param("calculate_rmsd", "no", "yes no") == "yes") ? true : false;
    if (calc_rmsd) {
        constant_rmsd_ref =
            (parm.query_param("use_rmsd_reference_mol", "no") ==
             "yes") ? true : false;
        if (constant_rmsd_ref) {
            constant_rmsd_ref_file =
                parm.query_param("rmsd_reference_filename", "ligand_rmsd.mol2");
        }
    } else {
        constant_rmsd_ref = false;
    }
}

/************************************************/
void
Library_File::input_parameters_output(Parameter_Reader & parm, Master_Score & score, bool USE_MPI)
{

    rank_secondary_ligands = false;
    use_secondary_score = false;
    cluster_ranked_poses = false;
    write_conformers = false;
    write_secondary_conformers = false;
    num_scored_poses = 1;

    if (score.use_secondary_score)
        use_secondary_score = true;

    cout << "\nMolecule Library Output Parameters" << endl;
    cout <<
        "------------------------------------------------------------------------------------------"
        << endl;

    output_file_prefix = parm.query_param("ligand_outfile_prefix", "output");

    write_orients = (parm.query_param("write_orientations", "no", "yes no") == "yes") ? true : false;
    if (write_orients){
        if (USE_MPI){
            cout <<
                "ERROR:  DOCK cannot write orientations while running in parallel."
                << endl;
            finalize_mpi();
            exit(0);
        }
    } 

    if(score.use_secondary_score){
        num_scored_poses =
        	atoi(parm.query_param("num_primary_scored_conformers_rescored", "1").
                 c_str());
        if (num_scored_poses <= 0) {
        	cout <<
                "ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
        	exit(0);
        }
        if (num_scored_poses > 1) {
            write_conformers = true;
        	cluster_ranked_poses =
                (parm.query_param("cluster_primary_conformations", "yes", "yes no") ==
                 "yes") ? true : false;
        	if (cluster_ranked_poses) {
                cluster_rmsd_threshold =
                	atof(parm.query_param("cluster_rmsd_threshold", "2.0").
                         c_str());
                if (cluster_rmsd_threshold <= 0.0) {
                	cout <<
                        "ERROR:  Parameter must be a float greater than zero.  Program will terminate."
                        << endl;
                	exit(0);
                }
                num_clusterheads_rescore = atoi(parm.query_param("num_clusterheads_for_rescore", "5").c_str());
        	}

        } 
        if(!rank_secondary_ligands) {
            num_secondary_scored_poses =
        	    atoi(parm.query_param("num_secondary_scored_conformers_written", "1").c_str());
            if(num_secondary_scored_poses > 1)
                write_secondary_conformers = true;
            if(num_secondary_scored_poses > num_scored_poses){
                cout << "ERROR:  Number of secondary poses written cannot exceed number of primary ";
                cout << "poses rescored.  Program will terminate." << endl;
                exit(0);
            }
        	if (num_secondary_scored_poses <= 0) {
        	    cout <<
                	"ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
                	<< endl;
        	    exit(0);
        	}

        }
    } else {
        num_scored_poses =
        	atoi(parm.query_param("num_scored_conformers_written", "1").
                 c_str());
        if (num_scored_poses <= 0) {
        	cout <<
                "ERROR:  Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
        	exit(0);
        }
        if (num_scored_poses > 1) {
            write_conformers = true;
        	cluster_ranked_poses =
                (parm.query_param("cluster_conformations", "yes", "yes no") ==
                 "yes") ? true : false;
        	if (cluster_ranked_poses) {
                cluster_rmsd_threshold =
                	atof(parm.query_param("cluster_rmsd_threshold", "2.0").
                         c_str());
                if (cluster_rmsd_threshold <= 0.0) {
                	cout <<
                        "ERROR:  Parameter must be a float greater than zero.  Program will terminate."
                        << endl;
                	exit(0);
                }
        	}
        } 
    }
    if(score.use_secondary_score){
        rank_ligands = (parm.query_param("rank_primary_ligands", "no", "yes no") == "yes") ? true : false;
        if(rank_ligands){
            max_ranked = atoi(parm.query_param("max_primary_ranked", "500").c_str());
            if (max_ranked <= 0) {
                cout <<
                    "ERROR:  Parameter must be integer greater than zero.  Program will terminate."
                    << endl;
                exit(0);
            }
        }
        rank_secondary_ligands = (parm.query_param("rank_secondary_ligands", "no", "yes no") == "yes") ? true : false;
        if(rank_secondary_ligands){
            max_secondary_ranked = atoi(parm.query_param("max_secondary_ranked", "500").c_str());
            if (max_secondary_ranked <= 0) {
                cout <<
                    "ERROR:  Parameter must be integer greater than zero.  Program will terminate."
                    << endl;
                exit(0);
            }
            if(max_ranked <  max_secondary_ranked){
                cout << "ERROR:  Maximum number of poses rescored and reranked by secondary score must be" << endl;
                cout << " greater than maximum number of poses scored and ranked by primary score." << endl;
                exit(0);
            }
        }
    } else {
        rank_ligands = (parm.query_param("rank_ligands", "no", "yes no") == "yes") ? true : false;
        if (rank_ligands) {
            max_ranked =
                atoi(parm.query_param("max_ranked_ligands", "500").c_str());
            if (max_ranked <= 0) {
                cout <<
                    "ERROR:  Parameter must be integer greater than zero.  Program will terminate."
                    << endl;
                exit(0);
            }
        }
    }

}
/************************************************/
void
Library_File::initialize(int argc, char **argv, bool USE_MPI)
{

    cout << endl;
    cout << "Initializing Library File Routines...\n";
    total_mols = 0;
    completed = 0;

    output_file_orient = output_file_prefix + "_orients.mol2";
    if(use_secondary_score){
        output_file_confs = output_file_prefix + "_primary_conformers.mol2";
        output_file_scored = output_file_prefix + "_primary_scored.mol2";
        output_file_ranked = output_file_prefix + "_primary_ranked.mol2";
        secondary_output_file_confs = output_file_prefix + "_secondary_conformers.mol2";
        secondary_output_file_scored = output_file_prefix + "_secondary_scored.mol2";
        secondary_output_file_ranked = output_file_prefix + "_secondary_ranked.mol2";
    } else {
        output_file_confs = output_file_prefix + "_conformers.mol2";
        output_file_scored = output_file_prefix + "_scored.mol2";
        output_file_ranked = output_file_prefix + "_ranked.mol2";
    }

    bestmol.clear_molecule();
    bestscore = 0.0;

    if (rank_ligands)
        ranked_list.reserve(max_ranked);


    if ((!USE_MPI) || (is_master_node()))
        open_files();

    rank_list_highest_energy = 0.0;
    rank_list_highest_pos = -1;

    read_success = false;
}

/************************************************/
void
Library_File::open_files()
{
    DOCKMol         tmp_mol;
    int             i;
    string          extension;

    // identify the input file type: MOL2 or HDB (if not clear- assume MOL2)
    extension =
        input_file_name.substr(input_file_name.rfind("."),
                               input_file_name.size());

    if (extension == ".db") {
        db_type = 2;
    } else if (extension == ".mol2") {
        db_type = 1;
    } else {
        cout << "Warning: Filetype not recognized for " << input_file_name <<
            ".  Assuming MOL2 format" << endl;
        db_type = 1;
    }

    ligand_in.open(input_file_name.c_str());
    if (ligand_in.fail()) {
        cout << "\n\nCould not open " << input_file_name <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    if (write_orients) {
        ligand_out_orients.open(output_file_orient.c_str());
        if (ligand_out_orients.fail()) {
            cout << "\n\nCould not open " << output_file_orient <<
                " for writing.  Program will terminate." << endl << endl;
            exit(0);
        }
    }

    if (write_conformers) {
        ligand_out_confs.open(output_file_confs.c_str());
        if (ligand_out_confs.fail()) {
            cout << "\n\nCould not open " << output_file_confs <<
                " for writing.  Program will terminate." << endl << endl;
            exit(0);
        }
    }

    if(write_secondary_conformers) {
        secondary_ligand_out_confs.open(secondary_output_file_confs.c_str());
        if (secondary_ligand_out_confs.fail()) {
            cout << "\n\nCould not open " << secondary_output_file_confs <<
                " for writing.  Program will terminate." << endl << endl;
            exit(0);
        }
    }

    if(!rank_ligands){
        ligand_out_scored.open(output_file_scored.c_str());
        if (ligand_out_scored.fail()) {
            cout << "\n\nCould not open " << output_file_scored <<
                " for writing.  Program will terminate." << endl << endl;
            exit(0);
        }
    }
    if(use_secondary_score){
        if(!rank_secondary_ligands){
            secondary_ligand_out_scored.open(secondary_output_file_scored.c_str());
            if (secondary_ligand_out_scored.fail()) {
                cout << "\n\nCould not open " << secondary_output_file_scored <<
                    " for writing.  Program will terminate." << endl << endl;
                exit(0);
            }
        }
    }

    if (rank_ligands) {
        ligand_out_ranked.open(output_file_ranked.c_str());
        if (ligand_out_ranked.fail()) {
            cout << "\n\nCould not open " << output_file_ranked <<
                " for writing.  Program will terminate." << endl << endl;
            exit(0);
        }
    }

    if (rank_secondary_ligands) {
        secondary_ligand_out_ranked.open(secondary_output_file_ranked.c_str());
        if (secondary_ligand_out_ranked.fail()) {
            cout << "\n\nCould not open " << secondary_output_file_ranked <<
                " for writing.  Program will terminate." << endl << endl;
            exit(0);
        }
    }

    // read through the initial skip ligands
    int             successful_skip = 0;
    for (i = 0; i < initial_skip; i++) {
        if (read_mol(tmp_mol, false))
            successful_skip++;
        else {
            cout <<
                "ERROR:  initial_skip value exceeds number of ligands in database.  Program will terminate."
                << endl << endl;
            exit(0);
        }

    }

    if (initial_skip > 0)
        cout << " Skipped " << successful_skip << " ligands" << endl << endl;

}

/************************************************/
void
Library_File::close_files()
{

    ligand_in.close();

    ligand_out_orients.close();
    ligand_out_confs.close();
    ligand_out_scored.close();
    ligand_out_ranked.close(); 
    if(use_secondary_score){
        secondary_ligand_out_confs.close();
        secondary_ligand_out_ranked.close();
        secondary_ligand_out_scored.close();
    }

}

/************************************************/
bool
Library_File::read_mol(DOCKMol & mol, bool read_amber)
{

    mol.clear_molecule();

    if (db_type == 1) {         // if file is a MOL2 file

        if (max_mol_limit && (total_mols >= max_mols))
            return false;

        if (Read_Mol2(mol, ligand_in, read_color, read_solvation, read_amber)) {
            total_mols++;
            return true;
        } else {
            return false;
        }

    } else if (db_type == 2) {  // if file is a hierarchy db
/**
	  // read a new mol if branch list is empty
	  if(mol_branches.size() == 0) {
		  if(!read_hierarchy_db(mol_branches, ligand_in))
			  return false;
	  }

	  // take one branch and copy it to mol
	  copy_molecule(mol, mol_branches[mol_branches.size()-1]);
	  mol_branches.pop_back();

	  // if last branch, count the molecule as read
	  if(mol_branches.size() == 0)
		  total_mols++;
**/

        if (read_hierarchy_db(mol, ligand_in)) {
            total_mols++;

            // fail if too many mols have been read
            if (max_mol_limit && (total_mols > max_mols))
                return false;
            else
                return true;

        } else {
            return false;
        }

    } else {

        return false;

    }

}

/************************************************/
void
Library_File::write_mol(DOCKMol & mol, ofstream & ofs)
{

    Write_Mol2(mol, ofs);

}

/************************************************/
bool
Library_File::get_mol(DOCKMol & mol, bool USE_MPI, bool amber, Master_Score & score, Simplex_Minimizer & min)
{
    RANKMol         tmp_mol;
    int             i;

    if (!USE_MPI) {             // single proc code
        if (read_success) {
            if (ranked_poses.size() > 0) {
                // sort ranked_poses
                sort(ranked_poses.begin(), ranked_poses.end());

                // cluster ranked poses if requested
                if (cluster_ranked_poses) {
                    cluster_list();
                }
                
                //write out data for conformers
                write_scored_poses(USE_MPI);
                
                //analyze and write data for secondary scored confomers
                if(num_scored_poses > 1)
		            submit_secondary_conformation(score, min);
            }
        }

        // pose ranking code
        ranked_poses.clear();
        high_position = 0;
        high_score = 0.0;

        if (read_mol(mol, amber)) {

            num_orients = 0;
            num_confs = 0;
            num_anchors = 0;

            if (calc_rmsd)
                submit_rmsd_reference(mol);

            read_success = true;

            return true;
        } else {

            if (rank_ligands)
                write_ranked_ligands();

            return false;
        }

    } else {
/***/
        if (is_master_node()) { // master node code

            while (listen_for_request()) {

                if (is_send_request()) {

                    while (continue_reading_mols()) {

                        if (read_mol(mol, amber)) {
                            add_to_send_queue(mol);

                            // if HDB mols in send queue, then set continue
                            // mols true to allow all to be sent
                            if (mol_branches.size() > 0)
                                continue_mols = true;

                            read_success = true;
                        } else {
                            continue_mols = false;
                        }

                    }           // End loop over work unit of mols

                    transmit_send_queue();

                }               // End send mol code

                if (is_receive_request()) {

                    receive_flag = true;

                    ranked_poses.clear();

                    while (get_from_recv_queue(tmp_mol)) {

                        ranked_poses.push_back(tmp_mol);

                    }           // End loop over received mols

                    // copy docking stats from MPI stats struct
                    num_anchors = stats[0];
                    num_orients = stats[1];
                    num_confs = stats[2];

                    if (read_success) {
                        if (ranked_poses.size() > 0) {
   		            // sort ranked_poses
                            sort(ranked_poses.begin(), ranked_poses.end());

                            // cluster ranked poses if requested
                            if (cluster_ranked_poses) {
                                cluster_list();
                            }

                            //write out data for conformers
                            write_scored_poses(USE_MPI);

                            //analyze and write data for secondary scored confomers
                            if(num_scored_poses > 1)
                                submit_secondary_conformation(score, min);

                        }
                    }

                }               // End receive mol code

            }                   // End looping over MPI listener

            if (rank_ligands)
                write_ranked_ligands();

            close_files();
            return false;

        } else if (is_client_node()) {  // client node code

            // sort ranked poses
            if (ranked_poses.size() > 0) {
                completed++;
                sort(ranked_poses.begin(), ranked_poses.end());
            }
            // add ranked poses to return list
            recv_queue.clear();
            for (i = 0; i < ranked_poses.size(); i++)
                add_to_recv_queue(ranked_poses[i]);

            // copy docking stats to MPI data struct
            stats[0] = num_anchors;
            stats[1] = num_orients;
            stats[2] = num_confs;

            // init pose ranking
            ranked_poses.clear();
            high_position = 0;
            high_score = 0.0;

            // get molecule from server
            if (get_from_send_queue(mol)) {

                num_anchors = 0;
                num_orients = 0;
                num_confs = 0;

                if (calc_rmsd)
                    submit_rmsd_reference(mol);

                read_success = true;

                return true;
            } else
                return false;

        }
/***/
        return false;           // to squelch warnings
    }


}

/************************************************/
bool
Library_File::submit_orientation(DOCKMol & mol, Master_Score & score, bool  orient_ligand)
{

    bool valid_orientation = false;
    num_orients++;
    if(orient_ligand){
        // compute the score for the molecule
        if(score.use_score){
            if (score.use_primary_score) 
                valid_orientation = score.compute_primary_score(mol);
            if (!valid_orientation) {
                return false;
            }
        }
    }

    if (write_orients) {

        ligand_out_orients << "\n########## Name:\t\t" << mol.
            title << "_" << num_orients << endl;

        if (calc_rmsd)
            ligand_out_orients << "########## RMSD:\t\t" << calculate_rmsd(mol)
                << endl;

        if(score.use_score)
            ligand_out_orients << mol.current_data << endl;
        
        write_mol(mol, ligand_out_orients);

    }
    return true;

}

/************************************************/
void
Library_File::submit_conformation(Master_Score & score)
{

    if (write_conformers) {
    
        if(score.use_score)
            sort(ranked_poses.begin(), ranked_poses.end());

        for(int i=0;i<ranked_poses.size();i++){
            ligand_out_confs << "\n########## Name:\t\t" << ranked_poses[i].mol.
                title << "_" << i + 1 << endl;

            if (calc_rmsd)
                ligand_out_confs << "########## RMSD:\t\t" << calculate_rmsd(ranked_poses[i].mol) << endl;
            if(score.use_score)
                ligand_out_confs << ranked_poses[i].mol.current_data << endl;

            write_mol(ranked_poses[i].mol, ligand_out_confs);
        }

    }
    
}
/************************************************/
void
Library_File::submit_secondary_conformation(Master_Score & score, Simplex_Minimizer & min)
{
    ostringstream   text;    
    int             i;
    DOCKMol         tmp_dockmol;
    for(i=0;i<ranked_poses.size();i++){
       
        if(cluster_ranked_poses){ 

            if(cluster_assignments[i] > 0){
                //collect data about primary score
                if (calc_rmsd) 
                    text << "########## RMSD:\t\t" <<
                        calculate_rmsd(ranked_poses[i].mol) << endl;

                text << "########## Cluster Size:\t" <<
                            cluster_assignments[i] << endl;

                text << ranked_poses[i].mol.current_data << endl;

                ranked_poses[i].mol.primary_data = text.str();
        
                text.str("");            

                //minimize or score using secondary function
                if(min.secondary_min_pose){
                    min.secondary_minimize_pose(ranked_poses[i].mol, score);
                } else {
                    score.compute_secondary_score(ranked_poses[i].mol);
                }
                ranked_poses[i].score = ranked_poses[i].mol.current_score;
            } else {
                ranked_poses[i].score = -MIN_FLOAT;
            }
        
        } else {
	    //collect data about primary score
            if (calc_rmsd)
                text << "########## RMSD:\t\t" <<
                    calculate_rmsd(ranked_poses[i].mol) << endl;

            text << ranked_poses[i].mol.current_data << endl;
           
            ranked_poses[i].mol.primary_data = text.str();

            text.str("");
         
            //minimize or score using secondary function
            if(min.secondary_min_pose){
                min.secondary_minimize_pose(ranked_poses[i].mol, score);
            } else {
                score.compute_secondary_score(ranked_poses[i].mol);
            }
            ranked_poses[i].score = ranked_poses[i].mol.current_score;

        }
    }

    //sort by secondary scores
    sort(ranked_poses.begin(), ranked_poses.end());

    while(ranked_poses.size() > num_secondary_scored_poses){
	    ranked_poses.pop_back();
    }

    //loop until user parameter reached
    for(i=0;i<ranked_poses.size();i++){
        secondary_ligand_out_confs << "\n########## Name:\t\t" << ranked_poses[i].mol.
            title << "_" << i + 1 << endl;
        
        secondary_ligand_out_confs << "########## SECONDARY SCORE" << endl;
        
        if (calc_rmsd)
            secondary_ligand_out_confs << "########## RMSD:\t\t" << calculate_rmsd(ranked_poses[i].mol) << endl;
        
        secondary_ligand_out_confs << ranked_poses[i].mol.current_data;
        
        secondary_ligand_out_confs << "########## PRIMARY SCORE" << endl;
        secondary_ligand_out_confs << ranked_poses[i].mol.primary_data << endl;

        write_mol(ranked_poses[i].mol, secondary_ligand_out_confs);
	
    }

    if(ranked_poses[0].mol.num_atoms > 0) {
        copy_molecule(tmp_dockmol, ranked_poses[0].mol);
        poses_for_rescore.push_back(tmp_dockmol);
    }

    
}
/************************************************/
void
Library_File::submit_scored_pose(DOCKMol & mol, Master_Score & score,
                                 Simplex_Minimizer & min)
{
    char            score_convert[100];
    RANKMol         tmp_rankmol;
    DOCKMol	        tmp_dockmol;
    int             i;
    float           max_score;
    int             max_position;
    bool            success;

    num_confs++;
    if (score.use_score) {
        if(!min.minimize_ligand){
            if(!min.final_min){
                score.compute_primary_score(mol);
            }
        }
    } else {
        mol.current_score = 0.0;
        mol.current_data = "";
    }

    // Multi-pose output
    if (ranked_poses.size() < num_scored_poses) {

        // Add the molecule
        ranked_poses.push_back(tmp_rankmol);
        ranked_poses[ranked_poses.size() - 1].score = mol.current_score;
        ranked_poses[ranked_poses.size() - 1].data = mol.current_data;
        copy_molecule(ranked_poses[ranked_poses.size() - 1].mol, mol);
        
        // find position with the highest score
        max_score = ranked_poses[0].score;
        max_position = 0;

        for (i = 0; i < ranked_poses.size(); i++) {
            if (ranked_poses[i].score > max_score)
                max_position = i;
        }

        high_score = max_score;
        high_position = max_position;


    } else if (mol.current_score < high_score) {

        // add mol to the high position
        ranked_poses[high_position].data = mol.current_data;
        ranked_poses[high_position].score = mol.current_score;
        copy_molecule(ranked_poses[high_position].mol, mol);
        
        // find position with the highest score
        max_score = ranked_poses[0].score;
        max_position = 0;

        for (i = 0; i < ranked_poses.size(); i++) {
            if (ranked_poses[i].score > max_score)
                max_position = i;
        }

        high_score = max_score;
        high_position = max_position;


    }


}

/************************************************/
void
Library_File::write_scored_poses(bool USE_MPI)
{
    int             i,
                    j;
    // int insert_point;
    ofstream        ofs;
    SCOREMol        tmp_mol;
    DOCKMol         tmp_dockmol;

  /**
  change if statement- only skip the writing if it is a client
  for both single proc. and master mode, writing should occur
  **/

    if (ranked_poses.size() > 0) {

        if (USE_MPI) {
            cout << endl << "-----------------------------------" << endl;
            cout << "Molecule: " << ranked_poses[0].mol.title << endl << endl;
        }
        // print out the ligand info
        cout << " Anchors:\t\t" << num_anchors << endl;
        cout << " Orientations:\t\t" << num_orients << endl;
        cout << " Conformations:\t\t" << num_confs << endl;

        cout << endl;
        if (ranked_poses[0].mol.num_atoms > 0)
            if(use_secondary_score)
                cout << " Primary Score" << endl;
            for (i = 0; i < ranked_poses[0].data.size(); i++)
                if (ranked_poses[0].data[i] != '#')
                    cout << ranked_poses[0].data[i];

        if (cluster_ranked_poses) {

            for (i = 0; i < ranked_poses.size(); i++) {

                if (ranked_poses[i].mol.num_atoms > 0) {

                    // if clustering is used, and molecule is a
                    // clusterhead, write it out
                    if (cluster_assignments[i] > 0) {
                        ligand_out_scored << "\n########## Name:\t\t" <<
                            ranked_poses[i].mol.title << endl;

                        if (calc_rmsd)
                            ligand_out_scored << "########## RMSD:\t\t" <<
                                calculate_rmsd(ranked_poses[i].mol) << endl;

                        ligand_out_scored << "########## Cluster Size:\t" <<
                            cluster_assignments[i] << endl;
                        ligand_out_scored << ranked_poses[i].data << endl;
                        write_mol(ranked_poses[i].mol, ligand_out_scored);
                     
                    }

                }

            } 
        } else {
            if (ranked_poses[0].mol.num_atoms > 0) {
                ligand_out_scored << "\n########## Name:\t\t" <<                
                    ranked_poses[0].mol.title << endl;

                if (calc_rmsd)
                    ligand_out_scored << "########## RMSD:\t\t" <<
                        calculate_rmsd(ranked_poses[0].mol) << endl;

                ligand_out_scored << ranked_poses[0].mol.current_data << endl;
                write_mol(ranked_poses[0].mol, ligand_out_scored);
            }
        }

        if (rank_ligands) {

            if (ranked_poses[0].mol.num_atoms > 0) {

                tmp_mol.first = ranked_poses[0].score;
                copy_molecule(tmp_mol.second, ranked_poses[0].mol);
                tmp_mol.second.score_text_data = ranked_poses[0].data;

                ranked_list.push_back(tmp_mol);

                if (ranked_list.size() > 2 * max_ranked) {
                    sort(ranked_list.begin(), ranked_list.end(),
                         less_than_pair);

                    while (ranked_list.size() > max_ranked)
                        ranked_list.pop_back();
                }

            }
        }
        if(num_scored_poses < 2){
            if(use_secondary_score){
                if(ranked_poses[0].mol.num_atoms > 0) {
                    copy_molecule(tmp_dockmol, ranked_poses[0].mol);
                    poses_for_rescore.push_back(tmp_dockmol);
                }
            }
        }

    }


}

/************************************************/
void
Library_File::write_ranked_ligands()
{
    int             i;

    sort(ranked_list.begin(), ranked_list.end(), less_than_pair);

    while (ranked_list.size() > max_ranked)
        ranked_list.pop_back();

    if (rank_ligands) {
        for (i = 0; i < ranked_list.size(); i++) {
            ligand_out_ranked << "\n########## Name:\t\t" << ranked_list[i].
                second.title << endl;

            if (calc_rmsd)
                ligand_out_ranked << "########## RMSD:\t\t" <<
                    calculate_rmsd(ranked_list[i].second) << endl;

            ligand_out_ranked << ranked_list[i].second.score_text_data << endl;

            write_mol(ranked_list[i].second, ligand_out_ranked);

        }
    }

}

/************************************************/
void
Library_File::submit_rmsd_reference(DOCKMol & mol)
{
    ifstream        rmsd_ref;

    if (!constant_rmsd_ref)
        copy_molecule(rmsd_reference, mol);
    else {
        rmsd_ref.open(constant_rmsd_ref_file.c_str());
        Read_Mol2(rmsd_reference, rmsd_ref, false, false, false);
        rmsd_ref.close();
    }

}

/************************************************/
float
Library_File::calculate_rmsd(DOCKMol & mol)
{
    float           rmsd;
    int             i;
    int hcount;

    rmsd = 0.00;

    if (rmsd_reference.num_atoms != mol.num_atoms)
        return 0.00;

    hcount = 0;

    for (i = 0; i < mol.num_atoms; i++) {
        if (mol.amber_at_heavy_flag[i]) {
            rmsd +=
                (pow((mol.x[i] - rmsd_reference.x[i]), 2) +
                 pow((mol.y[i] - rmsd_reference.y[i]),
                     2) + pow((mol.z[i] - rmsd_reference.z[i]), 2));
            hcount++;
        }
    }

    //rmsd = rmsd / mol.num_atoms;
    rmsd = rmsd/hcount;
    rmsd = sqrt(rmsd);

    return rmsd;

}

/************************************************/
float
Library_File::calculate_rmsd(DOCKMol & refmol, DOCKMol & mol)
{
    float           rmsd;
    int             i;
    int hcount;

    rmsd = 0.00;

    if (refmol.num_atoms != mol.num_atoms)
        return 0.00;

    hcount = 0;

    for (i = 0; i < mol.num_atoms; i++) {
        if (mol.amber_at_heavy_flag[i]) {
            rmsd +=
                (pow((mol.x[i] - refmol.x[i]), 2) +
                 pow((mol.y[i] - refmol.y[i]),
                     2) + pow((mol.z[i] - refmol.z[i]), 2));
            hcount++;
        }
    }

    // rmsd = rmsd / mol.num_atoms;
    rmsd = rmsd/hcount;
    rmsd = sqrt(rmsd);

    return rmsd;
}
/************************************************/
void
Library_File::cluster_list(){

    float           crmsd;
    int i, j;

    // initialize clustering data
    cluster_assignments.clear();
    cluster_assignments.resize(ranked_poses.size(), 1);

    // loop over ranked poses
    for (i = 0; i < ranked_poses.size() - 1; i++) {

        // if mol is not assigned to a cluster already
        if (cluster_assignments[i] > 0) {

            // loop over nested poses
            for (j = i + 1; j < ranked_poses.size(); j++) {

                if (cluster_assignments[j] > 0) {
                    crmsd =
                        calculate_rmsd(ranked_poses[i].mol,
                                       ranked_poses[j].mol);

                    // if second mol is w/in threhsold of the first
                    if (crmsd < cluster_rmsd_threshold) {
                        cluster_assignments[i]++;
                        cluster_assignments[j] = 0;
                    }
                }
            }
        }
    }
}
/************************************************/
void
Library_File::secondary_rescore_poses(Master_Score & score, Simplex_Minimizer & min)
{
    int i, j;
    ostringstream text;
    SCOREMol tmp_mol;

    if(poses_for_rescore.size() > 0){
        
        for(i=0;i<poses_for_rescore.size();i++){
            if(num_scored_poses < 2){
                //collect data from primary scoring function
                if (calc_rmsd) 
                    text << "########## RMSD:\t\t" <<
                            calculate_rmsd(poses_for_rescore[i]) << endl;
                text << poses_for_rescore[i].current_data;

                poses_for_rescore[i].primary_data = text.str();
            }

            //collect ligands for primary ranking before rescoring
            if(rank_ligands){

                tmp_mol.first = poses_for_rescore[i].current_score;
                copy_molecule(tmp_mol.second, poses_for_rescore[i]);

                ranked_secondary_list.push_back(tmp_mol);

            } else {
                //rescore if simply rescoring molecules
                if(num_scored_poses < 2){
                    if(min.secondary_min_pose){
                        min.secondary_minimize_pose(poses_for_rescore[i], score);
                    } else {
                        score.compute_secondary_score(poses_for_rescore[i]);
                    }
                }
                //write score to output file
                cout << endl << "-----------------------------------" << endl;
                cout << "Molecule: " << poses_for_rescore[i].title << endl << endl;
                cout << " Secondary Score"  << endl;
                for (j = 0; j < poses_for_rescore[i].current_data.size(); j++){
                    if (poses_for_rescore[i].current_data[j] != '#')
                        cout << poses_for_rescore[i].current_data[j];
                }
            }
            text.str("");
        }

        //sort ligands by primary scoring function
        sort(ranked_secondary_list.begin(), ranked_secondary_list.end(), less_than_pair);
        //prune poses by user defined maximum for primary scoring
        while(ranked_secondary_list.size() > max_ranked)
            ranked_secondary_list.pop_back();

        //rescore pruned list
        for(i=0;i<ranked_secondary_list.size();i++){
            if(num_scored_poses < 2){ 
                if(min.secondary_min_pose){
                    min.secondary_minimize_pose(ranked_secondary_list[i].second, score);
                } else {
                    score.compute_secondary_score(ranked_secondary_list[i].second);
                }
            }
            //write score to output file
            cout << endl << "-----------------------------------" << endl;
            cout << "Molecule: " << ranked_secondary_list[i].second.title << endl << endl;
            cout << " Secondary Score"  << endl;
            for (j = 0; j < ranked_secondary_list[i].second.current_data.size(); j++){
                if (ranked_secondary_list[i].second.current_data[j] != '#')
                    cout << ranked_secondary_list[i].second.current_data[j];
            }
        }

    }
    
}
/************************************************/
void
Library_File::submit_secondary_pose()
{
    int i;
    
    if(poses_for_rescore.size() > 0){
        
        if((!rank_ligands) && (!rank_secondary_ligands)){
            
            for(i=0;i<poses_for_rescore.size();i++){
            
                secondary_ligand_out_scored << "\n########## Name:\t\t" << poses_for_rescore[i].title << endl;
        
                secondary_ligand_out_scored << "########## SECONDARY SCORE" << endl;
                if (calc_rmsd) 
                    secondary_ligand_out_scored << "########## RMSD:\t\t" <<
                            calculate_rmsd(poses_for_rescore[i]) << endl;
                secondary_ligand_out_scored << poses_for_rescore[i].current_data;
            
                secondary_ligand_out_scored << "########## PRIMARY SCORE" << endl;
                secondary_ligand_out_scored << poses_for_rescore[i].primary_data << endl;

                write_mol(poses_for_rescore[i], secondary_ligand_out_scored);
            }
        }

        if((rank_ligands) && (!rank_secondary_ligands)) {
        
            for(i=0;i<ranked_secondary_list.size();i++){
           
                secondary_ligand_out_scored << "\n########## Name:\t\t" <<
                        ranked_secondary_list[i].second.title << endl;

                secondary_ligand_out_scored << "########## SECONDARY SCORE" << endl;
                if (calc_rmsd)
                    secondary_ligand_out_scored << "########## RMSD:\t\t" <<
                            calculate_rmsd(ranked_secondary_list[i].second);
                secondary_ligand_out_scored << ranked_secondary_list[i].second.current_data; 

                secondary_ligand_out_scored << "########## PRIMARY SCORE" << endl;
                secondary_ligand_out_scored << ranked_secondary_list[i].second.primary_data << endl;

                write_mol(ranked_secondary_list[i].second, secondary_ligand_out_scored);
           }
        }

    
        if(rank_secondary_ligands){

            for(i=0;i<ranked_secondary_list.size();i++)
                ranked_secondary_list[i].first = ranked_secondary_list[i].second.current_score;
            sort(ranked_secondary_list.begin(), ranked_secondary_list.end(), less_than_pair);	
        
            //prune poses by user defined maximum for secondary scoring
            while(ranked_secondary_list.size() > max_secondary_ranked)
                ranked_secondary_list.pop_back();

            for(i=0;i<ranked_secondary_list.size();i++){

                
                secondary_ligand_out_ranked << "\n########## Name:\t\t" <<
                            ranked_secondary_list[i].second.title << endl;

                secondary_ligand_out_ranked << "########## SECONDARY SCORE" << endl;
                if (calc_rmsd)
                    secondary_ligand_out_ranked << "########## RMSD:\t\t" <<
                             calculate_rmsd(ranked_secondary_list[i].second);
                secondary_ligand_out_ranked << ranked_secondary_list[i].second.current_data;

                secondary_ligand_out_ranked << "########## PRIMARY SCORE" << endl;
                secondary_ligand_out_ranked << ranked_secondary_list[i].second.primary_data << endl;

                write_mol(ranked_secondary_list[i].second, secondary_ligand_out_ranked);
            }

        }
    }

    close_files();
}
