// ******************************************************************************
// 
// score_chemgrid.cpp
// 
// implementation of class Shape_Filter
// implementation of class Chemgrid_Score 
// 
// This class implements grid-based scoring from CHEMGRID. 
// It uses an alternate model of electrostatic interaction caclulated 
// from electrostatic potential map generated by solving the PB equation
// for the protein receptor using the program DelPhi. In addition, it also 
// implements grid-based ligand and receptor desolvation methods.  
// It is based on the other derived classes of Base_Score.
// 
// Shape Filter - originally distmap in dock3.3.54. 
// Author/s: Brian K. Shoichet
// Ported to DOCK6 by Kaushik Raha
// Chemgrid_Score 
// o vdw and electrostatic interaction calculation from dock3.5.54
// Author/s: Brian K. Shoichet
// Ported do DOCK6 by Kaushik Raha
// o Ligand desolvation calculation from dock3.5.54
// Author/s: Brian K. Shoichet
// Ported to DOCK6 by Kaushik Raha
// o Receptor desolvation: Occupancy desolvation method
// Author/s: Kaushik Raha
// 
// Other authors: Terry Lang
// 
// Copyright 2006 
// University of California, San Francisco 
// 
// ******************************************************************************

#include "dock.h"
#include "tnt/tnt.h"            // use 3D array library for DelPhi and Chemgrid
using namespace TNT;            // kxr

const int MAXDIX = 130;         // max dimensions for distmap and Chemgrids kxr
const int MAXDIY = 130;         // dyn. memory alloc. to be implemented.
const int MAXDIZ = 130;         // for distmap grid kxr
const int PSIZE = 65;           // for delphi kxr

Array3D < float >phimap(PSIZE, PSIZE, PSIZE);
Array3D < float >solvgrd(MAXDIX, MAXDIY, MAXDIZ);
Array3D < float >rdsol(MAXDIX, MAXDIY, MAXDIZ);
Array3D < int  >shape(MAXDIX, MAXDIY, MAXDIZ);

/************************************************/
// shape filter equivalent to distmap from dock3.5.54 kxr
void
Shape_Filter::input_parameters(Parameter_Reader & parm)
{
    string          tmp;

    // kxr shape grid parameteres

    cout << "\nShape Filter Parameters" << endl;
    cout <<
        "------------------------------------------------------------------------------------------"
        << endl;

    tmp = parm.query_param("shape_filter", "no", "yes no");

    if (tmp == "yes")
        shape_filter = 1;
    else
        shape_filter = 0;

    if (shape_filter == 1) {
        shape_file_prefix = parm.query_param("shape_grid_prefix", "grid");
        max_mismatch =
            atoi(parm.query_param("max_shape_mismatch", "2").c_str());
        if (max_mismatch <= 0) {
            cout <<
                "ERROR: Parameter must be an integer greater than zero.  Program will terminate."
                << endl;
            exit(0);
        }
    }

}


/************************************************/
void
Shape_Filter::initialize()
{

    if (shape_filter) {
        cout << "Intializing Shape Filter Routines..." << endl;
        read_shape_grid();
        use_shape_score = true;
    } else
        use_shape_score = false;

}


/************************************************/
// implementation of distmap grid from dock3554 kxr
void
Shape_Filter::read_shape_grid()
{

    string          fname;
    string          tmp;
    char            line[100];
    char            cgrid;
    int             cadif1,
                    cadif2,
                    cadif3;
    int             nlist,
                    ascore,
                    sumlst;
    int             i,
                    j,
                    k,
                    l;


    fname = shape_file_prefix + ".map";
    shape_grid_in = fopen(fname.c_str(), "rb");

    if (shape_grid_in == NULL) {
        cout << "\n\nCould not open " << fname <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    cout << " Reading the shape grid from " << fname << endl;


    l = 0;
    sumlst = 0;
    while (fgets(line, 100, shape_grid_in) != NULL) {
        if (l == 0) {
            sscanf(line, "%i %i %i %i %i %i %i", &cadif1, &cadif2, &cadif3, &grddiv, &xmax, &ymax, &zmax);      // first 
                                                                                                                // line 
                                                                                                                // read 
                                                                                                                // grid 
                                                                                                                // params 
                                                                                                                // kxr
            l++;
            // Check memory bounds
            if (cadif1 > MAXDIX || cadif2 > MAXDIY || cadif3 > MAXDIZ) {
                cout <<
                    "\n\n Memory bounds exceeded in shape grid. Program will Terminate"
                    << endl;
                exit(0);
            }
        } else {
            for (i = 0; i < cadif1 + 1; ++i) {
                for (j = 0; j < cadif2 + 1; ++j) {
                    while (sumlst < cadif3 + 1) {
                        l++;
                        if (l > 2)
                          fgets(line, 100, shape_grid_in);  // readmap
                        tmp = line;
                        nlist = atoi(tmp.substr(0, 3).c_str());
                        ascore = atoi(tmp.substr(3, 3).c_str());
                        for (k = sumlst; k < sumlst + nlist; ++k) {
                            shape[i][j][k] = ascore;
                        }
                        sumlst = sumlst + nlist;
                    }
                    sumlst = 0;
                }
            }
        }


    }
    fclose(shape_grid_in);

}


/************************************************/
bool
Shape_Filter::match_shape(DOCKMol & mol, bool bypass, bool shp_score)
{
    int             num_mismatch;

    if (bypass == false)
        return true;

    if (shape_filter == 1) {

        num_mismatch = get_shape_score(mol);

        if (shp_score == true) {
            if ((num_mismatch <= max_mismatch)
                && (shape_score > 0.5f && shape_score < 2.0f)
                && (num_mismatch != -1))
                return true;
            else
                return false;


        } else {

            if ((num_mismatch <= max_mismatch) && (num_mismatch != -1))
                return true;
            else
                return false;

        }

    } else
        return true;

}


/************************************************/
int
Shape_Filter::get_shape_score(DOCKMol & mol)
{
    // kxr 
    int             atom;
    unsigned int    mismatch_count = 0;
    int             lx,
                    ly,
                    lz,
                    tmp;
    shape_score = 0;


    for (atom = 0; atom < mol.num_atoms; atom++) {
        if (mol.atom_active_flags[atom]) {
            lx = xmax - NINT(mol.x[atom] * grddiv);
            if ((lx >= 0) && (lx <= MAXDIX)) {
                ly = ymax - NINT(mol.y[atom] * grddiv);
                if ((ly >= 0) && (ly <= MAXDIY)) {
                    lz = zmax - NINT(mol.z[atom] * grddiv);
                    if ((lz >= 0) && (lz <= MAXDIZ)) {
                        tmp = shape[lx][ly][lz];
                        if (tmp == 128) {
                            mismatch_count++;
                        } else
                            shape_score += tmp;
                    } else
                        return -1;
                } else
                    return -1;
            } else
                return -1;
        }
    }
    shape_score = shape_score / mol.num_atoms;

    return (mismatch_count);
}

/************************************************/
/************************************************/
/************************************************/
Chemgrid_Score::Chemgrid_Score()
{

    avdw = NULL;
    bvdw = NULL;
    es = NULL;

    dslb = NULL;
    dslx = NULL;

    phi = NULL;

    vdwA = NULL;
    vdwB = NULL;

}
/************************************************/
Chemgrid_Score::~Chemgrid_Score()
{

    delete[]avdw;
    delete[]bvdw;
    delete[]es;

    delete[]dslb;
    delete[]dslx;

    delete[]phi;

    delete[]vdwA;
    delete[]vdwB;

}
/************************************************/
void
Chemgrid_Score::input_parameters(Parameter_Reader & parm, bool & primary_score,
                                 bool & secondary_score)
{
    // kxr Chemgrid score: Alternate Scoring Scheme for Electrostatics
    // Solvation and Sterics 
    // vdw calc. from chemgrids; electrostatics from DelPhi phimaps
    // various solvation models available see documentation

    string          tmp;

    use_primary_score = false;
    use_secondary_score = false;
    use_chemgrid_score = false;
    use_conf_entropy = false;
    add_ligand_internal = false;
    use_delphi_score = false;
    use_solv_score = false;
    total_ligand_dsolv = false;
    use_odm_score = false;
    use_recep_dsolv = false;
    write_atomic_energy = false;
    redist_pos_desol = false;


    cout << "\nDock3.5 Score Parameters" << endl;
    cout <<
        "------------------------------------------------------------------------------------------"
        << endl;

    if (!primary_score) {
        tmp = parm.query_param("dock3.5_score_primary", "no", "yes no");
        if (tmp == "yes")
            use_primary_score = true;
        else
            use_primary_score = false;

        primary_score = use_primary_score;
    }

    if (!secondary_score) {
        tmp = parm.query_param("dock3.5_score_secondary", "no", "yes no");
        if (tmp == "yes")
            use_secondary_score = true;
        else
            use_secondary_score = false;

        secondary_score = use_secondary_score;
    }


    if (use_primary_score || use_secondary_score)
        energy_score = 1;
    else
        energy_score = 0;

    if (energy_score == 1) {

        tmp = parm.query_param("dock3.5_vdw_score", "yes", "yes no");
        if (tmp == "yes") {
            use_chemgrid_score = true;
            file_prefix = parm.query_param("dock3.5_grd_prefix", "chem5");
        } else
            cout << " Warning: Steric interaction will not be calculated!!! " <<
                endl;

        // option to read in Delphi grids kxr 041305
        tmp = parm.query_param("dock3.5_electrostatic_score", "yes", "yes no");
        if (tmp == "yes") {
            use_delphi_score = true;
            interpol_method = 2;        // 1 for Demer method, 2 for dock3554 
        } else
            cout <<
                " Warning: Electrostatic interaction will not be calculated!!! "
                << endl;

        // silence conformational entropy for dock6 release
        // tmp =
        // parm.query_param("dock3.5_conformational_entropy_score","no","yes
        // no");
        // if(tmp == "yes") {
        // use_conf_entropy = true;
        // }

        // option to add ligand internal energy to score kxr 010506
        tmp =
            parm.query_param("dock3.5_ligand_internal_energy", "no", "yes no");
        if (tmp == "yes") {
            add_ligand_internal = true;
        }

        tmp =
            parm.query_param("dock3.5_ligand_desolvation_score", "no",
                             "no total volume");
        if (tmp == "total") {
            use_solv_score = true;
            total_ligand_dsolv = true;
        } else if (tmp == "volume") {
            use_solv_score = true;
            if (use_solv_score) {
                solv_file_prefix =
                    parm.query_param("dock3.5_solvent_occlusion_file",
                                     "solvmap");
                tmp = parm.
                    query_param("dock3.5_redistribute_positive_desolvation",
                                "no", "no yes");
                if (tmp == "yes")
                    redist_pos_desol = true;
            }
        }


        // Temporarily disabled by srb on December 4, 2006
        // See the cvs log for the email thread.
        //tmp = parm.query_param("dock3.5_occupancy_desolvation_score", "no",
        //                     "no yes");
        //if (tmp == "yes") {
        //    use_solv_score = true;
        //    use_odm_score = true;
        //    use_recep_dsolv = true; 
        //}

        tmp = parm.query_param("dock3.5_receptor_desolvation_score", "no",
                             "yes no");
        if (tmp == "yes") {
            use_solv_score = true;
            use_recep_dsolv = false;
            cout <<
                " Warning: Receptor desolvation not functional. Only ligand desolvation will be calculated"
                << endl;
            use_odm_score = false;
        }


        tmp =
            parm.query_param("dock3.5_write_atomic_energy_contrib", "no",
                             "yes no");
        if (tmp == "yes")
            write_atomic_energy = true;


        vdw_scale =
            atof(parm.query_param("dock3.5_score_vdw_scale", "1").c_str());
        if (vdw_scale <= 0.0) {
            bool            off;
            off =
                (parm.
                 query_param("dock3.5_score_turn_off_vdw", "yes",
                             "yes no") == "yes") ? true : false;
            if (!off) {
                cout <<
                    "ERROR: Parameter must be a float greater than zero.  Program will terminate."
                    << endl;
                exit(0);
            }
        }
        es_scale =
            atof(parm.query_param("dock3.5_score_es_scale", "1").c_str());
        if (es_scale <= 0.0) {
            bool            off;
            off =
                (parm.
                 query_param("dock3.5_score_turn_off_es", "yes",
                             "yes no") == "yes") ? true : false;
            if (!off) {
                cout <<
                    "ERROR: Parameter must be a float greater than zero.  Program will terminate."
                    << endl;
                exit(0);
            }
        }

        use_score = true;

    }
}


/************************************************/
void
Chemgrid_Score::initialize(AMBER_TYPER & typer)
{

    if (energy_score == 1) {

        cout << "Initializing Dock3.5 Score Routines..." << endl;

        if (use_chemgrid_score)
            read_chm_grid();
        // else
        // read_nrg_grid(); 

        if (use_delphi_score) {
            read_phi_grid();
            if (interpol_method == 1)
                phi_corner_coords();    // Read DelPhi grid kxr
        }

        if (use_solv_score && !total_ligand_dsolv && !use_odm_score)    // ligand 
                                                                        // desolvation 
                                                                        // only 
                                                                        // kxr
            read_solv_grid();

        if (use_solv_score && !total_ligand_dsolv && use_recep_dsolv)   // DelPhi 
                                                                        // computed 
                                                                        // receptor 
                                                                        // desolvation 
                                                                        // kxr
            read_rdsol_grid();

        if (use_solv_score && use_odm_score)
            read_odm_grid();    // Occupancy desolvation score kxr

        init_vdw_energy(typer);
        calc_corner_coords();
        use_score = true;

    } else
        use_score = false;


}
/************************************************/
// Read CHEMGRID generated vdw and elec. grids 
// Used by dock3.5.54 kxr 
void
Chemgrid_Score::read_chm_grid()
{
    string          fname;
    int             i,
                    j;

    fname = file_prefix + ".cmg";
    grid_in = fopen(fname.c_str(), "rb");

    if (grid_in == NULL) {
        cout << "\n\nCould not open " << fname <<
            " for reading chemgrids.  Program will terminate." << endl << endl;
        exit(0);
    }

    cout << " Reading chemgrids from " << fname << endl;

    fread(&size, sizeof(int), 1, grid_in);
    fread(&spacing, sizeof(float), 1, grid_in);
    fread(origin, sizeof(float), 3, grid_in);
    fread(span, sizeof(int), 3, grid_in);

    fread(&atom_model, sizeof(int), 1, grid_in);
    fread(&att_exp, sizeof(int), 1, grid_in);
    fread(&rep_exp, sizeof(int), 1, grid_in);

    avdw = new float[size];
    bvdw = new float[size];
    es = new float[size];

    fread((void *) bvdw, sizeof(float), size, grid_in);
    fread((void *) avdw, sizeof(float), size, grid_in);
    fread((void *) es, sizeof(float), size, grid_in);


    fclose(grid_in);

}

/************************************************/
// Read in the occupancy desolvation grid kxr 1205
void
Chemgrid_Score::read_odm_grid()
{
    string          fname;
    int             i,
                    j;

    fname = file_prefix + ".dsl";
    grid_in = fopen(fname.c_str(), "rb");

    if (grid_in == NULL) {
        cout << "\n\nCould not open " << fname <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    cout << " Reading occupancy desolvation grid from " << fname << endl;

    fread(&size, sizeof(int), 1, grid_in);
    fread(&spacing, sizeof(float), 1, grid_in);
    fread(origin, sizeof(float), 3, grid_in);
    fread(span, sizeof(int), 3, grid_in);

    // dslb = (float *)malloc(size * sizeof(float));
    // dslx = (float *)malloc(size * sizeof(float));
    dslb = new float[size];
    dslx = new float[size];

    fread((void *) dslb, sizeof(float), size, grid_in);
    fread((void *) dslx, sizeof(float), size, grid_in);


    fclose(grid_in);

}


/************************************************/
void
Chemgrid_Score::read_phi_grid()
{
    // Read in rec. phimap generated by Delphi -- kxr 0305

    int             i,
                    j,
                    k,
                    l;
    string          fname;

    fname = file_prefix + ".phi";
    grid_in = fopen(fname.c_str(), "rb");

    if (grid_in == NULL) {
        cout << "\n\nCould not open phimap " << fname <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    cout << " Reading DelPhi grid from " << fname << endl;

    fread(&dsize, sizeof(int), 1, grid_in);
    fread(&dspacing, sizeof(float), 1, grid_in);
    fread(oldmid, sizeof(float), 3, grid_in);
    fread(dspan, sizeof(int), 3, grid_in);

    // phi = (float *)malloc(dsize * sizeof(float));
    phi = new float[dsize];
    fread((void *) phi, sizeof(float), dsize, grid_in);

    fclose(grid_in);


    l = 0;
    for (i = 0; i < dspan[0]; ++i) {
        for (j = 0; j < dspan[1]; ++j) {
            for (k = 0; k < dspan[2]; ++k) {
                phimap[i][j][k] = phi[l];
                l++;
            }
        }
    }


}


/*****************************************************/
void
Chemgrid_Score::read_solv_grid()
{
    // Solvent Occlusion Grid reading and calculation
    // capability incorporated in DOCK 5.2. Code adapted
    // from dock 3.5.54 by BKS
    // kxr 050205 


    string          fname,
                    tmp;
    char            line[100];
    int             i,
                    j,
                    k,
                    l,
                    kk,
                    start;
    int             cadif1,
                    cadif2,
                    cadif3;

    fname = solv_file_prefix;
    solv_grid_in = fopen(fname.c_str(), "rb");

    if (solv_grid_in == NULL) {
        cout << "\n\nCould not open " << fname << endl;
        exit(0);
    }

    cout << " Reading ligand desolvation grid from " << fname << endl;

    l = 0;
    while (fgets(line, 100, solv_grid_in) != NULL) {
        if (l == 0) {
            sscanf(line, "%i %i %i %i %i %i %i", &cadif1, &cadif2, &cadif3,
                   &perang, &solv_grd_ex[0], &solv_grd_ex[1], &solv_grd_ex[2]);
            l++;
            if (cadif1 > MAXDIX || cadif2 > MAXDIY || cadif3 > MAXDIZ) {
                cout <<
                    "\n\n Memory bounds exceeded for solvent occlusion grid.";
                cout << " Program will terminate." << endl;
                exit(0);
            }
        } else {
            for (i = 0; i <= cadif1; i++) {
                for (j = 0; j <= cadif2; j++) {
                    for (k = 0; k <= cadif3; k = k + 13) {
                        l++;
                        if (k + 12 <= cadif3) {
                            if (l > 2)
                                fgets(line, 100, solv_grid_in);
                            tmp = line;
                            start = 0;
                            for (kk = k; kk <= k + 12; kk++) {
                                solvgrd[i][j][kk] =
                                    atof(tmp.substr(start, 6).c_str());
                                start = start + 6;
                            }
                        } else {
                            if (l > 2)
                                fgets(line, 100, solv_grid_in);
                            tmp = line;
                            start = 0;
                            for (kk = k; kk <= cadif3; kk++) {
                                solvgrd[i][j][kk] =
                                    atof(tmp.substr(start, 6).c_str());
                                start = start + 6;
                            }
                        }
                    }
                }
            }
        }
    }

    fclose(solv_grid_in);
}

/************************************************/
// receptor desolvation grid calculated using DelPhi kxr 0306
void
Chemgrid_Score::read_rdsol_grid()
{

    string          fname;
    string          tmp;
    char            line[100];
    char            cgrid;
    int             cadif1,
                    cadif2,
                    cadif3;
    int             nlist,
                    sumlst;
    int             i,
                    j,
                    k,
                    l;
    float           ascore;


    fname = file_prefix + ".rdl";
    rdsol_grid_in = fopen(fname.c_str(), "rb");

    if (rdsol_grid_in == NULL) {
        cout << "\n\nCould not open " << fname <<
            " for reading.  Program will terminate." << endl << endl;
        exit(0);
    }

    cout << " Reading receptor desolvation grid from " << fname << endl;


    l = 0;
    sumlst = 0;
    while (fgets(line, 100, rdsol_grid_in) != NULL) {
        if (l == 0) {
            sscanf(line, "%i %i %i %i %i %i %i", &cadif1, &cadif2, &cadif3, &rdgrdiv, &rdxmax, &rdymax, &rdzmax);       // first 
                                                                                                                        // line 
                                                                                                                        // read 
                                                                                                                        // grid 
                                                                                                                        // params 
                                                                                                                        // kxr
            l++;
            // Check memory bounds
            if (cadif1 > MAXDIX || cadif2 > MAXDIY || cadif3 > MAXDIZ) {
                cout <<
                    "\n\n Memory bounds exceeded in desolvation grid. Program will Terminate"
                    << endl;
                exit(0);
            }
        } else {
            for (i = 0; i < cadif1 + 1; ++i) {
                for (j = 0; j < cadif2 + 1; ++j) {
                    while (sumlst < cadif3 + 1) {
                        l++;
                        if (l > 2)
                          fgets(line, 100, rdsol_grid_in);  // readmap
                        tmp = line;
                        nlist = atoi(tmp.substr(0, 4).c_str());
                        ascore = atof(tmp.substr(4, 10).c_str());
                        for (k = sumlst; k < sumlst + nlist; ++k) {
                            if (ascore == 128.0)
                                rdsol[i][j][k] = 0.0;
                            else
                                rdsol[i][j][k] = ascore;
                        }
                        sumlst = sumlst + nlist;
                    }
                    sumlst = 0;
                }
            }
        }


    }
    fclose(rdsol_grid_in);

}

/*********************************************************
	kxr 042205
	The following is an alternate implementation
	of the interpolation code for the DelPhi phimap.
	
	The basic steps for calculating the electrostatic 
	potential on the ligand atom are 
		1) converting real space coordinates of the ligand 
		   atom to grid coordinates. this is done in 
		   real_to_phi_coords function
		2) Then the potential is calculated by trilinear
		   interpolation. the converted grid coordinates
		   phi_coords are changed to integer values as
					nx = int(phi_space)
		3) These integer numbers serve as lookup values 
		   in the phimap as
		   a8 = phimap[nx][ny][nz]
		4) However before this step, 1 needs to be substracted
		   from nx, ny and nz because in the fortran generated
		   DelPhi grid the grid points indices start at 1,1,1 
		   where as in C++ it starts at 0,0,0. Just a matter of 
		   bookkeeping. 
********************************************************/

/*******************************************************/
void
Chemgrid_Score::real_to_phi_coords(float x, float y, float z)
{
    float           goff;

    goff = (float (dspan[0]) + 1.0) /2.0;
    phi_space[0] = (x - oldmid[0]) * dspacing + goff;
    phi_space[1] = (y - oldmid[1]) * dspacing + goff;
    phi_space[2] = (z - oldmid[2]) * dspacing + goff;

}

/***********************************************/
float
Chemgrid_Score::interpolpot()
{
    int             i;
    int             nx,
                    ny,
                    nz;
    float           rgrid;
    float           phi_value;
    float           xgr,
                    ygr,
                    zgr;
    float           a1,
                    a2,
                    a3,
                    a4,
                    a5,
                    a6,
                    a7,
                    a8;

    rgrid = float   (dspan[0] - 1);
    for (i = 0; i <= 2; i++)
        if (phi_space[i] < 1 || phi_space[i] > rgrid) {
            phi_value = 0.0;    // return 0.0 if outside grid_in
            return phi_value;
        }


    nx = int        (phi_space[0]);
    ny = int        (phi_space[1]);
    nz = int        (phi_space[2]);


    if (nx > dspan[0] || ny > dspan[1] || nz > dspan[2]) {
        cout << " Array bounds exceeded for phimap" << endl;
        exit(0);
    }


    xgr = phi_space[0] - float (nx);
    ygr = phi_space[1] - float (ny);
    zgr = phi_space[2] - float (nz);


    nx = nx - 1;
    ny = ny - 1;
    nz = nz - 1;

    a8 = phimap[nx][ny][nz];
    a7 = phimap[nx][ny][nz + 1] - a8;
    a6 = phimap[nx][ny + 1][nz] - a8;
    a5 = phimap[nx + 1][ny][nz] - a8;
    a4 = phimap[nx][ny + 1][nz + 1] - a8 - a7 - a6;
    a3 = phimap[nx + 1][ny][nz + 1] - a8 - a7 - a5;
    a2 = phimap[nx + 1][ny + 1][nz] - a8 - a6 - a5;
    a1 = phimap[nx + 1][ny + 1][nz + 1] - a8 - a7 - a6 - a5 - a4 - a3 - a2;

    // printf("%8.3f%8.3f%8.3f%8.3f%8.3f%8.3f%8.3f%8.3f\n",
    // a1,a2,a3,a4,a5,a6,a7,a8);

    phi_value = a1 * xgr * ygr * zgr + a2 * xgr * ygr + a3 * xgr * zgr
        + a4 * ygr * zgr + a5 * xgr + a6 * ygr + a7 * zgr + a8;

    return phi_value;
}

/************************************************/
float
Chemgrid_Score::get_atom_shape_score(float x, float y, float z)
{
    // Only for debugging but could be useful. kxr
    float           atomic_shape_score;
    int             tmp;
    int             lx,
                    ly,
                    lz;

    lx = xmax - NINT(x * grddiv);
    if ((lx >= 0) && (lx <= MAXDIX)) {
        ly = ymax - NINT(y * grddiv);
        if ((ly >= 0) && (ly <= MAXDIY)) {
            lz = zmax - NINT(z * grddiv);
            if ((lz >= 0) && (lz <= MAXDIZ)) {
                tmp = shape[lx][ly][lz];
                if (tmp < 0) {
                    tmp += 128;
                }
                atomic_shape_score = float (tmp);
            } else
                return -1;
        } else
            return -1;
    } else
        return -1;

    return atomic_shape_score;

}

/***********************************************************/
// Buried fraction calculation expt. code kxr
float
Chemgrid_Score::compute_buried_fraction(DOCKMol & mol)
{

    float           buried_fraction = 0.0f;
    unsigned int    N_buried = 0;
    unsigned int    N_heavy = 0;
    unsigned int    atom;
    float           vdw_att = 0.0f;

    for (atom = 0; atom < mol.num_atoms; atom++) {
        if (mol.amber_at_heavy_flag[atom] && mol.atom_active_flags[atom]) {
            if (check_box_boundaries(mol.x[atom], mol.y[atom], mol.z[atom])) {
                find_grid_neighbors(mol.x[atom], mol.y[atom], mol.z[atom]);
                vdw_att = -vdwB[mol.amber_at_id[atom]] * interpolate(bvdw);
                if (vdw_att < -1.0f)
                    N_buried++;
                N_heavy++;
            }
        }
    }

    buried_fraction = float (N_buried) / float (N_heavy);
    return buried_fraction;
}

/***********************************************************/
// Atom buriedness kxr
bool
Chemgrid_Score::atom_buried(DOCKMol & mol, int atom_id)
{

    unsigned int    N_buried = 0;
    float           vdw_att = 0.0f;

    if (check_box_boundaries(mol.x[atom_id], mol.y[atom_id], mol.z[atom_id])) {
        find_grid_neighbors(mol.x[atom_id], mol.y[atom_id], mol.z[atom_id]);
        vdw_att = -vdwB[mol.amber_at_id[atom_id]] * interpolate(bvdw);
        // cout << "vdw_att" << vdw_att << endl;
        if (vdw_att < -10.0f)
            return true;
    }
    return false;
}

/******************************************************/
// redistribute positive partial atomic desolvation kxr
// adapted from BKS redist. code from dock3.5.54 
bool
Chemgrid_Score::redist_positve_desolv(DOCKMol & mol)
{
    float           totpos,
                    totneg,
                    totsol;
    float           polsolv;
    unsigned int    atom;

    totpos = totneg = totsol = 0.0;
    for (atom = 0; atom < mol.num_atoms; atom++) {

        if (mol.atom_active_flags[atom]) {
            polsolv = mol.atom_psol[atom];
            if (polsolv > 0.0f) {
                totpos += polsolv;
                mol.atom_psol[atom] = 0.0f;
            } else
                totneg += polsolv;
        }
    }

    for (atom = 0; atom < mol.num_atoms; atom++) {
        if (mol.atom_active_flags[atom]) {
            polsolv = mol.atom_psol[atom];
            if (mol.atom_psol[atom] < 0.0f)
                mol.atom_psol[atom] += (totpos * polsolv) / totneg;
            totsol += mol.atom_psol[atom];
        }
    }

    return true;

}

/***************************************************/
/*
 * Calculate Internal Energy of the ligand kxr 010506 float
 * Chemgrid_Score::compute_ligand_internal_energy(DOCKMol &mol){ float int_vdw, 
 * int_es, distance, ligand_internal_energy; unsigned int a1,a2;
 * 
 * float ie_diel = 2.0; int ie_rep_exp = 12; int ie_att_exp = 6; int_vdw = 0.0;
 * int_es = 0.0; for(a1=0;a1<mol.num_atoms-1;a1++) {
 * for(a2=a1+1;a2<mol.num_atoms;a2++) { if(check_box_boundaries(mol.x[a1],
 * mol.y[a1], mol.z[a1]) && check_box_boundaries(mol.x[a2], mol.y[a2],
 * mol.z[a2])){ if((mol.get_bond(a1,a2) == -1) &&
 * (!mol.atoms_are_one_three(a1,a2)) && (!mol.atoms_are_one_four(a1,a2))) {
 * 
 * distance = pow((mol.x[a1]-mol.x[a2]),2) + pow((mol.y[a1]-mol.y[a2]),2) +
 * pow((mol.z[a1]-mol.z[a2]),2); distance = sqrt(distance); int_vdw +=
 * (vdwA[a1]*vdwA[a2]) / pow(distance, (ie_rep_exp)) - (vdwB[a1]*vdwB[a2]) /
 * pow(distance, (ie_att_exp)); int_es += (mol.charges[a1] * mol.charges[a2] *
 * ie_diel) / distance; } } } } ligand_internal_energy = int_vdw + int_es;
 * return ligand_internal_energy; } 
 */

/************************************************/
// Calculate conformational entropy from # of lig. rot bonds kxr
float
Chemgrid_Score::get_conf_entropy(DOCKMol & mol)
{
    float           ligand_conf_entropy;
    int             rot_bond_count = 0;
    int             orig_rot_bond = 0;
    int             a1,
                    a2,
                    i;

    for (i = 0; i < mol.num_bonds; i++) {
        if (mol.bond_is_rotor(i)) {
            orig_rot_bond++;
            a1 = mol.bonds_origin_atom[i];
            if (atom_buried(mol, a1)) {
                rot_bond_count++;
            } else {
                a2 = mol.bonds_target_atom[i];
                if (atom_buried(mol, a2))
                    rot_bond_count++;
            }
        }

    }
    ligand_conf_entropy = 1.2f * float (rot_bond_count);
    return ligand_conf_entropy;
}

/************************************************/
// Calculate volume based ligand desolvation score kxr 0306
bool
Chemgrid_Score::compute_ligand_desolvation(DOCKMol & mol, int atom_id)
{
    int             lx,
                    ly,
                    lz;
    float           polsolv_coeff,
                    apolsolv_coeff,
                    temp;
    int             out_count = 0;

    lx = solv_grd_ex[0] - NINT(mol.x[atom_id] * perang);
    ly = solv_grd_ex[1] - NINT(mol.y[atom_id] * perang);
    lz = solv_grd_ex[2] - NINT(mol.z[atom_id] * perang);

    if ((lx >= 0 && ly >= 0 && lz >= 0) &&
        (lx <= MAXDIX && ly <= MAXDIY && lz <= MAXDIZ))
        temp = solvgrd[lx][ly][lz];
    else {
        out_count++;
        temp = 0.0;
    }

    if (mol.atom_data[atom_id].size() != 0) {

        polsolv_coeff = atof((mol.atom_data[atom_id].substr(23, 8)).c_str());
        apolsolv_coeff = atof((mol.atom_data[atom_id].substr(33, 8)).c_str());
        psolv_score += polsolv_coeff * temp;
        apsolv_score += apolsolv_coeff * temp;
        solv_score += polsolv_coeff * temp + apolsolv_coeff * temp;

    } else {

        if (total_ligand_dsolv) {
            psolv_score += mol.atom_psol[atom_id];
            apsolv_score += mol.atom_apsol[atom_id];
            solv_score += mol.atom_psol[atom_id] + mol.atom_apsol[atom_id];
        } else {
            psolv_score += mol.atom_psol[atom_id] * temp;
            apsolv_score += mol.atom_apsol[atom_id] * temp;
            solv_score +=
                mol.atom_psol[atom_id] * temp + mol.atom_apsol[atom_id] * temp;
        }

    }

    return true;

}

/*************************************************/
// Compute occupancy desolvation score for atom 1205 kxr
bool
Chemgrid_Score::compute_odm_score(DOCKMol & mol, int atom_id)
{
    float           sole_atom,
                    vol_atom;

    if (check_box_boundaries(mol.x[atom_id], mol.y[atom_id], mol.z[atom_id])) {
        find_grid_neighbors(mol.x[atom_id], mol.y[atom_id], mol.z[atom_id]);
        sole_atom = 0.25 * SQR(mol.charges[atom_id]) - 0.005;
        solx_val += sole_atom * interpolate(dslx);
        vol_atom = 4.887 * CUB(mol.amber_at_radius[atom_id]);
        solb_val += vol_atom * interpolate(dslb);
    }
    return true;
}

/*************************************************/
// Compute receptor desolvation score from DelPhi generated grid 0306
bool
Chemgrid_Score::compute_rdsol_score(DOCKMol & mol, int atom_id)
{

    int             lx,
                    ly,
                    lz;
    float           temp;
    int             out_count = 0;

    lx = rdxmax - NINT(mol.x[atom_id] * rdgrdiv);
    ly = rdymax - NINT(mol.y[atom_id] * rdgrdiv);
    lz = rdzmax - NINT(mol.z[atom_id] * rdgrdiv);

    if ((lx >= 0 && ly >= 0 && lz >= 0) &&
        (lx <= MAXDIX && ly <= MAXDIY && lz <= MAXDIZ))
        temp = rdsol[lx][ly][lz];
    else {
        out_count++;
        temp = 0.0;
    }

    rdsol_score += temp;
    return true;


}



/************************************************/
bool
Chemgrid_Score::compute_score(DOCKMol & mol)
{
    float           vdw_val,
                    es_val,
                    total;
    // float solx_val, solb_val;
    float           phi_value,
                    temp;
    float           vdw_atom,
                    grd_edge;
    float           es_atom,
                    atsh;
    float           polsolv_coeff,
                    apolsolv_coeff;
    float           ligand_conf_entropy;
    char            eline[100];
    unsigned int    atom,
                    atm_num;
    unsigned int    i;
    unsigned int    lx,
                    ly,
                    lz;
    string          estring,
                    atom_name,
                    tezt;
    char            line[100];


    atm_num = 0;
    grd_edge = -10000.0;
    total = vdw_val = es_val = 0.0;
    psolv_score = apsolv_score = 0.0;
    solb_val = solx_val = rdsol_score = 0.0;
    solv_score = shape_score = ligand_conf_entropy = 0.0;


    if (use_solv_score && redist_pos_desol)
        redist_positve_desolv(mol);

    if (write_atomic_energy) {
        estring.append(" Atomic Contributions:\n");
        estring.
            append
            ("                   vdW     es      pot    charge    pol    apol\n");
    }

    if (energy_score == 1) {

        for (atom = 0; atom < mol.num_atoms; atom++) {

            vdw_atom = 0.0;
            es_atom = 0.0;
            if (mol.atom_active_flags[atom]) {

                if (check_box_boundaries(mol.x[atom], mol.y[atom], mol.z[atom])) {
                    find_grid_neighbors(mol.x[atom], mol.y[atom], mol.z[atom]);

                    vdw_atom =
                        ((vdwA[mol.amber_at_id[atom]] * interpolate(avdw)) -
                         (vdwB[mol.amber_at_id[atom]] * interpolate(bvdw))) *
                        vdw_scale;
                    if (vdw_atom < grd_edge)
                        vdw_atom = -vdw_atom;
                    vdw_val += vdw_atom;

                } else {
                    mol.current_score = -MIN_FLOAT;  // arbitrarily large score
                    sprintf(line, "ERROR:  Conformation could not be scored by "
                        "DOCK.\nConformation not completely within grid box.\n");
                    mol.current_data = line;
                    return false;
                }

                if (interpol_method == 2) {

                    real_to_phi_coords(mol.x[atom], mol.y[atom], mol.z[atom]);
                    phi_value = interpolpot();
                    es_atom = mol.charges[atom] * phi_value * 0.5924 * es_scale;
                    es_val += es_atom;

                    // Calculate volume based ligand desolvation for atom kxr 
                    // if(use_solv_score && use_secondary_score) 
                    if (use_solv_score)
                        compute_ligand_desolvation(mol, atom);


                    // Calculate DelPhi receptor desolvation score for atom kxr
                    if (use_solv_score && use_recep_dsolv)
                        compute_rdsol_score(mol, atom);

                    // Calculate occupancy desolvation score for atom kxr 
                    if (use_solv_score && use_odm_score)
                        compute_odm_score(mol, atom);


                    if (write_atomic_energy && use_solv_score) {
                        atm_num++;
                        sprintf(eline,
                                "ATOM %-4d%-6s%8.3f%8.3f%8.3f%8.3f%8.3f%8.3f%8.1f\n",
                                atm_num, mol.atom_names[atom].c_str(), vdw_atom,
                                es_atom, phi_value, mol.charges[atom],
                                polsolv_coeff * temp, apolsolv_coeff * temp,
                                atsh);
                        // sprintf(eline,"ATOM %6d %-4sXXX 1
                        // %8.3f%8.3f%8.3f%8.3f\n",
                        // atm_num,mol.atom_names[atom].c_str(),mol.x[atom],mol.y[atom],mol.z[atom],
                        // mol.charges[atom]);
                        estring.append(eline);
                    } else {
                        atm_num++;
                        sprintf(eline, "ATOM %-4d%-6s%8.3f%8.3f%8.3f%8.3f\n",
                                atm_num, mol.atom_names[atom].c_str(), vdw_atom,
                                es_atom, phi_value, mol.charges[atom]);
                        estring.append(eline);
                    }
                }

                if (interpol_method == 1) {

                    // Use delphi score for electrostatics get different grid
                    // values 
                    if (phi_box_boundaries
                        (mol.x[atom], mol.y[atom], mol.z[atom])) {
                        find_phi_neighbors(mol.x[atom], mol.y[atom],
                                           mol.z[atom]);

                        phi_value = interpolphi(phi);
                        es_atom = mol.charges[atom] * phi_value
                            * 0.5924f * es_scale;
                        es_val += es_atom;

                        if (write_atomic_energy) {
                            atm_num++;
                            sprintf(eline,
                                    "ATOM %-4d%-6s%8.3f%8.3f%8.3f%8.3f\n",
                                    atm_num, mol.atom_names[atom].c_str(),
                                    vdw_atom, es_atom, phi_value,
                                    mol.charges[atom]);
                            // sprintf(eline,"ATOM %6d %-4sXXX 1
                            // %8.3f%8.3f%8.3f%8.3f\n",
                            // atm_num,mol.atom_names[atom].c_str(),mol.x[atom],mol.y[atom],mol.z[atom],
                            // mol.charges[atom]);
                            estring.append(eline);
                        }

                    } else {
                        mol.current_score = -MIN_FLOAT;  // arbitrarily large
                        sprintf(line, "ERROR:  Conformation could not be "
                            "scored by DOCK.\nConformation not completely "
                            "within grid box.\n");
                        mol.current_data = line;
                        return false;
                    }
                }
            }

        }



        solv_score = -solv_score;
        if (!use_solv_score)
            total = vdw_val + es_val;

        if (use_solv_score) {
            if (use_recep_dsolv)
                total = vdw_val + es_val + solv_score + rdsol_score;
            else
                total = vdw_val + es_val + solv_score;

            if (use_odm_score)
                total = vdw_val + es_val + solx_val + solb_val;
        }


        if (use_conf_entropy) {
            ligand_conf_entropy = get_conf_entropy(mol);
            total = total + ligand_conf_entropy;
        }

        if (add_ligand_internal)
            total = total + compute_ligand_internal_energy(mol);

        vdw_component = vdw_val;
        es_component = es_val;
        socc_component = solv_score;
        rdsol_component = rdsol_score;
        polsolv_component = -psolv_score;
        apolsolv_compontent = -apsolv_score;
        bulksolv_component = solb_val;
        explsolv_component = solx_val;
        conf_entropy_component = ligand_conf_entropy;

        mol.current_score = total;
        mol.current_data = output_score_summary(total);

        // add_score_to_mol(mol, "Energy Score", total);

    }

    if (write_atomic_energy)
        atomic_contrib = estring;
    return true;
}


/************************************************/
string
Chemgrid_Score::output_score_summary(float score)
{
    string          text;
    char            line[50000];

    if (use_score) {

        if (use_conf_entropy)
            sprintf(line,
                    "########## Energy Score:%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########\n",
                    score, "vdw:", vdw_component, "es:", es_component,
                    "conf_entropy:", conf_entropy_component);
        else
            sprintf(line,
                    "########## Energy Score:%20f\n##########%14s%20f\n##########%14s%20f\n##########\n",
                    score, "vdw:", vdw_component, "es:", es_component);

        if (use_solv_score && !use_recep_dsolv && !use_odm_score
            && !use_conf_entropy)
            sprintf(line,
                    "########## Energy Score:%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########\n",
                    score, "vdw:", vdw_component, "es:", es_component,
                    "polsol:", polsolv_component, "apolsol:",
                    apolsolv_compontent);
        else if (use_solv_score && use_conf_entropy && !use_odm_score
                 && !use_recep_dsolv)
            sprintf(line,
                    "########## Energy Score:%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########\n",
                    score, "vdw:", vdw_component, "es:", es_component,
                    "polsol:", polsolv_component, "apolsol:",
                    apolsolv_compontent, "conf_entropy:",
                    conf_entropy_component);
        else if (use_solv_score && use_odm_score && !use_conf_entropy)
            sprintf(line,
                    "########## Energy Score:%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########\n",
                    score, "vdw:", vdw_component, "es:", es_component,
                    "recep. desol:", bulksolv_component, "ligand desol:",
                    explsolv_component);
        else if (use_solv_score && use_odm_score && use_conf_entropy)
            sprintf(line,
                    "########## Energy Score:%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########\n",
                    score, "vdw:", vdw_component, "es:", es_component,
                    "recep. desol:", bulksolv_component, "ligand desol:",
                    explsolv_component, "conf_entropy:",
                    conf_entropy_component);
        else if (use_solv_score && use_recep_dsolv && !use_conf_entropy)
            sprintf(line,
                    "########## Energy Score:%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########\n",
                    score, "vdw:", vdw_component, "es:", es_component,
                    "recep. desol:", rdsol_component, "ligand desol:",
                    polsolv_component + apolsolv_compontent);
        else if (use_solv_score && use_recep_dsolv && use_conf_entropy)
            sprintf(line,
                    "########## Energy Score:%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########%14s%20f\n##########\n",
                    score, "vdw:", vdw_component, "es:", es_component,
                    "recep. desol:", rdsol_component, "ligand desol:",
                    polsolv_component + apolsolv_compontent, "conf_entropy:",
                    conf_entropy_component);


        text = line;
        if (write_atomic_energy)
            text.append(atomic_contrib);
        return text;
    } else
        return "";
}

/***************************************/
