#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include "gridio.h"
#include "lattice.h"
#include "timer.h"


void usage(const char *thisprog)
{
  fprintf(stderr, "\nsyntax:   %s [-c CUTOFF] [-d DELTA] [-n] [-p] [-v] "
      "QFILE EFILE\n", thisprog);
  fprintf(stderr, "\t-c CUTOFF sets cutoff distance (default 12 Angstroms)\n"
                  "\t-d DELTA sets grid spacing (default 3 Angstroms)\n"
		  "\t-n sets nonperiodic boundary conditions (default)\n"
		  "\t-p sets periodic boundary conditions\n"
		  "\t-v verify by also computing all pairs and comparing\n"
		  "\tQFILE is name of charge data file to read\n"
		  "\tEFILE is name of potential data file to write\n");
  fprintf(stderr, "\nTypical parameter ranges are:\n"
		  "\t8 <= CUTOFF <= 12  (Angstroms)\n"
                  "\t2 <= DELTA <= 3  (Angstroms)\n");
  exit(1);
}


int main(int argc, char *argv[])
{
  const char *thisprog = argv[0];
  const char *qfname;  /* charge grid file for reading */
  const char *efname;  /* potential grid file for writing */
  float a=0;  /* cutoff distance */
  float h=0;  /* grid spacing */
  float *e;   /* potential grid */
  float *q;   /* charge grid */
  float *wt;  /* weights grid */
  int nx, ny, nz;  /* dimensions of potential and charge grids */
  int nr;     /* radius of weights grid */
  int ch;
  char c;
  char verify = 0;
  char nonperiodic = 0;
  char periodic = 0;
  Timer timer;

  printf("Compute grid of potentials from grid of charges\n");

  /* parse command line options */
  while ((ch = getopt(argc, argv, "c:d:vnp")) != -1) {
    switch (ch) {
      case 'c':
	if (sscanf(optarg, "%f%c", &a, &c) != 1 || a <= 0) {
	  fprintf(stderr, "expecting CUTOFF to be positive real\n");
	  usage(thisprog);
	}
	break;
      case 'd':
	if (sscanf(optarg, "%f%c", &h, &c) != 1 || h <= 0) {
	  fprintf(stderr, "expecting DELTA to be positive real\n");
	  usage(thisprog);
	}
	break;
      case 'v':
	verify = 1;
	break;
      case 'n':
	nonperiodic = 1;
	break;
      case 'p':
	periodic = 1;
	break;
      default:
	usage(thisprog);
    }
  }
  argc -= optind;
  argv += optind;

  if (argc != 2) {
    usage(thisprog);
  }
  qfname = argv[0];
  efname = argv[1];

  /* take care of defaults */
  if (0==a) {
    printf("Setting CUTOFF distance to DEFAULT 12 Angstroms\n");
    a = 12;
  }
  else {
    printf("Setting CUTOFF distance to %g Angstroms\n", a);
  }
  if (0==h) {
    printf("Setting DELTA grid spacing to DEFAULT 3 Angstroms\n");
    h = 3;
  }
  else {
    printf("Setting DELTA grid spacing to %g Angstroms\n", h);
  }
  if (periodic && nonperiodic) {
    fprintf(stderr, "cannot choose both nonperiodic and periodic boundaries\n");
    usage(thisprog);
  }
  else if ( !(periodic || nonperiodic) ) {
    printf("Setting boundary conditions to DEFAULT nonperiodic\n");
    nonperiodic = 1;  /* default to nonperiodic */
  }
  else {
    printf("Setting boundary conditions to %s\n",
	(periodic ? "periodic" : "nonperiodic"));
  }

  /* read charge grid */
  q = gridio_read(qfname, &nx, &ny, &nz);
  if (NULL==q) {
    fprintf(stderr, "gridio_read() failed to read charge grid\n");
    exit(1);
  }
  printf("Grid of %d x %d x %d charges read from \"%s\"\n",
      nx, ny, nz, qfname);

  /* compute weights */
  wt = alloc_wt(&nr, a, h);
  if (NULL==wt) {
    fprintf(stderr, "alloc_wt() failed to create grid of weights\n");
    exit(1);
  }
  printf("Grid of weights has radius %d grid points\n", nr);

  /* allocate potential grid */
  e = gridio_alloc(nx, ny, nz);
  if (NULL==e) {
    fprintf(stderr, "gridio_alloc() failed to create grid of potential\n");
    exit(1);
  }

  /* compute potential grid */
  if (verify) {
    /* direct method */
    if (nonperiodic) {
      printf("Computing lattice direct potential "
          "with nonperiodic boundaries\n");
      timer_start(&timer);
      nonperiodic_lattice_direct(e, q, nx, ny, nz, a, h);
      printf("Completed in %.3f seconds\n", timer_delta(&timer));
    }
    else {
      printf("Computing lattice direct potential with periodic boundaries\n");
      timer_start(&timer);
      periodic_lattice_direct(e, q, nx, ny, nz, a, h);
      printf("Completed in %.3f seconds\n", timer_delta(&timer));
    }
  }
  else {
    /* cutoff method */
    if (nonperiodic) {
      printf("Computing lattice cutoff potential "
          "with nonperiodic boundaries\n");
      timer_start(&timer);
      nonperiodic_lattice_cutoff(e, q, nx, ny, nz, wt, nr);
      printf("Completed in %.3f seconds\n", timer_delta(&timer));
    }
    else {
      printf("Computing lattice cutoff potential with periodic boundaries\n");
      timer_start(&timer);
      periodic_lattice_cutoff(e, q, nx, ny, nz, wt, nr);
      printf("Completed in %.3f seconds\n", timer_delta(&timer));
    }
  }

  /* write potential grid */
  if (GRIDIO_FAIL==gridio_write(efname, e, nx, ny, nz)) {
    fprintf(stderr, "gridio_write() failed to write potential grid\n");
    exit(1);
  }
  printf("Grid of %d x %d x %d potentials written to \"%s\"\n",
      nx, ny, nz, efname);

  /* cleanup */
  gridio_free(q);
  gridio_free(e);
  free_wt(wt);

  return 0;
}
