#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <math.h>
#include "gridio.h"


void usage(const char *thisprog)
{
  fprintf(stderr, "\nsyntax:   %s FILE1 FILE2\n", thisprog);
  fprintf(stderr, "\t-l LOWER_VAL gives value below which relative error "
      "is ignored\n\t-u UPPER_ERR gives upper relative error above which is "
      "reported\n");
  exit(1);
}


int main(int argc, char *argv[])
{
  const char *thisprog = argv[0];
  const char *f_estim, *f_exact;
  float *g_estim, *g_exact;
  int nx, ny, nz, nnx, nny, nnz;
  int i, j, k;
  int i_max_abserr=-1, j_max_abserr=-1, k_max_abserr=-1;
  int i_max_relerr=-1, j_max_relerr=-1, k_max_relerr=-1;
  int index;
  float lower_val = 1;
  float upper_err = 1;
  float max_abserr = 0;
  float max_relerr = 0;
  float abserr_sum = 0;
  float abserr, relerr;
  float abs_g_exact, avg;
  int ch;
  char c;

  printf("Determine error of one grid relative to a second grid\n");
  while ((ch = getopt(argc, argv, "l:u:")) != -1) {
    switch (ch) {
      case 'l':
        if (sscanf(optarg, "%f%c", &lower_val, &c) != 1 || lower_val <= 0) {
          fprintf(stderr, "expecting LOWER_VAL to be positive real "
              "(default is 1)\n");
          usage(thisprog);
        }
        break;
      case 'u':
        if (sscanf(optarg, "%f%c", &upper_err, &c) != 1 || upper_err <= 0) {
          fprintf(stderr, "expecting UPPER_ERR to be positive real "
              "(default is 1)\n");
          usage(thisprog);
        }
        break;
      default:
        usage(thisprog);
    }
  }
  argc -= optind;
  argv += optind;

  if (argc != 2) {
    usage(thisprog);
  }
  f_estim = argv[0];
  f_exact = argv[1];

  g_estim = gridio_read(f_estim, &nx, &ny, &nz);
  if (NULL==g_estim) {
    fprintf(stderr, "gridio_read() failed\n");
    exit(1);
  }
  g_exact = gridio_read(f_exact, &nnx, &nny, &nnz);
  if (NULL==g_exact) {
    fprintf(stderr, "gridio_read() failed\n");
    exit(1);
  }
  else if (nx != nnx || ny != nny || nz != nnz) {
    fprintf(stderr, "files have grids with different dimensions\n");
    exit(1);
  }

  printf("Grids of %d x %d x %d values (%d total)\n", nx, ny, nz, (nx*ny*nz));

  for (k = 0;  k < nz;  k++) {
    for (j = 0;  j < ny;  j++) {
      for (i = 0;  i < nx;  i++) {

        index = k*ny*nx + j*nx + i;

        abserr = fabsf(g_estim[index] - g_exact[index]);
        if (abserr > max_abserr) {
          max_abserr = abserr;
          i_max_abserr = i;
          j_max_abserr = j;
          k_max_abserr = k;
        } /* end absolute error */

        abserr_sum += abserr;

        abs_g_exact = fabsf(g_exact[index]);
        if (abs_g_exact > lower_val) {
          relerr = abserr / abs_g_exact;
          if (relerr > max_relerr) {
            max_relerr = relerr;
            i_max_relerr = i;
            j_max_relerr = j;
            k_max_relerr = k;
          }
          if (relerr >= upper_err) {
            printf("(%d,%d,%d):  approx=%f  exact=%f  relerr=%f\n",
                i, j, k, g_estim[index], g_exact[index], relerr);
          }
        } /* end relative error */

      }
    }
  } /* end loop over grid points */

  avg = abserr_sum / (nx*ny*nz);
  printf("The mean difference between the entries is %g\n", avg);
  if (i_max_abserr >= 0) {
    printf("The max absolute error is %g at grid point (%d,%d,%d)\n",
        max_abserr, i_max_abserr, j_max_abserr, k_max_abserr);
  }
  else {
    printf("The files are identical\n");
  }
  if (i_max_relerr >= 0) {
    printf("The max relative error is %g at grid point (%d,%d,%d)\n",
        max_relerr, i_max_relerr, j_max_relerr, k_max_relerr);
  }

  gridio_free(g_estim);
  gridio_free(g_exact);
  return 0;
}

