#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include "gridio.h"


#define DEFAULT_SEED  1


void generate_random_q(float *q, int nx, int ny, int nz, int seed)
{
  long int total = nx*ny*nz;
  long int i;
  double qtotal = 0, qavg;

  srandom(seed);
  for (i = 0;  i < total;  i++) {
    q[i] = 2 * (random() / (float) RAND_MAX) - 1;
    qtotal += q[i];
  }
  qavg = qtotal / total;
  /* create charge neutral system */
  for (i = 0;  i < total;  i++) {
    q[i] -= qavg;
  }
}


void usage(const char *thisprog)
{
  fprintf(stderr, "\nsyntax:   %s [-s SEED] [-v] NX NY NZ QFILE\n", thisprog);
  fprintf(stderr, "\t-s SEED sets seed for random number generator\n"
                  "\t-v verify file by reading and summing charges\n"
                  "\tNX, NY, NZ are dimensions of 3D lattice\n"
                  "\tQFILE is name of charge data file produced\n");
  exit(1);
}


int main(int argc, char *argv[])
{
  const char *thisprog = argv[0];
  const char *fname;
  float *q;
  int ch, nx, ny, nz;
  int seed = DEFAULT_SEED;
  char c;
  char verify = 0;

  printf("Generate a grid of random charge\n");
  while ((ch = getopt(argc, argv, "vs:")) != -1) {
    switch (ch) {
      case 'v':
        verify = 1;
        break;
      case 's':
        if (sscanf(optarg, "%d%c", &seed, &c) != 1) {
          fprintf(stderr, "expecting SEED to be an integer\n");
          usage(thisprog);
        }
        break;
      default:
        usage(thisprog);
    }
  }
  argc -= optind;
  argv += optind;

  if (argc != 4) {
    usage(thisprog);
  }
  if (sscanf(argv[0], "%d%c", &nx, &c) != 1 || nx <= 0 ||
      sscanf(argv[1], "%d%c", &ny, &c) != 1 || ny <= 0 ||
      sscanf(argv[2], "%d%c", &nz, &c) != 1 || nz <= 0) {
    fprintf(stderr, "expecting positive integers NX, NY, NZ\n");
    usage(thisprog);
  }
  fname = argv[3];

  q = gridio_alloc(nx, ny, nz);
  if (NULL==q) {
    fprintf(stderr, "gridio_alloc() failed to allocate %d x %d x %d "
        "grid of charges\n", nx, ny, nz);
    exit(1);
  }
  generate_random_q(q, nx, ny, nz, seed);
  if (GRIDIO_FAIL==gridio_write(fname, q, nx, ny, nz)) {
    fprintf(stderr, "gridio_write() failed to write charge grid "
        "to file \"%s\"\n", fname);
    exit(1);
  }
  gridio_free(q);
  printf("Grid of %d x %d x %d charges written to \"%s\"\n",
      nx, ny, nz, fname);

  if (verify) {
    int nnx, nny, nnz;
    long int i, total;
    double qsum = 0;

    printf("Verify by reading charge grid from \"%s\"\n", fname);
    q = gridio_read(fname, &nnx, &nny, &nnz);
    if (NULL==q) {
      fprintf(stderr, "gridio_read() failed\n");
      exit(1);
    }
    if (nnx != nx || nny != ny || nnz != nz) {
      fprintf(stderr, "nnx=%d  nny=%d  nnz=%d\n"
          "nx=%d   ny=%d   nz=%d\n", nnx, nny, nnz, nx, ny, nz);
      fprintf(stderr, "grid dimension verification failed\n");
      exit(1);
    }
    total = nnx*nny*nnz;
    for (i = 0;  i < total;  i++) {
      qsum += q[i];
    }
    gridio_free(q);
    printf("Sum over grid of %d x %d x %d charges is %.8e\n",
        nnx, nny, nnz, qsum);
  }

  return 0;
}
