/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * gridio.c - read and write binary data file of rectangular grid of floats
 *
 * File stored as:
 *   4 * int32:  1, nx, ny, nz
 *   (nx*ny*nz) * float:  data
 *
 * Leading value 1 is used for endianess check; int32 is 4-byte int.
 * Grid is treated as flat array, meaning that no assumptions are made
 * on the order of the 3D indexing.
 */

#include <limits.h>
#include <stdlib.h>
#include <stdio.h>
#include "gridio.h"


#if   INT_MAX == 2147483647
typedef int int32;
typedef unsigned int uint32;
enum {
  MAX_INT32 = INT_MAX,
  MIN_INT32 = INT_MIN
};
#elif INT_MAX == 32767 && LONG_MAX == 2147483647
typedef long int32;
typedef unsigned long uint32;
enum {
  MAX_INT32 = LONG_MAX,
  MAX_INT32 = LONG_MIN
};
#elif INT_MAX == 9223372036854775807L && SHRT_MAX == 2147483647
typedef short int32;
typedef unsigned short uint32;
enum {
  MAX_INT32 = SHRT_MAX,
  MAX_INT32 = SHRT_MIN
};
#endif


/*
 * performs in-place byte reordering to correct for different endianism,
 * where x is a 4-byte quantity
 */
#define REORDER(x) \
do { \
  unsigned char _tmp; \
  unsigned char *_p = (unsigned char *) &(x); \
  _tmp = _p[0];  _p[0] = _p[3];  _p[3] = _tmp; \
  _tmp = _p[1];  _p[1] = _p[2];  _p[2] = _tmp; \
} while (0)


int gridio_write(const char *name, float *g, int nx, int ny, int nz)
{
  int32 n[4];
  long int total;
  FILE *f;

  if (NULL==name || NULL==g || nx <= 0 || ny <= 0 || nz <= 0) {
    fprintf(stderr, "(%s, line %d): illegal parameters\n", __FILE__, __LINE__);
    abort();
  }
  n[0] = 1;
  n[1] = nx;
  n[2] = ny;
  n[3] = nz;
  total = nx*ny*nz;
  if (NULL==(f = fopen(name, "wb"))) {
    fprintf(stderr, "unable to open file \"%s\" for binary writing\n", name);
    return GRIDIO_FAIL;
  }
  if (fwrite(n, sizeof(int32), 4, f) != 4) {
    fprintf(stderr, "unable to write %ld bytes of grid dimensions "
        "to file \"%s\"\n", 4*sizeof(int32), name);
    fclose(f);
    return GRIDIO_FAIL;
  }
  if (fwrite(g, sizeof(float), total, f) != total) {
    fprintf(stderr, "unable to write %ld bytes of grid data "
        "to file \"%s\"\n", total*sizeof(float), name);
    fclose(f);
    return GRIDIO_FAIL;
  }
  if (fclose(f)) {
    fprintf(stderr, "unable to close file \"%s\" after writing\n", name);
    return GRIDIO_FAIL;
  }
  return 0;
}


float *gridio_read(const char *name, int *nx, int *ny, int *nz)
{
  int32 n[4];
  long int total;
  int reorderbytes = 0;
  float *g;
  FILE *f;
  unsigned char c;

  if (NULL==name || NULL==nx || NULL==ny || NULL==nz) {
    fprintf(stderr, "(%s, line %d): illegal parameters\n", __FILE__, __LINE__);
    abort();
  }
  if (NULL==(f = fopen(name, "rb"))) {
    fprintf(stderr, "unable to open file \"%s\" for binary reading\n", name);
    return NULL;
  }
  if (fread(n, sizeof(int32), 4, f) != 4) {
    fprintf(stderr, "unable to read %ld bytes of grid dimensions "
        "from file \"%s\"\n", 4*sizeof(int32), name);
    fclose(f);
    return NULL;
  }

  /* check to see if byte reordering is necessary */
  if (n[0] != 1) {
    REORDER(n[0]);
    if (n[0] != 1) {
      fprintf(stderr, "cannot comprehend contents of file \"%s\"\n", name);
      fclose(f);
      return NULL;
    }
    reorderbytes = 1;
    REORDER(n[1]);
    REORDER(n[2]);
    REORDER(n[3]);
  }

  /* determine validity of grid dimensions */
  if (n[1] <= 0 || n[2] <= 0 || n[3] <= 0) {
    fprintf(stderr, "inappropriate grid dimensions in file \"%s\"\n", name);
    fclose(f);
    return NULL;
  }

  /* allocate memory for grid */
  if (NULL==(g = gridio_alloc(n[1], n[2], n[3]))) {
    fclose(f);
    return NULL;
  }
  total = n[1]*n[2]*n[3];
  if (fread(g, sizeof(float), total, f) != total) {
    fprintf(stderr, "unable to read %ld bytes of grid data "
        "from file \"%s\"\n", total*sizeof(float), name);
    fclose(f);
    gridio_free(g);
    return NULL;
  }
  if (fread(&c, sizeof(unsigned char), 1, f) == 1) {
    fprintf(stderr, "extra data found in file \"%s\"\n", name);
    fclose(f);
    gridio_free(g);
    return NULL;
  }
  else if (!feof(f)) {
    fprintf(stderr, "unable to find end of file \"%s\"\n", name);
    fclose(f);
    gridio_free(g);
    return NULL;
  }
  if (fclose(f)) {
    fprintf(stderr, "unable to close file \"%s\" after reading\n", name);
    gridio_free(g);
    return NULL;
  }

  /* reorder bytes if necessary */
  if (reorderbytes) {
    long int i;
    for (i = 0;  i < total;  i++) {
      REORDER(g[i]);
    }
  }

  *nx = n[1];
  *ny = n[2];
  *nz = n[3];
  return g;
}


float *gridio_alloc(int nx, int ny, int nz)
{
  long int total;
  float *g;

  if (nx <= 0 || ny <= 0 || nz <= 0) {
    fprintf(stderr, "(%s, line %d): illegal parameters\n", __FILE__, __LINE__);
    abort();
  }
  total = nx*ny*nz;
  if (NULL==(g = (float *) calloc(total, sizeof(float)))) {
    fprintf(stderr, "unable to calloc() space for %ld floats\n", total);
    return NULL;
  }
  return g;
}


void gridio_free(float *g)
{
  free(g);
}
