/*
 * Copyright (C) 2007 by David J. Hardy.  All rights reserved.
 *
 * lattice_direct.c - lattice direct computation to check correctness
 *
 * All lattices are stored in column-major order, i.e. loop with:
 *   for (k = 0;  k < nz;  k++) {
 *     for (j = 0;  j < ny;  j++) {
 *       for (i = 0;  i < nx;  i++) {
 *         e[k*nj*ni + j*ni + i] = ...
 *       }
 *     }
 *   }
 * to access the array elements in consecutive order.
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>


/*
 * C2 Taylor softening of 1/r
 */
static float g(float a, float r) {
  if (r < a) {
    float r_a = r/a;
    return (1/a)*(1.875f - 1.25f*r_a*r_a + 0.375f*r_a*r_a*r_a*r_a);
  }
  else {
    return 1/r;
  }
}


/*
 * e  - 3D lattice of potentials to be computed
 * q  - 3D lattice of charges
 * nx, ny, nz  - dimensions of e and q lattices
 * a  - cutoff distance
 * h  - grid spacing
 *
 * returns potential lattice computed as weighted sums of surrounding charges
 * (for nonperiodic boundaries, weighted sum stops at lattice edges)
 */
void nonperiodic_lattice_direct(float *e, const float *q,
    int nx, int ny, int nz, float a, float h)
{
  float esum;
  float r2, r;
  float dx, dy, dz;
  int i, j, k;
  int ii, jj, kk;

  /* loop through e lattice */
  for (k = 0;  k < nz;  k++) {
    for (j = 0;  j < ny;  j++) {
      for (i = 0;  i < nx;  i++) {

        /* second loop through e lattice, sum weighted q contributions */
        esum = 0;
        for (kk = 0;  kk < nz;  kk++) {
          for (jj = 0;  jj < ny;  jj++) {
            for (ii = 0;  ii < nx;  ii++) {
              dx = h*(ii - i);
              dy = h*(jj - j);
              dz = h*(kk - k);
              r2 = dx*dx + dy*dy + dz*dz;
              if (r2 >= 4*a*a) continue;
              r = sqrtf(r2);
              esum += q[kk*ny*nx + jj*nx + ii] * (g(a,r) - g(2*a,r));
            }
          }
        }
        e[k*ny*nx + j*nx + i] = esum;
      }
    }
  }

}


/*
 * e  - 3D lattice of potentials to be computed
 * q  - 3D lattice of charges
 * nx, ny, nz  - dimensions of e and q lattices
 * a  - cutoff distance
 * h  - grid spacing
 *
 * returns potential lattice computed as weighted sums of surrounding charges
 * (for periodic boundaries, weighted sum wraps around lattice edges)
 */
void periodic_lattice_direct(float *e, const float *q,
    int nx, int ny, int nz, float a, float h)
{
  float esum;
  float r2, r;
  float dx, dy, dz;
  int i, j, k;
  int ii, jj, kk;

  float lx = h*nx;
  float ly = h*ny;
  float lz = h*nz;

  float hlx = 0.5*lx;
  float hly = 0.5*ly;
  float hlz = 0.5*lz;

  /* loop through e lattice */
  for (k = 0;  k < nz;  k++) {
    for (j = 0;  j < ny;  j++) {
      for (i = 0;  i < nx;  i++) {

        /* second loop through e lattice, sum weighted q contributions */
        esum = 0;
        for (kk = 0;  kk < nz;  kk++) {
          for (jj = 0;  jj < ny;  jj++) {
            for (ii = 0;  ii < nx;  ii++) {
              dx = h*(ii - i);
              dy = h*(jj - j);
              dz = h*(kk - k);

              /* find closest periodic image */
              if (dx < -hlx) dx += lx;
              else if (dx > hlx) dx -= lx;
              if (dy < -hly) dy += ly;
              else if (dy > hly) dy -= ly;
              if (dz < -hlz) dz += lz;
              else if (dz > hlz) dz -= lz;

              r2 = dx*dx + dy*dy + dz*dz;
              if (r2 >= 4*a*a) continue;
              r = sqrtf(r2);
              esum += q[kk*ny*nx + jj*nx + ii] * (g(a,r) - g(2*a,r));

#if defined(ANALYSIS)
              printf("i=%d  j=%d  k=%d  ii=%d  jj=%d  kk=%d  q[%d]=%g  wt=%g\n",
                  i, j, k, ii, jj, kk, kk*ny*nx + jj*nx + ii,
                  q[kk*ny*nx + jj*nx + ii], g(a,r)-g(2*a,r));
#endif
            }
          }
        }
        e[k*ny*nx + j*nx + i] = esum;

#if defined(ANALYSIS)
        printf("i=%d  j=%d  k=%d  e[%d]=%g\n",
            i, j, k, k*ny*nx + j*nx + i, e[k*ny*nx + j*nx + i]);
#endif
      }
    }
  }

}
