/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: AtomRep.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.7 $      $Date: 1994/12/11 01:14:47 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Parse and maintain the data for how a molecule should be represented.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: AtomRep.C,v $
 * Revision 1.7  1994/12/11  01:14:47  dalke
 * Added ribbons - warning - it workds, bu the ribbon interface
 *  WILL change
 *
 * Revision 1.6  1994/11/03  01:03:05  billh
 * Changed text and enum name for 'off' option in atom representation.
 *
 * Revision 1.5  1994/11/02  23:32:44  billh
 * Added 'none' option to AtomRep to quickly turn off graphics reps
 *
 * Revision 1.4  1994/11/02  01:32:41  billh
 * Reversed direction bonds drawn for 2nd half of cylinders.
 * Make CPK use 0.25 scale factor to VDW radius and bond radius.
 *
 * Revision 1.3  94/10/31  20:34:06  billh
 * Made more consistent, by setting default parameters in constructor and
 * having these values only be changed when the new commands explicitely
 * specify new values.
 * 
 * Revision 1.2  94/10/26  23:19:44  billh
 * Removed 'ok' member; removed initial string in constructor; added 'cmd_parse'
 * and operator= routines, as well as 'change' routine.
 * 
 * Revision 1.1  94/09/23  06:01:39  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/AtomRep.C,v 1.7 1994/12/11 01:14:47 dalke Exp $";
#endif

#include <string.h>
#include <stdlib.h>
#include "AtomRep.h"
#include "Inform.h"
#include "utilities.h"
#include "config.h"

// a string array with text descriptions of representation methods
char *AtomRepName[AtomRep::TOTAL] = { "Lines", "CPK", "VDW", "Dotted",
	"Licorice", "Ribbons", "Off" };


//////////////////////////  constructor and destructor
// constructor; parse string and see if OK
AtomRep::AtomRep(void) {
  
  MSGDEBUG(2,"Creating new AtomRep object ..." << sendmsg);

  // initialize variables
  lineThickness = 3;
  sphereRad = 1.0;
  bondRad = 0.5;
  sphereRes = 6;
  bondRes = 10;
  repMethod  = DEFAULT_ATOMREP;
  strcpy(cmdStr, AtomRepName[DEFAULT_ATOMREP]);
}


// copy constructor
AtomRep::AtomRep(AtomRep& ar) {
  strcpy(cmdStr, ar.cmdStr);
  repMethod = ar.repMethod;
  sphereRad = ar.sphereRad;
  bondRad = ar.bondRad;
  sphereRes = ar.sphereRes;
  bondRes = ar.bondRes;
  lineThickness = ar.lineThickness;
}


//////////////////////////  private routines

// parse the given command, and store results.  Return success.
int AtomRep::parse_cmd(char *newcmd) {
  int argc, ok = TRUE;
  char *argv[128], *cmdStrTok = NULL;

  RepMethod newMethod = repMethod;
  int newThick = lineThickness;
  int newBondRes = bondRes, newSphRes = sphereRes;
  float newSphRad = sphereRad, newBondRad = bondRad;
  
  // make sure the new command is not too long
  if(newcmd && strlen(newcmd) > MAX_ATOMREP_CMD) {
    msgErr << "Atom representation string is too long (over ";
    msgErr << MAX_ATOMREP_CMD << " characters)." << sendmsg;
    return FALSE;
  }

  // tokenize the command
  if(!newcmd || !(cmdStrTok = str_tokenize(newcmd, &argc, argv))) {
    // no command; keep current settings
    return TRUE;
  }

  // now parse the command
  if(!strupncmp(argv[0], "default", CMDLEN) && argc == 1) {
    newMethod = DEFAULT_ATOMREP;
  } else if(!strupncmp(argv[0], AtomRepName[LINES], CMDLEN) && argc < 3) {
    newMethod = LINES;
    if(argc > 1)
      newThick = atoi(argv[1]);
  } else if(!strupncmp(argv[0], AtomRepName[CPK], CMDLEN) && argc < 6) {
    newMethod = CPK;
    if(argc > 1)
      newSphRad = atof(argv[1]);
    if(argc > 2)
      newBondRad = atof(argv[2]);
    if(argc > 3)
      newSphRes = atoi(argv[3]);
    if(argc > 4)
      newBondRes = atoi(argv[4]);
  } else if(!strupncmp(argv[0], AtomRepName[VDW], CMDLEN) && argc < 4) {
    newMethod = VDW;
    if(argc > 1)
      newSphRad = atof(argv[1]);
    if(argc > 2)
      newSphRes = atoi(argv[2]);
  } else if(!strupncmp(argv[0], AtomRepName[DOTTED], CMDLEN) && argc < 4) {
    newMethod = DOTTED;
    if(argc > 1)
      newSphRad = atof(argv[1]);
    if(argc > 2)
      newSphRes = atoi(argv[2]);
  } else if(!strupncmp(argv[0], AtomRepName[LICORICE], CMDLEN) && argc < 5) {
    newMethod = LICORICE;
    if(argc > 1)
      newBondRad = atof(argv[1]);
    if(argc > 2)
      newSphRes = atoi(argv[2]);
    if(argc > 3)
      newBondRes = atoi(argv[3]);
  } else if(!strupncmp(argv[0], AtomRepName[RIBBONS], CMDLEN) && argc == 1) {
     newMethod = RIBBONS;
  } else if(!strupncmp(argv[0], AtomRepName[REPOFF], CMDLEN) && argc == 1) {
    newMethod = REPOFF;
  } else {
    // unknown representation
    ok = FALSE;
  }
  
  // check if error
  if(!ok) {
    msgErr << "Incorrect atom representation command '" << newcmd << "'";
    msgErr << sendmsg;
  } else {
    // command was successful, save new settings
    repMethod = newMethod;
    sphereRad = newSphRad;
    bondRad = newBondRad;
    sphereRes = newSphRes;
    bondRes = newBondRes;
    lineThickness = newThick;
    strcpy(cmdStr, newcmd);
  }

  // delete parsing space
  delete [] cmdStrTok;
  
  return ok;
}


//////////////////////////  public routines

// equal operator, to change the current settings.
AtomRep& AtomRep::operator=(const AtomRep &ar) {

  // copy values
  strcpy(cmdStr, ar.cmdStr);
  repMethod = ar.repMethod;
  sphereRad = ar.sphereRad;
  bondRad = ar.bondRad;
  sphereRes = ar.sphereRes;
  bondRes = ar.bondRes;
  lineThickness = ar.lineThickness;
  
  return *this;
}
