/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Buttons.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 1994/11/08 09:12:15 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Base class for button type inputs which are associated with a tracker/
 * sensor pair.  Also includes two derived classes, KbdButtons (for keyboard
 * entry with the F keys) and CaveButtons (for input from the Cave).
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: Buttons.h,v $
 * Revision 1.3  1994/11/08  09:12:15  dalke
 * set up CaveButtons correctly
 *
 * Revision 1.2  1994/09/17  10:53:18  dalke
 * added RCS header
 *
 ***************************************************************************/
#ifndef BUTTONS_H
#define BUTTONS_H
#include "utilities.h"

// base class for the different types of button inputs
// (derived classes are KbdButtons and CaveButtons)
class Buttons {
protected:
  int *state;      // place to store the button info so I don't
                   //   constantly repoll the same data
  int *changes;    // not FALSE indicated that that button changed since
                   //   the last polling
  int has_changed; // has anything changed since the last time?  Here for
                   // speed
public:
  Buttons(void) {
   state = NULL;
  }
  
  virtual ~Buttons(void) {
    state = NULL;  // better be deleted by this time
    changes = NULL;
  }
  
  virtual int num(void) = 0;  // number of buttons
  
  // return state of a button -- TRUE is on, FALSE is off
  int button(int buttonnum) { 
    return 0<= buttonnum && buttonnum < num() ?
             state[buttonnum] : FALSE;
  }
  
  // return the value by which a button has changed
  int change(int buttonnum) {
     return 0<= buttonnum && buttonnum < num() ?
              changes[buttonnum] : 0;
  }

  // for speed, I want to know if anything has changed since the last
  // polling w/o actually checking all the state variables
  int changed(void) { 
    return has_changed; 
  }
  
  // get the current button state.  If it changed, generate a Command
  virtual void check_event(void) = 0;
};


// Takes input from the keyboard
class KbdButtons : public Buttons {
private:
  int my_bank;  // which bank of F keys to use, (F1, F2, F3) , (F5, ...)
  int *tmp_state;  // temp, but I don't want to new mem every time
public:
  KbdButtons(int bank) {
    if (bank<0) bank=0;
    if (bank>3) bank=3;
    my_bank = bank;
    state = new int[3];
    state[0] = state[1] = state[2] = FALSE;
    changes = new int[3];
    changes[0] = changes[1] = changes[2] = FALSE;
    tmp_state = new int[3];
  } 
  virtual ~KbdButtons(void) {
    delete state;
    delete changes;
    delete tmp_state;
  }
  int num(void) { return 3;  } // yep, 3 keys for now
  void check_event(void);
};

#ifdef VMDCAVE
// Takes input from the CAVE buttons
class CaveButtons : public Buttons {
  int *tmp_state;  // temp, but I don't want to new mem every time
public:
  CaveButtons( void ) : Buttons() {
    state = new int[3];
    state[0] = state[1] = state[2] = FALSE;
    changes = new int[3];
    changes[0] = changes[1] = changes[2] = FALSE;
    tmp_state = new int[3];
  }
  ~CaveButtons( void ) {
    delete state;
    delete changes;
    delete tmp_state;
  }
  int num(void) { return 3; }
  void check_event(void);
};
#endif

#endif // BUTTONS_H
