/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdMol.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 1995/02/12 07:18:37 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Command objects for affecting molecules.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: CmdMol.C,v $
 * Revision 1.6  1995/02/12  07:18:37  dalke
 * added several new "Molecule" types
 *
 * Revision 1.5  1994/11/22  02:31:09  billh
 * Improved way new molecule command specifies structure and coordiante
 * file type ... can be used (mostly) unchanged when new types are added.
 *
 * Revision 1.4  94/10/26  23:21:34  billh
 * Added routines to change the settings for graphics representations
 * (MOL_MODREP, MOL_MODREPITEM).
 * 
 * Revision 1.3  94/10/21  03:50:45  billh
 * Updated for new simulation control commands, and remoteList object.
 * 
 * Revision 1.2  1994/10/04  20:29:22  billh
 * Changes to get to compile on HP's ... not there yet,though.
 *
 * Revision 1.1  1994/10/03  01:42:46  dalke
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/CmdMol.C,v 1.6 1995/02/12 07:18:37 dalke Exp $";
#endif

#include "CmdMol.h"
#include "Global.h"
#include "MoleculeFilePDB.h"
#include "MoleculeFilePSF.h"
#include "MoleculeFileEDM.h"
#include "MoleculeFileGraph.h"
#include "MoleculeFileRaster3D.h"
  
// except this, for remote connections
#ifdef VMDREMOTE
#include "MoleculeRemote.h"
#include "RemoteList.h"
#endif

// the following uses the Cmdtypes MOL_NEW, MOL_LIST, MOL_DEL, MOL_ACTIVE,
// MOL_FIX, MOL_ON, MOL_TOP, MOL_SELECT, MOL_REP, MOL_COLOR, MOL_ADDREP,
// MOL_MODREP, MOL_DELREP, MOL_MODREPITEM


/*
 * NOTES:
 *
 *	1) When referring to a molecule in a command by a number, the
 * unique molecule ID is used, NOT the relative index of the molecule into
 * the molecule list.  This is because the relative index changes as molecules
 * are added/removed, but the unique ID stays the same until the molecule is
 * deleted.
 */


///////// load a new molecule, either from a file, or from a remote connection
 int CmdMolNew::do_execute(void) {
    Molecule *newmol = NULL;

    if(moleculeList != NULL) {
    
      // first get the proper type of molecule object ready
      if(molSource == MOLFILE) {
        // read in a new molecule from a file
	 switch(molStrFileType) {
	  case MoleculeFile::PDB :
	    newmol = new MoleculeFilePDB(molStrFile, moleculeList);
	    break;
	  case MoleculeFile::PSF :
	    newmol = new MoleculeFilePSF(molStrFile, moleculeList);
	    break;
	  case MoleculeFile::EDM :
	    newmol = new MoleculeFileEDM(molStrFile, moleculeList);
	    break;
	  case MoleculeFile::GRAPH :
	    newmol = new MoleculeFileGraph(molStrFile, moleculeList);
	    break;
	  case MoleculeFile::RASTER3D :
	    newmol = new MoleculeFileRaster3D(molStrFile, moleculeList);
	    break;
	  default:
	    msgErr << "Cannot understand that file type\n" << sendmsg;
	    return FALSE;
	 }
      } else if(molSource == MOLREMOTE) {
#ifdef VMDREMOTE
        // read in molecule from a remote connection
	MoleculeRemote *remmol = new MoleculeRemote(remote, moleculeList);
	newmol = remmol;
	remote = NULL;
	
	// add the remote molecule to the remoteList
	if(remoteList)
	  remoteList->add_molecule(remmol);
#else
        msgErr << "Remote connection capability has not been included in this";
	msgErr << "executable.\nRecompile with VMDREMOTE defined." << sendmsg;
	return(FALSE);
#endif
      }

      // now load structure
      if(!newmol->create()) {	// does real initialization and creation
	msgErr << "Cannot create new molecule." << sendmsg;
	delete newmol;
	newmol = NULL;
      } else {
        // load was successful ...
	
        // now load coordinate file, if any
        if(molCoorFile)
          newmol->read_coor_file(molCoorFile, molCoorFileType);
	  
	// add the molecule to the molecule list
	newID = moleculeList->mol_index_from_id(
			 moleculeList->add_molecule(newmol));
      }
    }
    
    return (newmol != NULL);
  }

// constructor A: read struct from file
CmdMolNew::CmdMolNew(char *fname, int sftype, char *cfile, int cftype, 
                      int newUIid)
  : Command(Command::MOL_NEW, newUIid) {

  newID = (-1);
  molSource = MOLFILE;
  molStrFile = stringdup(fname);
  molStrFileType = sftype;
  *cmdText << "newmol ";
  *cmdText << structureFileTypeNames[sftype] << " " << molStrFile;

  if(cfile) {
    molCoorFile = stringdup(cfile);
    molCoorFileType = cftype;
    *cmdText << CoorFileSuffix[cftype] << " " << molCoorFile;
  } else
    molCoorFile = NULL;

  *cmdText << ends;
}

// constructor B: read struct from remote.  Don't specify anything.
CmdMolNew::CmdMolNew(int newUIid) : Command(Command::MOL_NEW, newUIid) {
  newID = (-1);
  molSource = MOLREMOTE;
  molStrFile = molCoorFile = NULL;
  *cmdText << ends;		// this has no text command equiv ... it is
    			      // only run by other commands
}

// destructor
CmdMolNew::~CmdMolNew(void) {
  if(molStrFile)  delete [] molStrFile;
  if(molCoorFile) delete [] molCoorFile;
}


///////////////  list the current set of loaded molecules
  // print out molecule summary line
  void CmdMolList::print_mol_summary(int i) {
    if(i >= 0 && i < moleculeList->num()) {
      Molecule *mol = moleculeList->molecule(i);
      msgInfo << mol->name;
      msgInfo << "  Atoms:" << mol->nAtoms;
      msgInfo << "  Frames (C):" << mol->num() << "(" << mol->frame() << ")";
      msgInfo << "  Status:";
      msgInfo << (moleculeList->active(mol) 	? 'A' : 'a');
      msgInfo << (moleculeList->displayed(mol)	? 'D' : 'd');
      msgInfo << (moleculeList->fixed(mol)	? 'F' : 'f');
      msgInfo << (moleculeList->is_top(mol)	? 'T' : 't');
      msgInfo << sendmsg;
    } else {
      msgErr << "Illegal molecule ID." << sendmsg;
    }
  }
  
  // print out atom rep summary line
  void CmdMolList::print_arep_summary(Molecule *mol, int i) {
    DrawMolItem *arep = mol->component(i);
    msgInfo << i << ": " << (arep->displayed() ? " on" : "off") << ", ";
    msgInfo << (arep->atomSel)->selected << " atoms selected." << sendmsg;
    msgInfo << "  Coloring method: " << (arep->atomColor)->cmdStr << sendmsg;
    msgInfo << "   Representation: " << (arep->atomRep)->cmdStr << sendmsg;
    msgInfo << "        Selection: " << (arep->atomSel)->cmdStr << sendmsg;
  }
  
int CmdMolList::do_execute(void) {
    int i,retval;
    Molecule *mol;

    if(retval = (moleculeList != NULL)) {
      if(moleculeList->num() == 0) {
        msgInfo << "No molecules are currently loaded." << sendmsg;
      } else {
        if(idList || whichMol < 0) {
          msgInfo << "Currently loaded molecules:" << sendmsg;
          msgInfo << "---------------------------" << sendmsg;
	  if(idList) {
	    for(i=0; i < numList; i++)
	      print_mol_summary(moleculeList->mol_index_from_id(idList[i]));
	  } else {
            for(i=0; i < moleculeList->num(); i++)
	      print_mol_summary(i);
	  }
	} else if((i = moleculeList->mol_index_from_id(whichMol)) >= 0) {
	  mol = moleculeList->molecule(i);
	  msgInfo << "Status of molecule " << mol->name << ":" << sendmsg;
	  print_mol_summary(whichMol);
	  msgInfo << "Atom representations: " << mol->components() << sendmsg;
	  msgInfo << "-------------------------" << sendmsg;
	  for(i=0; i < mol->components(); i++)
	    print_arep_summary(mol, i); 
	} else {
	  msgErr << "Illegal molecule ID " << whichMol << sendmsg;
	  retval = FALSE;
	}
      }
    }
    return retval;
  }

  CmdMolList::CmdMolList(int wm, int newUIid) 
    : Command(Command::MOL_LIST, newUIid) {
    whichMol = wm;
    idList = NULL;
    numList = 0;
    *cmdText << "mol list ";
    if(wm >= 0)
      *cmdText << wm;
    *cmdText << ends;
  }
  
  // constructor for a whole list of molecules.  The list will be deleted
  // when this is done, but not allocated, so the caller must new an array
  // of integers and pass it to this object.
  CmdMolList::CmdMolList(int n, int *list, int newUIid)
    : Command(Command::MOL_LIST, newUIid) {
    whichMol = (-1);
    if(list && n > 0) {
      idList = list;
      numList = n;
      for(int i=0; i < numList; i++) {
	if(i > 0)
	  *cmdText << "\n";
        *cmdText << "mol list " << list[i];
      }
    } else {
      idList = NULL;
      numList = 0;
      *cmdText << "mol list";
    }
    *cmdText << ends;
  }
  
  CmdMolList::~CmdMolList(void) {
    if(idList)  delete [] idList;
  }


///////////////  delete the Nth molecule in the molecule List
// peform actual action for molecule i.  return success.
int CmdMolDelete::do_action(int i) {
  int retval = FALSE;
  
#ifdef VMDREMOTE
  // remove from remoteList, if necessary
  remoteList->del_molecule(remoteList->rem_index_from_id(i));
#endif

  if(!moleculeList->del_molecule(moleculeList->mol_index_from_id(i)))
    msgErr << "Cannot delete molecule with ID " << i << sendmsg;
  else
    retval = TRUE;
  return retval;
}

int CmdMolDelete::do_execute(void) {
    int retval = TRUE;
    if(retval = (moleculeList != NULL)) {
      if(idList)
        for(int i=0; i < numList; i++)
	  retval = retval && do_action(idList[i]);
      else
        retval = retval && do_action(n);
    }
    return retval;
  }

  CmdMolDelete::CmdMolDelete(int nmol,int newUIid) 
   : Command(Command::MOL_DEL,newUIid) {
    n = nmol;
    idList = NULL;
    numList = 0;
    *cmdText << "mol delete " << n << ends;
  }

  // alternate constructor where a list of molecules is given
  CmdMolDelete::CmdMolDelete(int numn, int *list, int newUIid)
    : Command(Command::MOL_DEL,newUIid) {
    idList = list;
    numList = numn;
    for(int i=0; i < numList; i++) {
      if(i > 0)
	*cmdText << "\n";
      *cmdText << "mol delete " << idList[i];
    }
    *cmdText << ends;
  }
  
 CmdMolDelete::~CmdMolDelete(void) {
    if(idList)  delete [] idList;
  }


///////////////  make the Nth molecule 'active' or 'inactive'
  // peform actual action for molecule i.  return success.
  int CmdMolActive::do_action(int i) {
    int retval = FALSE;
    int m = moleculeList->mol_index_from_id(i);
    if(m >= 0) {
      retval = TRUE;
      if(yn)
        moleculeList->activate(m);
      else
        moleculeList->inactivate(m);
    } else
      msgErr << "Cannot set active status of molecule with ID " << i <<sendmsg;
    return retval;
  }

int CmdMolActive::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if(idList) {
        for(int i=0; i < numList; i++)
	  retval = retval && do_action(idList[i]);
      } else if(n < 0) {
        for(int i=0; i < moleculeList->num(); i++)
          retval = retval && do_action((moleculeList->molecule(i))->id());
      } else {
        retval = retval && do_action(n);
      }
    }
    return retval;
  }
  
  CmdMolActive::CmdMolActive(int nmol, int newyn, int newUIid)
    : Command(Command::MOL_ACTIVE,newUIid) {
    n = nmol;
    yn = newyn;
    idList = NULL;
    numList = 0;
    *cmdText << "mol ";
    *cmdText << (yn ? "active " : "inactive ");
    if(n >= 0)
      *cmdText << n;
    *cmdText << ends;
  }

  // alternate constructor where a list of molecules is given
  CmdMolActive::CmdMolActive(int numn, int *list, int newyn, int newUIid)
    : Command(Command::MOL_ACTIVE,newUIid) {
    yn = newyn;
    idList = list;
    numList = numn;
    for(int i=0; i < numList; i++) {
      if(i > 0)
	*cmdText << "\n";
      *cmdText << "mol ";
      *cmdText << (yn ? "active " : "inactive ");
      *cmdText << idList[i];
    }
    *cmdText << ends;
  }
  
  CmdMolActive::~CmdMolActive(void) {
    if(idList)  delete [] idList;
  }

///////////////  make the Nth molecule 'fixed' or 'free'
  // peform actual action for molecule i.  return success.
  int CmdMolFix::do_action(int i) {
    int retval = FALSE;
    int m = moleculeList->mol_index_from_id(i);
    if(m >= 0) {
      retval = TRUE;
      if(yn)
        moleculeList->fix(m);
      else
        moleculeList->unfix(m);
    } else
      msgErr << "Cannot set fixed status of molecule with ID " << i <<sendmsg;
    return retval;
  }

  int CmdMolFix::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if(idList) {
        for(int i=0; i < numList; i++)
	  retval = retval && do_action(idList[i]);
      } else if(n < 0) {
        for(int i=0; i < moleculeList->num(); i++)
          retval = retval && do_action((moleculeList->molecule(i))->id());
      } else {
        retval = retval && do_action(n);
      }
    }
    return retval;
  }
  
  CmdMolFix::CmdMolFix(int nmol, int newyn, int newUIid)
    : Command(Command::MOL_FIX,newUIid) {
    n = nmol;
    yn = newyn;
    idList = NULL;
    numList = 0;
    *cmdText << "mol ";
    *cmdText << (yn ? "fixed " : "free ");
    if(n >= 0)
      *cmdText << n;
    *cmdText << ends;
  }

  // alternate constructor where a list of molecules is given
  CmdMolFix::CmdMolFix(int numn, int *list, int newyn, int newUIid)
    : Command(Command::MOL_FIX,newUIid) {
    yn = newyn;
    idList = list;
    numList = numn;
    for(int i=0; i < numList; i++) {
      if(i > 0)
	*cmdText << "\n";
      *cmdText << "mol ";
      *cmdText << (yn ? "fixed " : "free ");
      *cmdText << idList[i];
    }
    *cmdText << ends;
  }
  
  CmdMolFix::~CmdMolFix(void) {
    if(idList)  delete [] idList;
  }




///////////////  make the Nth molecule 'on' or 'off'
  // peform actual action for molecule i.  return success.
  int CmdMolOn::do_action(int i) {
    int retval = FALSE;
    int m = moleculeList->mol_index_from_id(i);
    if(m >= 0) {
      retval = TRUE;
      if(yn)
        moleculeList->show(m);
      else
        moleculeList->hide(m);
    } else
      msgErr<<"Cannot set displayed status of molecule with ID "<< i <<sendmsg;
    return retval;
  }

  int CmdMolOn::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if(idList) {
        for(int i=0; i < numList; i++)
	  retval = retval && do_action(idList[i]);
      } else if(n < 0) {
        for(int i=0; i < moleculeList->num(); i++)
          retval = retval && do_action((moleculeList->molecule(i))->id());
      } else {
        retval = retval && do_action(n);
      }
    }
    return retval;
  }

  CmdMolOn::CmdMolOn(int nmol, int newyn, int newUIid)
    : Command(Command::MOL_ON,newUIid) {
    n = nmol;
    yn = newyn;
    idList = NULL;
    numList = 0;
    *cmdText << "mol ";
    *cmdText << (yn ? "show " : "hide ");
    if(n >= 0)
      *cmdText << n;
    *cmdText << ends;
  }

  // alternate constructor where a list of molecules is given
  CmdMolOn::CmdMolOn(int numn, int *list, int newyn, int newUIid)
    : Command(Command::MOL_ON,newUIid) {
    yn = newyn;
    idList = list;
    numList = numn;
    for(int i=0; i < numList; i++) {
      if(i > 0)
	*cmdText << "\n";
      *cmdText << "mol ";
      *cmdText << (yn ? "show " : "hide ");
      *cmdText << idList[i];
    }
    *cmdText << ends;
  }
  
  CmdMolOn::~CmdMolOn(void) {
    if(idList)  delete [] idList;
  }



///////////////  make the Nth molecule 'top'
 int CmdMolTop::do_execute(void) {
    int retval;
    if(retval=(moleculeList!=NULL) && moleculeList->mol_index_from_id(n) >=0) {
      moleculeList->make_top(moleculeList->mol_index_from_id(n));
    } else {
      msgErr << "Cannot make molecule " << moleculeList->mol_index_from_id(n);
      msgErr << " the top molecule" << sendmsg;
    }
    return retval;
  }
  
  CmdMolTop::CmdMolTop(int nmol, int newUIid)
    : Command(Command::MOL_TOP,newUIid) {
    n = nmol;
    *cmdText << "mol top ";
    *cmdText << n << ends;
  }


///////////// set the default atom selection in moleculeList
  int CmdMolSelect::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if(sel) {
        retval = moleculeList->set_selection(sel);
	if(!retval)
	  msgErr << "Unable to change current selection." << sendmsg;
      } else {
        msgInfo << "Current atom selection:";
	msgInfo << "\n  '" << moleculeList->selection() << "'" << sendmsg;
      }
    }
    return retval;
  }

  CmdMolSelect::CmdMolSelect(char *newsel, int newUIid)
    : Command(Command::MOL_SELECT,newUIid) {
    *cmdText << "mol selection ";
    if(newsel) {
      *cmdText << newsel;
      sel = stringdup(newsel);
    } else {
      sel = NULL;
    }
    *cmdText << ends;
  }
  
  CmdMolSelect::~CmdMolSelect(void) {
    if(sel)  delete [] sel;
  }


///////////// set the default atom representation in moleculeList
 int CmdMolRep::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if(sel) {
        retval = moleculeList->set_representation(sel);
	if(!retval)
	  msgErr << "Unable to change current representation setting."
	         << sendmsg;
      } else {
        msgInfo << "Current atom representation:";
	msgInfo << "\n  '" << moleculeList->representation() << "'" << sendmsg;
      }
    }
    return retval;
  }

  CmdMolRep::CmdMolRep(char *newsel, int newUIid)
    : Command(Command::MOL_REP,newUIid) {
    *cmdText << "mol representation ";
    if(newsel) {
      *cmdText << newsel;
      sel = stringdup(newsel);
    } else {
      sel = NULL;
    }
    *cmdText << ends;
  }
  
  CmdMolRep::~CmdMolRep(void) {
    if(sel)  delete [] sel;
  }


///////////// set the default atom coloring method in moleculeList
 int CmdMolColor::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if(sel) {
        retval = moleculeList->set_color(sel);
	if(!retval)
	  msgErr << "Unable to change current color setting." << sendmsg;
      } else {
        msgInfo << "Current atom coloring method:";
	msgInfo << "\n  '" << moleculeList->color() << "'" << sendmsg;
      }
    }
    return retval;
  }

  CmdMolColor::CmdMolColor(char *newsel, int newUIid)
    : Command(Command::MOL_COLOR,newUIid) {
    *cmdText << "mol color ";
    if(newsel) {
      *cmdText << newsel;
      sel = stringdup(newsel);
    } else {
      sel = NULL;
    }
    *cmdText << ends;
  }
  
  CmdMolColor::~CmdMolColor(void) {
    if(sel)  delete [] sel;
  }


///////////// add a new representation to the active molecules
  int CmdMolAddRep::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
       if(n < 0 || moleculeList->mol_index_from_id(n) >= 0) {
        retval = moleculeList->add_rep(moleculeList->mol_index_from_id(n));
	if(!retval) {
	  msgErr << "Unable to add new molecule representation." << sendmsg;
	}
      } else {
        msgErr << "Illegal molecule ID " << n << sendmsg;
        retval = FALSE;
      }
    }
    return retval;
  }

  CmdMolAddRep::CmdMolAddRep(int newmol, int newUIid)
    : Command(Command::MOL_ADDREP,newUIid) {
       *cmdText << "mol addrep ";
       if(n >= 0) {
	  *cmdText << n;
	  n = newmol;
       }
       else if (n==-1) {   // check in case of "top"
	  *cmdText << "top";
	  if (moleculeList && moleculeList -> top()) {
	     n = moleculeList -> top() -> id();
	  }
       }
       *cmdText << ends;
    }


///////////// change a representation for the specified molecule
  int CmdMolChangeRep::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
       if (n < 0 && moleculeList -> top()) {
	  n =moleculeList -> top() -> id();
       }
      if( moleculeList->mol_index_from_id(n) >= 0) {
        retval =
	   moleculeList->change_rep(repn, moleculeList->mol_index_from_id(n));
	if(!retval) {
          msgErr << "Illegal mol representation index " << repn;
	  msgErr << " for molecule " << n << sendmsg;
	}
      } else {
        msgErr << "Illegal molecule ID " << n << sendmsg;
        retval = FALSE;
      }
    }
    return retval;
  }

  CmdMolChangeRep::CmdMolChangeRep(int rpos, int newmol, int newUIid)
    : Command(Command::MOL_MODREP,newUIid) {
       repn = rpos;
       *cmdText << "mol modrep " << repn << " ";
       if (n==-1) {
	  *cmdText << "top";
	  if (moleculeList && moleculeList -> top()) {
	     n = moleculeList -> top() -> id();
	  }
       } else {
	  n = newmol;
          *cmdText << n;
       }
       *cmdText << ends;
    }



///////////// change 1 representation characteristic for the specified molecule
  int CmdMolChangeRepItem::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if( moleculeList->mol_index_from_id(n) >= 0) {
        if(repData == COLOR)
          retval = moleculeList->change_repcolor(repn,
	  	moleculeList->mol_index_from_id(n), str);
	else if(repData == REP)
          retval = moleculeList->change_repmethod(repn,
	  	moleculeList->mol_index_from_id(n), str);
	else if(repData == SEL)
          retval = moleculeList->change_repsel(repn,
	  	moleculeList->mol_index_from_id(n), str);
	if(!retval) {
          msgErr << "Cannot change representation";
	  msgErr << " for molecule " << n << sendmsg;
	}
      } else {
        msgErr << "Illegal molecule ID " << n << sendmsg;
        retval = FALSE;
      }
    }
    return retval;
  }

  CmdMolChangeRepItem::CmdMolChangeRepItem(int rpos, int newmol,
  		RepData rd, char *s, int newUIid)
	: Command(Command::MOL_MODREPITEM,newUIid) {
    n = newmol;
    repn = rpos;
    repData = rd;
    str = stringdup(s);
    *cmdText << "mol mod";
    if(repData == COLOR)
      *cmdText << "color ";
    else if(repData == REP)
      *cmdText << "style ";
    else if(repData == SEL)
      *cmdText << "select ";
    *cmdText << repn << " ";
    if (n == -1) {
       *cmdText << "top";
       if (moleculeList && moleculeList->top()) {
	  n = moleculeList->top()->id();
       }
    } else {
       *cmdText << n;
    }
    *cmdText << " " << str << ends;
  }

  CmdMolChangeRepItem::~CmdMolChangeRepItem(void) {
    if(str)  delete [] str;
  }


///////////// delete a representation for the specified molecule
  int CmdMolDeleteRep::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if( moleculeList->mol_index_from_id(n) >= 0) {
        retval =
           moleculeList->del_rep(repn, moleculeList->mol_index_from_id(n));
	if(!retval) {
          msgErr << "Illegal mol representation index " << repn;
	  msgErr << " for molecule " << n << sendmsg;
	}
      } else {
        msgErr << "Illegal molecule ID " << n << sendmsg;
        retval = FALSE;
      }
    }
    return retval;
  }

  CmdMolDeleteRep::CmdMolDeleteRep(int rpos, int newmol, int newUIid)
    : Command(Command::MOL_DELREP,newUIid) {
    n = newmol;
    repn = rpos;
    *cmdText << "mol delrep " << repn << " ";
    if (n==-1) {
       *cmdText << "top";
       if (moleculeList && moleculeList -> top()) {
	  n = moleculeList -> top() -> id();
       }
    } else {
       *cmdText << n;
    }       
    *cmdText << ends;
  }
