/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DrawMolItem.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.17 $	$Date: 1995/03/04 05:08:26 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Child Displayable component of a molecule; this is responsible for doing
 * the actual drawing of a molecule.  It contains an atom color, atom
 * selection, and atom representation object to specify how this component
 * should look.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: DrawMolItem.C,v $
 * Revision 1.17  1995/03/04  05:08:26  billh
 * Added picking points at the atom positions
 *
 * Revision 1.16  1995/02/22  08:43:11  dalke
 * moved ribbons binary
 *
 * Revision 1.15  1994/12/13  05:39:39  billh
 * For now, ribbons do not use material characteristics.
 *
 * Revision 1.14  1994/12/13  04:54:52  billh
 * Lines drawn to midpoint of bonds even if other atom is not drawn.
 * Fixed ribbons display to use color scale, and material characteristics.
 *
 * Revision 1.13  1994/12/11  01:16:53  dalke
 * Added ribbons - warning - it works, but the render3D interface
 * WILL change
 *
 * Revision 1.12  1994/12/06  08:24:25  billh
 * Added ability to create color categories, due to being now derived from
 * ColorUser.
 * Added routines to have a new color list specified, and to check for when
 * a color in a relevant category is changed.
 *
 * Revision 1.11  94/11/12  09:56:43  billh
 * Increased allocated space for disp list, and draw lone atoms as '.'
 * 
 * Revision 1.10  1994/11/10  12:34:58  billh
 * Increased space that will be allocated for new representation, to be
 * proportional to max number of atoms, not number selected (which can now
 * change).
 *
 * Revision 1.9  94/11/03  01:03:05  billh
 * Changed text and enum name for 'off' option in atom representation.
 * 
 * Revision 1.8  1994/11/02  23:32:44  billh
 * Added 'none' option to AtomRep to quickly turn off graphics reps
 *
 * Revision 1.7  1994/11/02  01:35:07  billh
 * Fixed CPK and Licorice drawing  routines.
 *
 * Revision 1.6  94/10/31  23:51:57  billh
 * Uses line width, and added initial versions of CPK and licorice (but they
 * don't yet work).
 * 
 * Revision 1.5  94/10/26  23:22:30  billh
 * Added routines to allow commands to change the settings for the
 * current molecule representation options.
 * 
 * Revision 1.4  94/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 * 
 * Revision 1.3  1994/09/28  22:14:34  billh
 * Removed comment from include; cannot do this in HPUX.
 *
 * Revision 1.2  1994/09/24  20:26:09  billh
 * Added support for color; added routines to setup color index lists,
 * default atom/residue names, and initialization routines.
 *
 * Revision 1.1  94/09/23  06:01:39  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/DrawMolItem.C,v 1.17 1995/03/04 05:08:26 billh Exp $";
#endif

#include "DrawMolItem.h"
#include "DrawMolecule.h"
#include "DispCmds.h"
#include "Inform.h"
#include "Scene.h"

#include <sys/types.h>
#include <unistd.h>  // for getuid
#include <stdio.h>
#include "CoorPDB.h" // for write_pdb_record

// draw commands used by this object to indicate points to pick
static DispCmdPickPoint pickPoint;
static DispCmdPickPointIndex pickPointIndex;

//////////////////////////  constructor  

DrawMolItem::DrawMolItem(DrawMolecule *dm, AtomColor *ac, AtomRep *ar,
	AtomSel *as) 
	: Displayable3D(MULT, dm->name, dm, dm->nAtoms/8 + 2),
	  cmdTextX(".") {

  MSGDEBUG(1,"Creating new molecule representation, for parent molecule:\n");
  MSGDEBUG(1," ==> '" << dm->name << "'" << sendmsg);

  // save data and pointers to drawing method objects
  mol = dm;
  atomColor = ac;
  atomRep = ar;
  atomSel = as;
  
  // create some useful display commands
  dataCmd = new DispCmdData(3 * mol->nAtoms, NULL);

  // signal that we need a complete refresh next prepare
  needRegenerate = needNewFrame = TRUE;

  // register with the pick list, to indicate we have objects that may
  // be selected with the mouse
  register_with_picklist(this->origScene);
}


//////////////////////////  destructor  

DrawMolItem::~DrawMolItem(void) {

  MSGDEBUG(2,"Deleting DrawMolItem '" << name << "' ..." << sendmsg);

  delete atomColor;
  delete atomRep;
  delete atomSel;

  delete dataCmd;
}


//////////////////////////////// private routines 

// regenerate the command list
void DrawMolItem::create_cmdlist(void) {
  float *framepos;

  // do we need to recreate everything?
  if(needRegenerate) {
  
    MSGDEBUG(2, "Regenerating display list for molecule '"<< name<< "' ...");
    MSGDEBUG(2, sendmsg);

    // regenerate both data block and display commands
    needRegenerate = needNewFrame = FALSE;
    reset_disp_list();

    // only put in commands if there is a current frame
    if(mol->frame() >= 0) {
      // get current position coordinates
      framepos = (mol->current())->pos;

      // first put in data command
      dataCmd->putdata(framepos, this);
      
      // now put in drawing commands, which index into the above block
      if(atomRep->method() == AtomRep::LINES)
        draw_lines(framepos);
      else if(atomRep->method() == AtomRep::CPK)
        draw_cpk(framepos);
      else if(atomRep->method() == AtomRep::VDW)
        draw_vdw();
      else if(atomRep->method() == AtomRep::DOTTED)
        draw_dotted();
      else if(atomRep->method() == AtomRep::LICORICE)
        draw_licorice(framepos);
      else if(atomRep->method() == AtomRep::RIBBONS)
        draw_ribbons();
      else if(atomRep->method() == AtomRep::REPOFF) {
       	// don't do anything ... draws a blank object
      } else
        msgErr << "Illegal atom representation in DrawMolecule." << sendmsg;
    }

  } else if(needNewFrame) {
    // just update the coordinates ... assume there is a current frame
    needNewFrame = FALSE;
    dataCmd->reputdata((mol->current())->pos, this);
  }
}

//////////////////////////////// protected virtual routines
// do action due to the fact that a color for the given ColorList for
// the specified category has changed
void DrawMolItem::do_color_changed(ColorList *collist, int ccat) {

  // check the category ... see if is for the current coloring method
  if(atomColor && atomColor->current_color_use(collist, ccat))
    change_color(atomColor);		// recalcs colors; signals need redraw
}


//////////////////////////////// public routines 
// prepare for drawing ... do any updates needed right before draw.
void DrawMolItem::prepare(DisplayDevice *) {

  // see if the frame has changed
  if(mol->has_frame_changed()) {
    needNewFrame = TRUE;
    needRegenerate = TRUE;
  }
  
  create_cmdlist();
}


// change the atom coloring method.  Return success.
int DrawMolItem::change_color(AtomColor *ac) {
  if(ac) {
    *atomColor = *ac;
    needRegenerate = TRUE;
    return TRUE;
  } else
    return FALSE;
}


// change the atom rep method.  Return success.
int DrawMolItem::change_rep(AtomRep *ar) {
  if(ar) {
    *atomRep = *ar;
    needRegenerate = TRUE;
    return TRUE;
  } else
    return FALSE;
}


// change the atom selection method.  Return success.
int DrawMolItem::change_sel(AtomSel *as) {
  if(as) {
    *atomSel = *as;
    needRegenerate = TRUE;
    return TRUE;
  } else
    return FALSE;
}


//////////////////////////////// drawing rep routines 
void DrawMolItem::draw_lines(float *framepos) {
  float pos2[3], *fp1, *fp2;
  Atom *a1, *a2;
  int i,j,k,a2n,bondsdrawn;
  
  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms with lines and points...");
  MSGDEBUG(2, sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn off material characteristics, set line style
  cmdMaterials.putdata(FALSE,this);
  cmdLineType.putdata(SOLIDLINE, this);
  cmdLineWidth.putdata(atomRep->line_thickness(), this);

  // check all atoms if they are displayed, and if so draw bonds as lines
  for(i=0; i < mol->nAtoms; i++) {
    // for each atom, draw half-bond to other displayed atoms
    if(atomSel->on[i]) {
      a1 = mol->atom(i);
      bondsdrawn = 0;

      // set color
      cmdColorIndex.putdata(atomColor->color[i], this);
      fp1 = framepos + 3*i;
      
      // draw half-bond to each bonded, displayed partner
      // NOTE: bond mid-points are calculated TWICE, once for each atom.
      // Why? So each atom is drawn entirely, including all it's half-bonds,
      // after a single set-color command, instead of having a set-color
      // command for each bond.  This reduces total drawing primitives
      // considerably, at the cost of extra calculation here.
      for(j=0; j < a1->bonds; j++) {
        a2n = a1->bondTo[j];
//        if(atomSel->on[a2n]) {	// bonded atom displayed?
	  fp2 = framepos + 3*a2n;
          a2 = mol->atom(a2n);
	  for(k=0; k < 3; k++)		// calc midpoint
	    pos2[k] = 0.5 * ( *(fp1 + k) + *(fp2 + k) );
	  cmdLine.putdata(fp1, pos2, this);	// draw line
	  bondsdrawn++;			// increment counter of bonds to atom i
//	}
      }
      
      // if this atom is all alone (no bonds drawn), make an 'x'
      if(!bondsdrawn) {
        cmdTextPosIndex.putdata(3*i, this);
	cmdTextX.put(this);
      }

      // indicate this atom can be picked
      pickPointIndex.putdata(3*i, i, this);
    }
  }
}


// draw all atoms just as spheres of vdw radius
void DrawMolItem::draw_just_spheres(void) {
  float radscale;
  int i;

  radscale = atomRep->sphere_rad();

  // check all atoms if they are displayed, and if so draw a sphere
  for(i=0; i < mol->nAtoms; i++) {
    if(atomSel->on[i]) {
      // set color
      cmdColorIndex.putdata(atomColor->color[i], this);
      
      // draw sphere
      cmdSphereIndex.putdata(3*i, (mol->atom(i))->radius() * radscale, this);

      // indicate this atom can be picked
      pickPointIndex.putdata(3*i, i, this);
    }
  }
}


// draws molecule as solid, lighted VDW spheres
void DrawMolItem::draw_vdw(void) {
  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms as solid VDW spheres...");
  MSGDEBUG(2, sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn on material characteristics
  cmdMaterials.putdata(TRUE,this);

  // set sphere mode and resolution
  cmdSphres.putdata(atomRep->sphere_res(), this);
  cmdSphtype.putdata(SOLIDSPHERE, this);
  
  // draw all atoms as spheres
  draw_just_spheres();

}


// draws molecule as dotted VDW spheres
void DrawMolItem::draw_dotted(void) {

  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms as dotted VDW spheres...");
  MSGDEBUG(2,sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn off material characteristics
  cmdMaterials.putdata(FALSE,this);

  // set sphere mode and resolution
  cmdSphres.putdata(atomRep->sphere_res(), this);
  cmdSphtype.putdata(POINTSPHERE, this);
  
  // draw all atoms as spheres
  draw_just_spheres();
}


void DrawMolItem::draw_cpk(float *framepos) {
  float pos2[3], *fp1, *fp2;
  Atom *a1, *a2;
  int i,j,k,a2n,bondsdrawn;
  float radscale = 0.25 * atomRep->sphere_rad();
  float brad = 0.25 * atomRep->bond_rad();
  int bres = atomRep->bond_res();

  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms with CPK method ...");
  MSGDEBUG(2,sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn on material characteristics
  cmdMaterials.putdata(TRUE,this);
  
  // set sphere and cylinder mode and resolution
  cmdSphres.putdata(atomRep->sphere_res(), this);
  cmdSphtype.putdata(SOLIDSPHERE, this);

  // check all atoms if they are displayed, and if so draw bonds as cylinders,
  // and the atom as a sphere, with radius = 1/4 vdw radius
  for(i=0; i < mol->nAtoms; i++) {
    // for each atom, draw half-bond to other displayed atoms
    if(atomSel->on[i]) {
      a1 = mol->atom(i);
      bondsdrawn = 0;

      // set color
      cmdColorIndex.putdata(atomColor->color[i], this);
      fp1 = framepos + 3*i;
      
      // draw a sphere for the atom
      cmdSphereIndex.putdata(3*i, a1->radius() * radscale, this);
      
      // draw half-bond to each bonded, displayed partner
      for(j=0; j < a1->bonds; j++) {
        a2n = a1->bondTo[j];
//        if(atomSel->on[a2n]) {	// bonded atom displayed?
	  fp2 = framepos + 3*a2n;
          a2 = mol->atom(a2n);
	  for(k=0; k < 3; k++)		// calc midpoint
	    pos2[k] = 0.5 * ( *(fp1 + k) + *(fp2 + k) );
	  if(i < a2n)
	    cmdCylinder.putdata(fp1, pos2, brad, bres, this);	// draw cyl
	  else
	    cmdCylinder.putdata(pos2, fp1, brad, bres, this);
	  bondsdrawn++;			// increment counter of bonds to atom i
//	}
      }

      // indicate this atom can be picked
      pickPointIndex.putdata(3*i, i, this);
    }
  }
}


void DrawMolItem::draw_licorice(float *framepos) {
  float pos2[3], *fp1, *fp2;
  Atom *a1, *a2;
  int i,j,k,a2n,bondsdrawn;
  float brad = atomRep->bond_rad();
  int bres = atomRep->bond_res();

  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms with licorice bonds ...");
  MSGDEBUG(2,sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn on material characteristics
  cmdMaterials.putdata(TRUE,this);
  
  // set sphere and cylinder mode and resolution
  cmdSphres.putdata(atomRep->sphere_res(), this);
  cmdSphtype.putdata(SOLIDSPHERE, this);

  // check all atoms if they are displayed, and if so draw bonds as cylinders,
  // and the atom as a sphere
  for(i=0; i < mol->nAtoms; i++) {
    // for each atom, draw half-bond to other displayed atoms
    if(atomSel->on[i]) {
      a1 = mol->atom(i);
      bondsdrawn = 0;

      // set color
      cmdColorIndex.putdata(atomColor->color[i], this);
      fp1 = framepos + 3*i;
      
      // draw a sphere for the atom, with same radius as bond
      cmdSphereIndex.putdata(3*i, brad, this);
      
      // draw half-bond to each bonded, displayed partner
      for(j=0; j < a1->bonds; j++) {
        a2n = a1->bondTo[j];
//        if(atomSel->on[a2n]) {	// bonded atom displayed?
	  fp2 = framepos + 3*a2n;
          a2 = mol->atom(a2n);
	  for(k=0; k < 3; k++)		// calc midpoint
	    pos2[k] = 0.5 * ( *(fp1 + k) + *(fp2 + k) );
	  if(i < a2n)
	    cmdCylinder.putdata(fp1, pos2, brad, bres, this);	// draw cyl
	  else
	    cmdCylinder.putdata(pos2, fp1, brad, bres, this);
	  bondsdrawn++;			// increment counter of bonds to atom i
//	}
      }

      // indicate this atom can be picked
      pickPointIndex.putdata(3*i, i, this);
    }
  }
}

// use 'ribbon' ver 2.0 (though a "system" call) to create the ribbons
// ref: Bacon and Anderson, 1988; Merritt and Murphy, 1994)
void DrawMolItem::draw_ribbons(void) {

  // check that there are coordinates
  if (!mol->is_current()) {
    return;  // nope
  }
  // and get them
  Timestep *ts = mol->current();

  // write the different pfrags out to a temp file - a different
  // file per person
  long uid = getuid();
  char filename[100];
  sprintf(filename, "/tmp/vmdribbon.out.%ld", uid);
  FILE *outfile = fopen(filename, "w");
  if (!outfile) {
     msgErr << "Cannot open file " << filename << ": ribbons not made";
     msgErr << sendmsg;
     return;
  }

  // go through the list of protein fragments
  int res;
  int atomnum1, atomnum2;
  for (int i=0; i<mol->pfragList.num(); i++) {// go down each fragment
    for (int frag=0; frag<mol->pfragList[i]->num(); frag++){// get the residue
      res = (*mol->pfragList[i])[frag];                  // find the residue
      atomnum1 = mol->find_atom_in_residue("CA", res);   // and the right
      atomnum2 = mol->find_atom_in_residue("O", res);    // atoms for ribbons
      if (atomnum2 < 0) {  // ribbons doesn't use these; I'm just making
        atomnum2 = mol->find_atom_in_residue("OT1", res); // a double check
      }
      if (atomnum2 < 0) {
        atomnum2 = mol->find_atom_in_residue("OT2", res);
      }
      if (atomnum1 < 0 || atomnum2 < 0) {  // can't hurt to double check
        msgInfo << "Ooops! didn't find both of CA (" << atomnum1;
        msgInfo << ") and O (" << atomnum2 << ')' << sendmsg;
      } else {  // if all okay, write the ribbon to the file
        if (atomSel->on[atomnum1] && atomSel->on[atomnum2] ) {
  	  write_pdb_record(mol, ts, atomnum1, outfile);
	  write_pdb_record(mol, ts, atomnum2, outfile);
        }
      } 
    }
  }
  fclose(outfile);
      
  // call ribbons by making up the string to execute via system
  char exestr[100];
  sprintf(exestr, "rm -f /tmp/vmdribbon.in.%ld; /usr/local/bin/ribbon -d2 %s > /tmp/vmdribbon.in.%ld 2>/dev/null", 
    uid, filename, uid, filename);
  system(exestr);

  // parse the input created from ribbons
  sprintf(filename, "/tmp/vmdribbon.in.%ld", uid);
  FILE *infile = fopen(filename, "r");
  if (!infile || feof(infile) ){
    msgErr << "Could not start ribbons" << sendmsg;
    return;
  }
  char dataline[301];   // get by the header
  int linenum = 0;
  while (linenum < 20) {
    fgets(dataline, 300, infile);
    if (feof(infile)) {
      msgErr << "Couldn't read the ribbons header" << sendmsg;
      return;
    }
//    msgInfo << dataline << sendmsg;
    linenum++;
  }

  int tmpi;
  float f[9], c[3];
  int oddFace = TRUE;
  cmdMaterials.putdata(FALSE,this); // make it look pretty

  while (fgets(dataline, 300, infile)) {
    sscanf(dataline, "%d\n", &tmpi);
    if (tmpi != 1) {  // get position of the triangle and the color
      msgWarn << "Cannot understand render3d identifier " << tmpi;
      fgets(dataline, 300, infile);
      msgWarn << "skipped line is: " << dataline << sendmsg;
    } else {
      if (fgets(dataline, 300, infile)) {
        tmpi = sscanf(dataline, "%f %f %f %f %f %f %f %f %f %f %f %f",
                   f, f+1, f+2, f+3, f+4, f+5, f+6, f+7, f+8, c, c+1, c+2);
        if (tmpi != 12) {
          msgErr << "Could not read all of a triangle data!" << sendmsg;
          continue;
        }
        // add it to the drawing list
/*
        c[0]=c[0]*1.5 + 0.25; if (c[0]>1) c[0]=1; // empirical color fix
        c[1]=c[1]*1.5 + 0.25; if (c[1]>1) c[1]=1;
        c[2]=c[2]*1.5 + 0.25; if (c[2]>1) c[2]=1;
        cmdColorRGB.putdata(c, this);  // I don't know how to use my own colors
*/

	// set the color from the colorscale:
	// assume colors run from 0 ... 0.4; clamp to boundary if they exceed
	// this.
	
	// use blue component; use its value to find position in colorscale
	int colpos = (int)((float)(MAPCLRS-1)*(c[2] > 0.4 ? 1.0 : c[2] * 2.5));
	cmdColorIndex.putdata((atomColor->transparent() ? 
		MAPTRANSLUCENT(colpos) : MAPCOLOR(colpos)) , this);
		
	// draw triangle
	if(oddFace) {
          cmdTriangle.putdata(f, f+3, f+6, this);
	} else {
          cmdTriangle.putdata(f, f+6, f+3, this);
	}
	oddFace = !oddFace;
      } else {
        msgErr << "Unexpected end of file when reading render3d file";
        msgErr << sendmsg;
        break;
      }
    }
  }
}


