/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: GLDisplayDevice.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.9 $	$Date: 1995/01/09 08:51:24 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Subclass of DisplayDevice, this object has routines used by all the
 * different display devices that are GL-specific.  Will render drawing
 * commands into a single GL window.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: GLDisplayDevice.C,v $
 * Revision 1.9  1995/01/09  08:51:24  billh
 * Now turns OFF depth-cueing by default.
 *
 * Revision 1.8  1994/10/28  18:31:10  billh
 * Added definitions for line styles; disabled near and far clipping for
 * HP's using NPGL library.
 *
 * Revision 1.7  1994/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.6  1994/10/04  22:09:10  billh
 * Took out sphere library usage if __NPGL__ is defined, since the npgl library
 * does not have a version of the sphere drawing routines.  Spheres are
 * just drawn as points for npgl rendering.
 *
 * Revision 1.5  94/09/23  00:50:31  billh
 * Now derived from 'GLRenderer'; this level now just contains routines
 * to open a GL window and deal with stereo, projection, etc.
 * 
 * Revision 1.4  94/09/03  11:05:29  dalke
 * redid triangle to use precomputed norm, and added a square
 * 
 * Revision 1.3  1994/08/30  07:18:09  dalke
 * Added a triangle command
 *
 * Revision 1.2  1994/08/26  00:02:15  billh
 * Now does crystal-eyes stereo, both full-screen and in-a-window.
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/GLDisplayDevice.C,v 1.9 1995/01/09 08:51:24 billh Exp $";
#endif

#include <stdlib.h>
#include <math.h>
#include "GLDisplayDevice.h"
#include "Inform.h"
#include "utilities.h"

#ifndef YMAXSTEREO
#define YMAXSTEREO      491
#define YOFFSET         532
#endif

// static data for this object
static char *glStereoNameStr[GL_STEREO_MODES] = { "Off",
	"CrystalEyes", "SideBySide" };
	
/////////////////////////  constructor and destructor  

// constructor ... open a window and set initial default values
GLDisplayDevice::GLDisplayDevice(int *size, int *loc)
	: GLRenderer("GL Display") {

  MSGDEBUG(1, "Creating GLDisplayDevice ... " << sendmsg);

  // set up data possible before opening window
  stereoNames = glStereoNameStr;
  stereoModes = GL_STEREO_MODES;
  
  // open the window
  windowID = open_window(name, size, loc);

  // set flags for the capabilities of this display
  has2D = TRUE;
  aaAvailable = (getgdesc(GD_PNTSMOOTH_RGB) && getgdesc(GD_LINESMOOTH_RGB));
  cueingAvailable = TRUE;

  // turn on antialiasing, turn off depth-cueing
  aa_on();
  cueing_off();

  // reshape and clear the display, which initializes some other variables
  reshape();
  normal();
  clear();
  update();
}

// destructor ... close the window
GLDisplayDevice::~GLDisplayDevice(void) {
  winclose(windowID);
}
  
/////////////////////////  protected nonvirtual routines  

// create a new window and set it's characteristics
long GLDisplayDevice::open_window(char *nm, int *size, int *loc) {
  long retval;
  int i;
  
  MSGDEBUG(2,"GLDisplayDevice: opening window ..." << sendmsg);

  // keep this program in the foreground
  foreground();
  
  // open the graphics window
  if(loc && size)
    prefposition(loc[0], loc[0] + size[0] + 1, loc[1], loc[1] + size[1] + 1);
  else if(size)
    prefsize(size[0], size[1]);
  retval = winopen(nm);
  winconstraints();
  
  // set up graphics parameters
  doublebuffer();
#ifdef GD_STEREO_IN_WINDOW
  if(getgdesc(GD_STEREO_IN_WINDOW) == 1)
      stereobuffer();
#endif
  RGBmode();
  gconfig();
  zbuffer(1);
  glcompat(GLC_ZRANGEMAP, 0);
  lsetdepth(getgdesc(GD_ZMIN),getgdesc(GD_ZMAX));
  backface(FALSE);
  shademodel(GOURAUD);
  mmode(MVIEWING);
  subpixel(TRUE);

  // define graphics state based on already-set default values
#ifndef __NPGL__
  sphmode(SPH_HEMI,TRUE);
  sphmode(SPH_ORIENT,TRUE);
  sphmode(SPH_TESS,SPH_OCT);
#endif
  deflinestyle(1, 0xFFFF);
  deflinestyle(2, 0x3333);
  set_sphere_mode(sphereMode);
  set_sphere_res(sphereRes);
  set_line_width(lineWidth);
  set_line_style(lineStyle);

  // disable clipping, if using NPGL on HP's
#ifdef __NPGL__
#ifdef ARCH_HPUX9
  npglcompat(NPGLC_NEAR_CLIP, NPGLC_CLIP_OFF);
  npglcompat(NPGLC_FAR_CLIP, NPGLC_CLIP_OFF);
#endif
#endif

  MSGDEBUG(2,"GLDisplayDevice: defining lights ..." << sendmsg);

  // turn on lights if necessary
  for(i=0; i < DISP_LIGHTS; i++) {
    if(lightDefined[i]) {
      do_define_light(i, lightColor[i], lightPos[i]);
      do_activate_light(i, lightOn[i]);
    } else
      do_activate_light(i, FALSE);
  }

  // define the light model ... this never changes.
  // material display is turned on/off by binding MATERIALS.
  lmdef(DEFLMODEL,1,0,(float *)NULL);
  lmbind(LMODEL, 1);

  MSGDEBUG(2,"GLDisplayDevice: defining materials ..." << sendmsg);

  // define materials if necessary
  for(i=0; i < MAXCOLORS; i++) {
    if(matDefined[i])
      do_define_material(i, matData[i]);
  }
  do_activate_material(materialOn, materialsActive);
  
  // load current transformation matrix on stack
  ::loadmatrix((transMat.top()).mat);
  
  return retval;
}


// set the current perspective, based on the eye position and where we
// are looking.  This form is for a non-stereo perspective
void GLDisplayDevice::set_persp(DisplayEye my_eye) {

  if(dim() == 3) {	// use window perspective for 3D view
    // set projection
    mmode(MPROJECTION);
    window((Coord)cpLeft, (Coord)cpRight, (Coord)cpDown, (Coord)cpUp,
           (Coord)nearClip, (Coord)farClip);
	   
    // define eye and look at some point.  Assumes up vector = (0,1,0)
    if(my_eye == NOSTEREO) {
      lookat(eyePos[0], eyePos[1], eyePos[2],
      	eyePos[0] + eyeDir[0], eyePos[1] + eyeDir[1], eyePos[2] + eyeDir[2],
	0);
    } else if(my_eye == LEFTEYE) {
      lookat(eyePos[0] - eyeSepDir[0], eyePos[1] - eyeSepDir[1],
      	eyePos[2] - eyeSepDir[2],
	eyePos[0] + eyeDir[0], eyePos[1] + eyeDir[1], eyePos[2] + eyeDir[2],
	0);
    } else {	// right eye
      lookat(eyePos[0] + eyeSepDir[0], eyePos[1] + eyeSepDir[1],
      	eyePos[2] + eyeSepDir[2],
	eyePos[0] + eyeDir[0], eyePos[1] + eyeDir[1], eyePos[2] + eyeDir[2],
	0);
    }
    
    // return matrix mode back to setting the view matrix
    mmode(MVIEWING);
  } else {                     // use (0 .. 1, 0 ..1) window for 2D view
    ortho2(0.0, 1.0, 0.0, 1.0);
  }
}

/////////////////////////  public virtual routines  

//
// virtual routines for preparing to draw, drawing, and finishing drawing
//

// reshape the display after a shape change
void GLDisplayDevice::reshape(void) {

  MSGDEBUG(2,"GLDisplayDevice: reshaping display." << sendmsg);

#ifdef GD_STEREO_IN_WINDOW
  if(getgdesc(GD_STEREO_IN_WINDOW) < 1 || getgdesc(GD_XPMAX) > 1200) {
    if(inStereo == GL_STEREO_CRYSTAL)
      return;		// now allowed to reshape crystal-eyes mode
  }
#else
  if(inStereo == GL_STEREO_CRYSTAL)
    return;		// now allowed to reshape crystal-eyes mode
#endif

  if(winget() != windowID)  winset(windowID);
  reshapeviewport();
  getsize(&xSize, &ySize);
  getorigin(&xOrig, &yOrig);
  if(inStereo != GL_STEREO_SIDE) {
    set_screen_pos((float)xSize / (float)ySize);
  } else {
    set_screen_pos(0.5 * (float)xSize / (float)ySize);
  }
}

// clear the display
void GLDisplayDevice::clear(void) {
  unsigned long l;
  
  MSGDEBUG(3, "GLDisplayDevice: clearing display." << sendmsg);

  // set viewport properly
  if(inStereo == GL_STEREO_SIDE) {
    viewport(0, (short)xSize, 0, (short)ySize);
  } else if(inStereo == GL_STEREO_CRYSTAL) {
#ifndef GD_STEREO_IN_WINDOW
    viewport(0,(short)getgdesc(GD_XPMAX),0,(short)getgdesc(GD_YPMAX));
#else
    if(getgdesc(GD_STEREO_IN_WINDOW) != 1 || getgdesc(GD_XPMAX) > 1200)
      viewport(0,(short)getgdesc(GD_XPMAX),0,(short)getgdesc(GD_YPMAX));
#endif
  }

  l = ( ( (short)(backColor[0] * 255.0) ) |
  	( (short)(backColor[1] * 255.0) << 8) |
  	( (short)(backColor[2] * 255.0) << 16) );
  czclear(l, getgdesc(GD_ZMAX));
}

// prepare to draw a 2D image
void GLDisplayDevice::prepare2D(int do_clear) {

  MSGDEBUG(3, "GLDisplayDevice: preparing to draw 2D." << sendmsg);

  if(winget() != windowID)  winset(windowID);
  Dim = 2;
  if(do_clear)
    clear();
  else
    zclear();
}

// prepare to draw a 3D image
void GLDisplayDevice::prepare3D(int do_clear) {

  MSGDEBUG(3, "GLDisplayDevice: preparing to draw 3D." << sendmsg);

  if(winget() != windowID)  winset(windowID);
  Dim = 3;
  if(do_clear)
    clear();
  else
    zclear();
}

// set up for normal (non-stereo) drawing.  Sets the viewport and perspective.
void GLDisplayDevice::normal(void) {
  viewport(0, (short)xSize, 0, (short)ySize);
  set_persp();
}

// set up for drawing the left eye image.  Assume always the left eye is
// drawn first (so no zclear is needed before it)
void GLDisplayDevice::left(void) {
  if(inStereo == GL_STEREO_SIDE) {
    viewport(0, (short)(xSize / 2), 0, (short)ySize);
    set_persp(RIGHTEYE);	// backwards since we're crossing our eyes
  } else if(inStereo == GL_STEREO_CRYSTAL) {
#ifndef GD_STEREO_IN_WINDOW
    viewport(0,getgdesc(GD_XPMAX),YOFFSET,YOFFSET + YMAXSTEREO);
#else
    if(getgdesc(GD_STEREO_IN_WINDOW) == 1 && getgdesc(GD_XPMAX) < 1200) {
//      zclear();
      leftbuffer(TRUE);
      rightbuffer(FALSE);
    } else {
      viewport(0,(short)getgdesc(GD_XPMAX),YOFFSET,YOFFSET + YMAXSTEREO);
    }
#endif
    set_persp(LEFTEYE);
  } else {			// left called even though we're non-stereo
    normal();
  }
}

// set up for drawing the right eye image.  Assume always the right eye is
// drawn last (so a zclear IS needed before it)
void GLDisplayDevice::right(void) {
  if(inStereo == GL_STEREO_SIDE) {
    viewport((short)(xSize / 2), (short)xSize, 0, (short)ySize);
    set_persp(LEFTEYE);		// backwards since we're crossing our eyes
  } else if(inStereo == GL_STEREO_CRYSTAL) {
#ifndef GD_STEREO_IN_WINDOW
    viewport(0,getgdesc(GD_XPMAX),0,YMAXSTEREO);
#else
    if(getgdesc(GD_STEREO_IN_WINDOW) == 1 && getgdesc(GD_XPMAX) < 1200) {
      zclear();
      leftbuffer(FALSE);
      rightbuffer(TRUE);
    } else {
      viewport(0,(short)getgdesc(GD_XPMAX),0,YMAXSTEREO);
    }
#endif
    set_persp(RIGHTEYE);
  } else {			// right called even though we're non-stereo
    normal();
  }
}

// update after drawing
void GLDisplayDevice::update(int do_update) {

  MSGDEBUG(3, "GLDisplayDevice: updating after drawing." << sendmsg);

  if(do_update)
    swapbuffers();

#ifdef GD_STEREO_IN_WINDOW
  if(getgdesc(GD_STEREO_IN_WINDOW) == 1 && getgdesc(GD_XPMAX) < 1200) {
    leftbuffer(TRUE);
    rightbuffer(TRUE);
  }
#endif
}

//
// stereo
//

// change to a different stereo mode (0 means 'off')
void GLDisplayDevice::set_stereo_mode(int newMode) {
  if(newMode == GL_STEREO_OFF && inStereo != newMode) {
    if(!inStereo)  return;
    
    if(inStereo == GL_STEREO_CRYSTAL) {
#ifdef GD_STEREO_IN_WINDOW
      if(getgdesc(GD_STEREO_IN_WINDOW) < 1 || getgdesc(GD_XPMAX) > 1200) {
#endif
        system("offstereo");

        // restore the window to the size it was before being set to stereo
        if(winget() != windowID)  winset(windowID);
        winposition(xOrig, xOrig + xSize, yOrig, yOrig + ySize);
        winconstraints();
#ifdef GD_STEREO_IN_WINDOW
      }
#endif
    }

    inStereo = newMode;
    reshape();
    normal();
    clear();
    update();

    MSGDEBUG(2,"GLDisplayDevice stereo turned off." << sendmsg);

  } else if(newMode == GL_STEREO_SIDE && inStereo != newMode) {
    set_stereo_mode(GL_STEREO_OFF);
    inStereo = newMode;
    reshape();
    normal();
    MSGDEBUG(2,"GLDisplayDevice set to side-by-side stereo." << sendmsg);

  } else if(newMode == GL_STEREO_CRYSTAL && inStereo != newMode) {
    set_stereo_mode(GL_STEREO_OFF);
#ifdef GD_STEREO_IN_WINDOW
    if(getgdesc(GD_STEREO_IN_WINDOW) == 1 && getgdesc(GD_XPMAX) < 1200) {
      if(!getgconfig(GC_DOUBLE) || !getgconfig(GC_STEREO)) {
        msgErr << "Doublebuffer stereo not available." << sendmsg;
        return;
      }
    }
#endif

    inStereo = newMode;

#ifdef GD_STEREO_IN_WINDOW
    if(getgdesc(GD_STEREO_IN_WINDOW) < 1 || getgdesc(GD_XPMAX) > 1200) {
#endif
      // full-screen (not in-a-window) stereo
      if(winget() != windowID)  winset(windowID);

      // execute the SGI program to put the monitor in stereo mode
      system("onstereo");

      // resize the window to be the whole screen
      prefposition(0,getgdesc(GD_XPMAX),0,getgdesc(GD_YPMAX));
      winconstraints();
      reshapeviewport();
      viewport(0,(short)getgdesc(GD_XPMAX),0,(short)getgdesc(GD_YPMAX));
      set_screen_pos(0.5*(float)getgdesc(GD_XPMAX)/(float)YMAXSTEREO);
      clear();
      update();

#ifdef GD_STEREO_IN_WINDOW
    }
#endif

    MSGDEBUG(2,"GLDisplayDevice set to crystal-eyes stereo." << sendmsg);
  }
}
