/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: GLMouse.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 1995/03/04 05:13:35 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The GL-Specific Mouse object; this uses the GL library to access the
 * state of the mouse, and uses a GL pop-up menu accessed by pressing the
 * third button for the Mouse UI commands.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: GLMouse.C,v $
 * Revision 1.4  1995/03/04  05:13:35  billh
 * Added more keyboard controls; changed so that pick mode 1
 * is used when the left button is clicked, and pick mode 2 is
 * used when the middle button is clicked.
 *
 * Revision 1.3  1995/02/22  03:57:58  billh
 * Now supports picking objects with a 2D or 3D pointer.
 * Issues CmdPick commands; adds a submenu for the PickModes, similar to
 * changing to rotate or translate modes.
 * Has a current pick mode; setes mouse mode via keyboard commands as well.
 *
 * Revision 1.2  94/11/27  07:57:27  billh
 * Changed to have separate menu on/menu off selections.
 * 
 * Revision 1.1  1994/10/28  18:34:46  billh
 * Initial revision
 *
 * Revision 1.5  1994/10/19  20:22:20  billh
 * Added pop-up menu commands to turn forms on and off.
 *
 * Revision 1.4  1994/10/05  05:33:01  billh
 * arguments changed to allow to compile on HP's.
 *
 * Revision 1.3  1994/09/12  20:53:07  billh
 * Added 'view reset' command.
 *
 * Revision 1.2  94/08/26  00:01:05  billh
 * Improved pop-up menu; now produces submenus to attach mouse to lights,
 * set the stereo mode, toggle lights on/off, and a couple other utilities.
 * 
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/GLMouse.C,v 1.4 1995/03/04 05:13:35 billh Exp $";
#endif

#include "GLMouse.h"
#include "Global.h"
#include "Inform.h"
#include "Scene.h"
#include "DisplayDevice.h"
#include "LightList.h"
#include "UIList.h"
#include "CmdTrans.h"
#include "CmdDisplay.h"
#include "CmdMenu.h"
#include "CmdPick.h"
#include "CmdUtil.h"
#include "CmdAnimate.h"
#include "Animation.h"
#include "MolAction.h"


// bitmap for different cursors
// translation cursor
unsigned short transCursor[32] = { 0x0000, 0x0080, 0x01C0, 0x03E0,
                                   0x0080, 0x0080, 0x1084, 0x3086,
                                   0x7FFF, 0x3086, 0x1804, 0x0080,
                                   0x0080, 0x03E0, 0x01C0, 0x0080,

                                   0x0000, 0x0140, 0x0220, 0x0410,
                                   0x0770, 0x194C, 0x294A, 0x4F79,
                                   0x0000, 0x4F79, 0x294A, 0x194C,
                                   0x0770, 0x0410, 0x0220, 0x0140 };

// scaling cursor
unsigned short scaleCursor[32] = { 0x0000, 0x000E, 0x0006, 0x000A,
                                   0x0010, 0x0020, 0x00E0, 0x01C0,
                                   0x0790, 0x4F00, 0x7F00, 0x7E00,
                                   0x7C00, 0x7C00, 0x7E00, 0x0000,

                                   0x001F, 0x0011, 0x0019, 0x0035,
                                   0x006F, 0x01D8, 0x0310, 0x0E30,
                                   0xF860, 0xB0C0, 0x8080, 0x8180,
                                   0x8300, 0x8300, 0x8100, 0xFF00 };

// picking cursor
unsigned short pickCursor[32] = { 0x0180, 0x0180, 0x0180, 0x0180,
                                  0x0180, 0x0180, 0x0180, 0xFFFF,
                                  0xFFFF, 0x0180, 0x0180, 0x0180,
                                  0x0180, 0x0180, 0x0180, 0x0180,

                                  0x0000, 0x0000, 0x0000, 0x0000,
                                  0x0000, 0x0000, 0x0000, 0x0000,
                                  0x0000, 0x0000, 0x0000, 0x0000,
                                  0x0000, 0x0000, 0x0000, 0x0000 };

// commands we are interested in
static int numCmds = 4;
static int cmdList[4] = { Command::WIN_EVENT, Command::DISP_STEREO,
	Command::DISP_LIGHT_ON, Command::MENU_SHOW };


//////////////////////////// constructor  ////////////////////////////
GLMouse::GLMouse(UIList *uil, CommandQueue *cq) : Mouse(uil, cq) {

  // record which commands we want
  for(int i=0; i < numCmds; command_wanted(cmdList[i++]));
  
  // define the cursors
#ifdef __NPGL__
  curstype(C16X1);
  defcursor(1,(unsigned short *)transCursor);
  defcursor(2,(unsigned short *)scaleCursor);
  defcursor(3,(unsigned short *)pickCursor);
#else
  curstype(C16X2);
  defcursor(1,(Cursor)transCursor);
  defcursor(2,(Cursor)scaleCursor);
  defcursor(3,(Cursor)pickCursor);
#endif
  curorigin(1,8,8);
  curorigin(2,0,15);
  curorigin(3,8,8);

  // initialize pop-up menu values
  popMain = popStereo = popMouseMode = popLight = popLightToggle = (-1);
  popPickMode = popMenuList = popoffMenuList = popAnimate = (-1);
  freeMain= freeStereo= freeMouseMode= freeLight= freeLightToggle= FALSE;
  freePickMode= freeMenuList= freeAnimate = FALSE;
  
  // set the cursor mode
  setcursor(0,0,0);
 
  // do reset
  reset();
}


// destructor
GLMouse::~GLMouse(void) {
  // nothing here
}


////////////////////////////  private routines  /////////////////////////

// routine to create the pop-up menu
void GLMouse::create_popup_menu(void) {
  int i;
  char mbuf[64];
  MSGDEBUG(2,"GLMouse: creating pop-up menu ..." << sendmsg);

  // free old menus, if necessary
  if(freeMain)		freepup(popLight);
  if(freeMouseMode)	freepup(popMouseMode);
  if(freePickMode)	freepup(popPickMode);
  if(freeStereo)	freepup(popStereo);
  if(freeMain)		freepup(popMain);
  if(freeAnimate)	freepup(popAnimate);
  if(freeMenuList) {
    freepup(popMenuList);
    freepup(popoffMenuList);
  }
  
  // create submenus, and the higher-level menus
  popStereo = newpup();		// stereo modes, 31 ... 40
  freeStereo = TRUE;
  for(i=0; i < display->num_stereo_modes(); i++) {
    sprintf(mbuf,"%s%%x%2d", display->stereo_name(i), 31+i);
    addtopup(popStereo, mbuf, 0);
  }
  setpup(popStereo, display->stereo_mode() + 1, PUP_CHECK);
  
  popAnimate = newpup();	// animation modes, 51 ... 60
  freeAnimate = TRUE;
  sprintf(mbuf,"%s%%x%2d", "play forward", 51);
  addtopup(popAnimate, mbuf, 0);
  sprintf(mbuf,"%s%%x%2d", "play reverse", 52);
  addtopup(popAnimate, mbuf, 0);
  sprintf(mbuf,"%s%%x%2d", "step 1 forward", 53);
  addtopup(popAnimate, mbuf, 0);
  sprintf(mbuf,"%s%%x%2d", "step 1 reverse", 54);
  addtopup(popAnimate, mbuf, 0);
  sprintf(mbuf,"%s%%x%2d", "pause", 55);
  addtopup(popAnimate, mbuf, 0);
  sprintf(mbuf,"%s%%x%2d", "goto start", 56);
  addtopup(popAnimate, mbuf, 0);
  sprintf(mbuf,"%s%%x%2d", "goto end", 57);
  addtopup(popAnimate, mbuf, 0);

  popLightToggle = newpup();	// toggle lights, 21 ... 30
  freeLightToggle = TRUE;
  popLight = newpup();		// move lights, 141 ... 150
  freeLight = TRUE;
  lights->reset();
  i = 0;
  while(lights->is_current()) {
    sprintf(mbuf,"Light %1d%%x%3d", i, 141 + i);
    addtopup(popLight, mbuf, 0);
    sprintf(mbuf,"Light %1d%%x%2d", i, 21 + i);
    addtopup(popLightToggle, mbuf, 0);
    setpup(popLightToggle, i + 1,
    	((lights->current())->light_displayed() ? PUP_CHECK : PUP_BOX) );
    lights->next();
    i++;
  }
  if(curr_move_mode() == LIGHT)
    setpup(popLight, moveObj + 1, PUP_CHECK);

  popPickMode = newpup();	// pick mode, 301 ... 400
  freePickMode = TRUE;
  if(scene) {
    for(i=0; i < scene->num_pick_modes(); i++) {
      sprintf(mbuf,"%s%%x%2d",scene->pick_mode_name(i), 301+i);
      addtopup(popPickMode, mbuf, 0);
    }
  }
  if(curr_pick_mode() >= 0)
    setpup(popPickMode, curr_pick_mode() + 1, PUP_CHECK);

  popMouseMode = newpup();	// mouse modes, 11 ... 20
  freeMouseMode = TRUE;
  sprintf(mbuf, "%s%%x%2d", move_mode_name(ROTATION), 11 + ROTATION);
  addtopup(popMouseMode, mbuf, 0);
  sprintf(mbuf, "%s%%x%2d", move_mode_name(TRANSLATION), 11 + TRANSLATION);
  addtopup(popMouseMode, mbuf, 0);
  sprintf(mbuf, "%s%%x%2d", move_mode_name(SCALING), 11 + SCALING);
  addtopup(popMouseMode, mbuf, 0);
  sprintf(mbuf,"%s%%x%2d%%m", move_mode_name(LIGHT), 11 + LIGHT);
  addtopup(popMouseMode, mbuf, popLight);
  sprintf(mbuf,"%s%%x%2d%%m", move_mode_name(PICKING), 11 + PICKING);
  addtopup(popMouseMode, mbuf, popPickMode);
  setpup(popMouseMode, curr_move_mode() + 1, PUP_CHECK );

  popMenuList = newpup();	// list of menus, 91 ... 140
  popoffMenuList = newpup();	// list of menus, 191 ... 300
  freeMenuList = TRUE;
  int m_added = 0;
  for (int mn = 0; mn < uiList->num(); mn++) {
    UIObject *menu = uiList->item(mn);
    if(menu->is_menu()) {
      ++m_added;
      sprintf(mbuf, "%s%%x%3d", menu->name, 91 + menu->id());
      addtopup(popMenuList, mbuf, 0);
      setpup(popMenuList, m_added, (menu->is_on() ? PUP_CHECK : PUP_BOX));

      sprintf(mbuf, "%s%%x%3d", menu->name, 191 + menu->id());
      addtopup(popoffMenuList, mbuf, 0);
      setpup(popoffMenuList, m_added, (menu->is_on() ? PUP_CHECK : PUP_BOX));
    }
  }

  popMain = newpup();
  freeMain = TRUE;
  addtopup(popMain, "Mouse Mode%m", popMouseMode);
  addtopup(popMain, "Toggle Light%m", popLightToggle);
  addtopup(popMain, "Stereo Mode%m%l", popStereo);
  addtopup(popMain, "Stop Rotation", 0);
  addtopup(popMain, "Reset View%l", 0);
  addtopup(popMain, "Animation%m%l", popAnimate);
  addtopup(popMain, "Show Form%m", popMenuList);
  addtopup(popMain, "Hide Form%m%l", popoffMenuList);
  addtopup(popMain, "Quit",0);
}


////////////////////////////  public routines  /////////////////////////

// set the mouse move mode to the given state; return success
int GLMouse::move_mode(MoveMode mm, int mobj) {

  // set mouse mouse in base class routine first, and see if successful
  if(Mouse::move_mode(mm, mobj)) {
  
    // command was successful ... change the cursor style
    if(mm == LIGHT)
      setcursor(0, 0, 0);
    else if(mm == PICKING)
      setcursor(3, 0, 0);
    else
      setcursor(mm, 0, 0);

    create_popup_menu();		// redo menus after setting mode   
    return TRUE;
  } else
    return FALSE;
}


// set the pick mode for this mouse (does things like change cursor style)
// return success.
int GLMouse::pick_mode(int n) {

  // set pick mode in base case routine first, and see if successful
  if(Mouse::pick_mode(n)) {
    // for now, just need to reconstruct menus.  Later need to change cursor
    // style
    create_popup_menu();
    return TRUE;
  } else
    return FALSE;
}


////////////////// virtual routines for UI init/display  /////////////////////
   
// reset the mouse to original settings
void GLMouse::reset(void) {
  // get the current x or y position of the mouse
  oldX = currX = (int)getvaluator(MOUSEX);
  oldY = currY = (int)getvaluator(MOUSEY);

  // get the current button states
  b1Down = (int)getbutton(LEFTMOUSE);
  b2Down = (int)getbutton(MIDDLEMOUSE);

  Mouse::reset();		// do base-class reset
  create_popup_menu();		// redo the popup menus
}
  

///////////////// virtual routines for entering/processing commands and events

// update the display due to a command being executed.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int GLMouse::act_on_command(int type, Command *cmd, int) {
  int dev, val;
  
  MSGDEBUG(3,"GLMouse: acting on command " << type << sendmsg);

  // check if GL event
  if(type == Command::DISP_STEREO || type == Command::DISP_LIGHT_ON ||
	type == Command::MENU_SHOW) {
    // status changed; redo menus
    create_popup_menu();

  } else if(type == Command::WIN_EVENT) {
    dev = ((WinEvent *)cmd)->eventCode;
    val = ((WinEvent *)cmd)->val;

    if (dev == WinEvent::WIN_REDRAW) {
      addcommand(new CmdDisplayReshape(id()));

/*
    } else if(dev == WinEvent::WIN_MOUSEX) {		// mouse moved in x-dir
      mouse_moved(val, y());

    } else if(dev == WinEvent::WIN_MOUSEY) {		// mouse moved in y-dir
      mouse_moved(x(), val);
*/

    } else if(dev == WinEvent::WIN_LEFT) {
      if(val != b1Down) {	// mouse button state changed
        // check for whether the state is changing to up or to down
        if(val) {		// button pressed down
	  b1Moved = FALSE;
          oldX = currX = (int)getvaluator(MOUSEX);
          oldY = currY = (int)getvaluator(MOUSEY);
	  xRotVel = yRotVel = 0.0;

	  // if picking, but not active pick yet, generate a pick command.
	  if(curr_move_mode() == PICKING) {
	    if(picking() == 0) {
	      pickInProgress = 1 + B_LEFT;
	      float mx = (float) currX;
	      float my = (float) currY;
	      display->rel_screen_pos(mx, my);
	      addcommand(new CmdPickStart(B_LEFT,curr_pick_mode(),
	      			mx,my,id()));
	    }
	  }
	} else {		// button released
	  if(picking() == (1 + B_LEFT)) {	// finished picking object
	    pickInProgress = 0;
	    float mx = (float) getvaluator(MOUSEX);
	    float my = (float) getvaluator(MOUSEY);
	    oldX = currX = (int) mx;
	    oldY = currY = (int) my;
	    display->rel_screen_pos(mx, my);
	    addcommand(new CmdPickEnd(B_LEFT,curr_pick_mode(),
	      			mx,my,id()));
	  } else {
	    // if the button was pressed and released, but the mouse did not
	    // move, act as if a pick was done with mode 1
	    if(!b1Moved) {
	      float mx = (float) x();
	      float my = (float) y();
	      display->rel_screen_pos(mx, my);
	      addcommand(new CmdPickStart(B_LEFT, 1, mx, my, id()));
	      addcommand(new CmdPickEnd(B_LEFT, 1, mx, my, id()));
	    }
	  }
	}
        // change settings for the mouse button
        b1Down = (int)val;
      }

    } else if(dev == WinEvent::WIN_MIDDLE) {
      if(val != b2Down) {	// mouse button state changed
        // check for whether the state is changing to up or to down
        if(val) {		// button pressed down
	  b2Moved = FALSE;
          oldX = currX = (int)getvaluator(MOUSEX);
          oldY = currY = (int)getvaluator(MOUSEY);
	  zRotVel = 0.0;

	  // if picking, but not active pick yet, generate a pick command.
	  if(curr_move_mode() == PICKING) {
	    if(picking() == 0) {
	      pickInProgress = 1 + B_MIDDLE;
	      float mx = (float) currX;
	      float my = (float) currY;
	      display->rel_screen_pos(mx, my);
	      addcommand(new CmdPickStart(B_MIDDLE,curr_pick_mode(),
	      			mx,my,id()));
	    }
	  }
	} else {		// button released
	  if(picking() == (1 + B_MIDDLE)) {	// finished picking object
	    pickInProgress = 0;
	    float mx = (float) getvaluator(MOUSEX);
	    float my = (float) getvaluator(MOUSEY);
	    oldX = currX = (int) mx;
	    oldY = currY = (int) my;
	    display->rel_screen_pos(mx, my);
	    addcommand(new CmdPickEnd(B_MIDDLE,curr_pick_mode(),
	      			mx,my,id()));
	  } else {
	    // if the button was pressed and released, but the mouse did not
	    // move, act as if a pick was done with mode 2
	    if(!b2Moved) {
	      float mx = (float) x();
	      float my = (float) y();
	      display->rel_screen_pos(mx, my);
	      addcommand(new CmdPickStart(B_MIDDLE, 2, mx, my, id()));
	      addcommand(new CmdPickEnd(B_MIDDLE, 2, mx, my, id()));
	    }
	  }
	}
        // change settings for the mouse button
        b2Down = (int)val;
      }

    } else if(dev == WinEvent::WIN_RIGHT) {		// pop-up menu
      if (val == 1) {
        long menuval = dopup(popMain);
        MSGDEBUG(2,"Pop-up menu value returned: " << menuval << sendmsg);

        if (menuval >= 141 && menuval <= 150) {		// light mode ss-menu
	  move_mode(LIGHT, (int)menuval - 141);

	} else if(menuval >= 301 && menuval < 401) {	// set pick mode
	  move_mode(PICKING, (int)menuval - 301);

	} else if(menuval >= 91 && menuval < 141) {
	  // toggle on given menu
          int m = (int)menuval - 91;
          addcommand(new CmdMenuShow(m, TRUE, id()));

	} else if(menuval >= 191 && menuval < 241) {
	  // toggle off given menu
          int m = (int)menuval - 191;
          addcommand(new CmdMenuShow(m, FALSE, id()));

	} else if(menuval >= 21 && menuval <= 30) {	// light toggle s-menu
	  addcommand(new CmdDisplayLightOn((int)menuval - 21,
	  	! (lights->item((int)menuval - 21))->light_displayed(), id()));

	} else if(menuval >= 31 && menuval <= 40) {	// stereo mode s-menu
	  addcommand(new CmdDisplayStereo((int)menuval - 31, id()));

	} else if(menuval >= 51 && menuval <= 55) {	// animation control
	  Animation::AnimDir newdir;
	  switch(menuval) {
	    case 51: newdir = Animation::FORWARD; break;
	    case 52: newdir = Animation::REVERSE; break;
	    case 53: newdir = Animation::FORWARD1; break;
	    case 54: newdir = Animation::REVERSE1; break;
	    case 55: newdir = Animation::PAUSE; break;
          }
          addcommand(new CmdAnimDir(newdir, id()));

	} else if(menuval >= 56 && menuval <= 57) {	// animation control
	  int newpos = (menuval == 56 ? SET_FRAME_START : SET_FRAME_END);
	  addcommand(new CmdAnimJump(newpos, id()));

	} else if(menuval == 11 + ROTATION) {		// mouse mode s-menu
          move_mode(ROTATION);
        } else if (menuval == 11 + TRANSLATION) {
          move_mode(TRANSLATION);
        } else if (menuval == 11 + SCALING) {
          move_mode(SCALING);

	} else if (menuval == 4) {			// stop rotation
	  addcommand(new CmdStopRot(id()));

	} else if (menuval == 5) {			// stop rotation
          addcommand(new CmdResetView(id()));

	} else if (menuval == 9) {			// quit program
          addcommand(new CmdQuit(TRUE,id()));
	}
      }

    } else if (dev == WinEvent::WIN_INPUTCHANGE) {
      ;		// who cares, just return

    } else if(dev == WinEvent::WIN_KEYBD) {		// keyboard pressed
      if(val == 'R' || val == 'r')
        move_mode(ROTATION);
      else if(val == 'T' || val == 't')
        move_mode(TRANSLATION);
      else if(val == 'S' || val == 's')
        move_mode(SCALING);
      else if(val >= '0' && val <= '9')
        move_mode(PICKING, 0 + val - '0');
      else if(val == ' ' || val == 'f' || val == 'F')
        addcommand(new CmdAnimDir(Animation::FORWARD1, id()));
      else if(val == 'b' || val == 'B')
        addcommand(new CmdAnimDir(Animation::REVERSE1, id()));
      else if(val == ',' || val == '<')
        addcommand(new CmdAnimDir(Animation::REVERSE, id()));
      else if(val == '.' || val == '>')
        addcommand(new CmdAnimDir(Animation::FORWARD, id()));
      else if(val == '/' || val == '?')
        addcommand(new CmdAnimDir(Animation::PAUSE, id()));

    } else			// unrecognized device
      return FALSE;

  } else {			// unrecognized command
    return FALSE;
  }
  
  // if here, command we can use was found
  return TRUE;
}



// do action when the mouse is moved
void GLMouse::mouse_moved(int newx, int newy) {
  int dx, dy, mouseMoved;

  MSGDEBUG(3,"GLMouse: mouse moved to " << newx << ", " << newy);
  MSGDEBUG(3," ... " << sendmsg);

  // get new mouse positions if any button is currently 'down'
  if(b1Down || b2Down) {
    // get current mouse position
    currX = newx;
    currY = newy;
    // and calculate distance mouse has moved
    dx = 5 * (currX - oldX);
    dy = -5 * (currY - oldY);
    // and then save the settings
    oldX = currX;
    oldY = currY;
  }
  
  mouseMoved = (dx != 0 || dy != 0);

  // check if we are picking something; if so, generate pick-move command
  if(mouseMoved && picking()) {
    float mx = (float) x();
    float my = (float) y();
    display->rel_screen_pos(mx, my);
    if(mx >= 0.0 && mx <= 1.0 && my >= 0.0 && my <= 1.0)
      addcommand(new CmdPickMove(picking()-1, curr_pick_mode(), mx, my, id()));
    return;
  }

  // Otherwise, if a button is pressed, check how far the mouse moved,
  // and transform the view accordingly.

  // check for button 1 action
  if(b1Down) {
    b1Moved = (b1Moved || mouseMoved);
    if(curr_move_mode() == ROTATION || curr_move_mode() == LIGHT) {
      xRotVel = rotInc * (float)dy;
      yRotVel = rotInc * (float)dx;
      if(curr_move_mode() == ROTATION) {	// rotate scene
	if(xRotVel != 0.0) {
          addcommand(new CmdRotate(xRotVel, 'x', CmdRotate::BY, id()));
          xRotVel *= RotVelScale;
	}
	if(yRotVel != 0.0) {
          addcommand(new CmdRotate(yRotVel, 'y', CmdRotate::BY, id()));
          yRotVel *= RotVelScale;
	}
      } else {			// rotate particular light
	if(xRotVel != 0.0) {
	  addcommand(new CmdDisplayLightRot(moveObj, xRotVel, 'x', id()));
          xRotVel *= RotVelScale;
	}
	if(yRotVel != 0.0) {
	  addcommand(new CmdDisplayLightRot(moveObj, yRotVel, 'y', id()));
          yRotVel *= RotVelScale;
	}
      }

    } else if(curr_move_mode() == TRANSLATION && mouseMoved) {
      addcommand(new CmdTranslate(transInc*(float)dx, -transInc*(float)dy,
        0.0, CmdTranslate::BY, id()));

    } else if(curr_move_mode() == SCALING && dx != 0) {
      float scf = scaling + scaleInc * (float)dx;
      if(scf < 0.0)
        scf = 0.0;
      addcommand(new CmdScale(scf, CmdScale::BY, id()));
    }
  }
  
  // check for button 2 action
  if(b2Down) {
    b2Moved = (b2Moved || mouseMoved);
    if(curr_move_mode() == ROTATION || curr_move_mode() == LIGHT) {
      zRotVel = rotInc * (float)dx;
      if(curr_move_mode() == ROTATION) {
	if(zRotVel != 0.0) {
          addcommand(new CmdRotate(zRotVel, 'z', CmdRotate::BY, id()));
          zRotVel *= RotVelScale;
	}
      } else {
	if(zRotVel != 0.0) {
	  addcommand(new CmdDisplayLightRot(moveObj, zRotVel, 'z', id()));
          zRotVel *= RotVelScale;
	}
      }

    } else if(curr_move_mode() == TRANSLATION && dx != 0) {
      addcommand(new CmdTranslate(0.0, 0.0, transInc*(float)dx,
	CmdTranslate::BY, id()));

    } else if(curr_move_mode() == SCALING && dx != 0) {
      float scf = scaling + 10.0 * scaleInc * (float)dx;
      if(scf < 0.0)
        scf = 0.0;
      addcommand(new CmdScale(scf, CmdScale::BY, id()));
    }
  }
}


// check for an event, and queue it if found.  Return TRUE if an event
// was generated.
int GLMouse::check_event(void) {

  // do action if a button is pressed and the mouse moves
  if(b1Down || b2Down) {
    mouse_moved((int)getvaluator(MOUSEX), (int)getvaluator(MOUSEY));
    return FALSE;
  }

  // If a button is not pressed, there may be some motion due to some constant
  // factor, like angular momentum or some earlier rotate command.

  // check for button 1 down; if not, and not picking, apply ang vel
  if(!b1Down && !picking()) {
    if(curr_move_mode() != LIGHT) {		// (possibly) rotate scene
      if(xRotVel != 0.0)
        addcommand(new CmdRotate(xRotVel, 'x', CmdRotate::BY, id()));
      if(yRotVel != 0.0)
        addcommand(new CmdRotate(yRotVel, 'y', CmdRotate::BY, id()));
    } else {				// (possibly) rotate particular light
      if(xRotVel != 0.0)
        addcommand(new CmdDisplayLightRot(moveObj, xRotVel, 'x', id()));
      if(yRotVel != 0.0)
        addcommand(new CmdDisplayLightRot(moveObj, yRotVel, 'y', id()));
    }
  }
  
  // check for button 1 down; if not, and not picking, apply ang vel
  if(!b2Down && !picking()) {
    if(curr_move_mode() != LIGHT) {		// (possibly) rotate scene
      if(zRotVel != 0.0)
        addcommand(new CmdRotate(zRotVel, 'z', CmdRotate::BY, id()));
    } else {				// (possibly) rotate particular light
      if(zRotVel != 0.0)
        addcommand(new CmdDisplayLightRot(moveObj, zRotVel, 'z', id()));
    }
  }
  
  // no events were generated
  return FALSE;
}
