/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: GeometryList.C,v $
 *      $Author: billh $        $Locker:  $                $State: Exp $
 *      $Revision: 1.1 $      $Date: 1995/03/04 05:33:08 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * This maintains a set of lists of geometry monitors, and draws them to
 * the scene.  This is a Displayable which keeps the graphical representations
 * of the geometry monitors; this is also a Pickable, and creates and controls
 * the PickMode objects which create new geometry monitors via the pointers.
 * This object keeps a set of ResizeArray, each of which is a 'category' for
 * geometry monitors (i.e Atoms, Bonds, etc.) which contain Geometry objects. 
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: GeometryList.C,v $
 * Revision 1.1  1995/03/04  05:33:08  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/GeometryList.C,v 1.1 1995/03/04 05:33:08 billh Exp $";
#endif

#include "GeometryList.h"
#include "DisplayDevice.h"
#include "Scene.h"
#include "DispCmds.h"
#include "ColorList.h"
#include "PickModeMolLabel.h"
#include "Inform.h"
#include "utilities.h"

// default colors to use
#define ATOMGEOMCOL	REGGREEN
#define BONDGEOMCOL     REGWHITE
#define ANGLEGEOMCOL	REGYELLOW
#define DIHEGEOMCOL	REGCYAN


////////////////////////  constructor  /////////////////////////
GeometryList::GeometryList(Scene *sc) : Displayable3D(MULT,"Geometries",sc,5) {

  // indicate we don't yet have a color object to use
  colorCat = (-1);

  // create default lists for atom, bond, angle, and dihedral measurements
  add_geom_list("Atoms", ATOMGEOMCOL);
  add_geom_list("Bonds", BONDGEOMCOL);
  add_geom_list("Angles", ANGLEGEOMCOL);
  add_geom_list("Dihedrals", DIHEGEOMCOL);

  // Displayable characteristics
  rot_off();
  scale_off();
  glob_trans_off();
  cent_trans_off();

  // register with the scene as a pickable item (even though we're not)
  register_with_picklist(sc);

  // create new pick modes to allow pointers to set up geometry monitors
  for(int m=0; m < 4; m++)
    add_pick_mode(geom_list_name(m), m);
}


////////////////////////  destructor  ////////////////////////////
GeometryList::~GeometryList(void) {

  // for all the lists, delete all geometry monitors
  for(int i=(num_lists() - 1); i >= 0; i--)
    del_geom_list(i);
}


///////////////////////////  protected virtual routines  ///////////////////

// do action when a new color list is provided
// This creates a color category for use by the Axes, with the colors for
// the components.  These colors can then be edited by the user.
void GeometryList::do_use_colors(void) {

  // add new category (or just get it's index if it exists)
  colorCat = colorList->add_color_category(this->name);
  
  // add components, and their default colors
  for(int i=(num_lists() - 1); i >= 0; i--)
    (colorList->color_category(colorCat))->add_name(geomLists.name(i),
			(geomLists.data(i))->defaultColor);
}


// do action due to the fact that a color for the given ColorList for
// the specified category has changed
void GeometryList::do_color_changed(ColorList *changelist, int clr) {
  if(changelist == colorList && clr == colorCat) {

    // color changed for us, recreate command list
    // SET FLAG TO REDRAW ITEMS HERE
  }
}


// create a new pick mode object for use as a separate pick mode to
// add geometry monitors via the mouse.
PickMode *GeometryList::create_pick_mode(int m) {

  // m is just a code used by this object to tell which one to create.
  return new PickModeMolLabel(geom_list_name(m), m+1);
}

///////////////////////////  public routines  ////////////////////////////

// add a new category: specify the name, and default color.  Return index of
// new list.
int GeometryList::add_geom_list(char *nm, int clr) {

  // make sure we do not already have a category with this name
  int oldlist = geom_list_index(nm);
  if(oldlist >= 0)
    return oldlist;
    
  // create a new struct
  GeomListStruct *newlist = new GeomListStruct;
  newlist->geomList = new ResizeArray<Geometry *>(8);
  newlist->defaultColor = clr;
  
  // add the new category to the big list
  return geomLists.add_name(nm, newlist);
}


// delete the Nth category.  Return success.
int GeometryList::del_geom_list(int n) {
  GeomListStruct *oldlist = NULL;

  // make sure we do have a category with this name
  if(n >= 0 && n < num_lists()) {
    // get data for Nth list
    oldlist = geomLists.data(n);
    GeomListPtr glist = oldlist->geomList;
    
    // go through the list and delete all current Geometry objects
    for(int i=(glist->num() - 1); i >= 0; i--)
      delete (*glist)[i];

    // delete the old list storage and structure
    delete glist;
    delete oldlist;
    geomLists.set_data(n, NULL);
  }

  // return whether we were successful
  return (oldlist != NULL);
}


// add a new geometry object to the list with the given index.  Return success.
int GeometryList::add_geometry(int glindex, Geometry *g) {

  // make sure the given Geometry object is OK
  if(g && g->ok() && glindex >= 0 && glindex < num_lists()) {
  
    // get the list of geometry objects for the given index
    GeomListPtr glist = geom_list(glindex);

    // if there is already an identical label in the list,
    // do not add this one, instead toggle the displayed
    // status of the old one and return FALSE
    for(int i=(glist->num() - 1); i >= 0; i--) {
      Geometry *g2 = (*glist)[i];
      if(!strcmp(g2->name(), g->name())) {
	// name matches; toggle status of orig Geometry
	g2->on( ! (g2->on()) );

	// also delete the given Geometry
	delete g;

	// signal we could not add the object
	return FALSE;
      }
    }

    // add the geometry object
    glist->append(g);
    
    // indicate we were successful
    return TRUE;
  }
  
  // if here, something did not work
  return FALSE;
}


// del a new geometry object from the list with the given index. Return success.
// if n < 0, delete all markers
int GeometryList::del_geometry(int glindex, int n) {

  // make sure the given Geometry object is OK
  if(glindex >= 0 && glindex < num_lists()) {
  
    // get the list of geometry objects for the given index
    GeomListPtr glist = geom_list(glindex);

    // make sure the geometry index is ok
    if(n >= 0 && n < glist->num())  {
      // delete and remove the geometry object
      delete (*glist)[n];
      glist->remove(n);
      
      // indicate we were successful
      return TRUE;
    } else if(n < 0) {
      // delete and remove all geometry objects
      for(int j=(glist->num() - 1); j >= 0; j--) {
        delete (*glist)[j];
	glist->remove(j);
      }
      
      // indicate we were successful
      return TRUE;
    }
  }
  
  // if here, something did not work
  return FALSE;
}

// toggle whether to show or hide a geometry monitor.  If the monitor 
// specified is < 0, does so for all monitors in the given category
int GeometryList::show_geometry(int glindex, int n, int s) {

  // make sure the given Geometry object is OK
  if(glindex >= 0 && glindex < num_lists()) {
  
    // get the list of geometry objects for the given index
    GeomListPtr glist = geom_list(glindex);

    // make sure the geometry index is ok
    if(n >= 0 && n < glist->num())  {
      // hide or show the specified object
      (*glist)[n] -> on(s);
      
      // indicate we were successful
      return TRUE;
    } else if(n < 0) {
      // delete and remove all geometry objects
      for(int j=(glist->num() - 1); j >= 0; j--)
        (*glist)[j] -> on(s);
      
      // indicate we were successful
      return TRUE;
    }
  }
  
  // if here, something did not work
  return FALSE;
}

///////////////////////  public virtual routines  ////////////////////////

// prepare for drawing ... do any updates needed right before draw.
// For now, this always recreates the display list
void GeometryList::prepare(DisplayDevice *) {
  register int i, j, c, geometries = 0;

  MSGDEBUG(2,"Geometries: creating command list ..." << sendmsg);

  reset_disp_list();

  // go through all the geometry objects, recalculate, and find out if
  // something is no longer 'ok'.  If not, delete it.
  for(i=(num_lists() - 1); i >= 0; i--) {
    GeomListPtr glist = geom_list(i);
    for(int j=(glist->num() - 1); j >= 0; j--) {
      Geometry *g = (*glist)[j];
      if(g->ok()) {
        g->calculate();
	geometries++;
      } else {
        del_geometry(i, j);
      }
    }
  }
  
  // if there are no geometries to display, we are done
  if(!geometries)
    return;

  // now, go on and draw all the geometry markers

  // set line style and width
  cmdMaterials.putdata(FALSE, this);
  cmdLineType.putdata(DASHEDLINE, this);
  cmdLineWidth.putdata(4, this);

  // draw all markers for all geometries
  for(i=(num_lists() - 1); i >= 0; i--) {
    // get geometry list pointer, and draw all markers
    GeomListPtr glist = geom_list(i);
    if(glist->num() > 0) {
      // set color for given list
      if(colorCat >= 0)
        c = (colorList->color_category(colorCat))->data(geom_list_name(i));
      else
        c = REGWHITE;
      cmdColor.putdata(REGCOLOR(c), this);

      // draw markers for all geometries which are on
      for(j=(glist->num() - 1); j >= 0; j--)
        if(((*glist)[j])->on())
          ((*glist)[j])->draw_marker(this);
    }
  }
}
