/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: TrackerFormsObj.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $	$Date: 1995/02/26 23:10:49 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The on-screen menu to control the spatial trackers and 3D pointers (tools).
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: TrackerFormsObj.C,v $
 * Revision 1.2  1995/02/26  23:10:49  billh
 * Fixed a couple small bugs.
 *
 * Revision 1.1  1995/02/26  22:22:57  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/TrackerFormsObj.C,v 1.2 1995/02/26 23:10:49 billh Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include "TrackerFormsObj.h"
#include "CmdTracker.h"
#include "TrackerList.h"
#include "Tracker.h"
#include "CmdTool.h"
#include "ToolControl.h"
#include "GrabTool.h"
#include "UIVR.h"
#include "Global.h"

// commands we are interested in
static int numCmds = 8;
static int cmdList[8] = { Command::TRACKER_START,
	Command::TRACKER_PAUSE,		Command::TRACKER_UNPAUSE,
	Command::TOOL_SIZE,		Command::TOOL_OFFSET,
	Command::TOOL_PUSH,		Command::TOOL_POP,
	Command::TOOL_REPLACE };


// default initial editing mode
#define DEFAULT_TR_EDIT_MODE		0


// lists of objects which make up the different menu modes
static FL_OBJECT **tracker_objs[] = { &tracker_start, &tracker_avail_list, 
	&tracker_running_list, &tracker_pause, &tracker_unpause,
	&tracker_box_1, &tracker_box_2 };

static FL_OBJECT **tool_objs[] = { &tool_list, &tool_type, &tool_length,
	&tool_scale, &tool_detail, &tool_offset_label, &tool_x_offset,
	&tool_y_offset, &tool_z_offset, &tool_box };

static FL_OBJECT ***tracker_menu_objs[] = { tracker_objs, tool_objs };

static int tracker_menu_obj_size[] = {
	sizeof(tracker_objs) / sizeof(FL_OBJECT **),
	sizeof(tool_objs) / sizeof(FL_OBJECT **) };

static FL_OBJECT **tracker_menu_mode_buttons[] = {
	&tracker_mode_trackers, &tracker_mode_tools };
	

/////////////////////////  constructor  
TrackerFormsObj::TrackerFormsObj(UIList *uil, CommandQueue *cq,
	int sh,int bd,int *npos) : FormsObj("tracker", uil, cq, bd, npos) {
	
  // set values of common menu elements
  form = trackerMenu;
  lightButton = tracker_menu_button;
  offButton = tracker_menu_off;
  
  // register which commands we are interested in
  for(int i=0; i < numCmds; command_wanted(cmdList[i++]));
  
  // turn on if required
  if(sh)
    On();
}


/////////////////////////  private routines  

// determine which editing mode has been selected by checking if the given
// FL_OBJECT is one of the editing-changing buttons.  Return index if
// the object is a mode, or (-1) otherwise.
int TrackerFormsObj::find_editing_mode(FL_OBJECT *obj) {
  register int i;
  
  for(i=(sizeof(tracker_menu_objs) / sizeof(FL_OBJECT ***)) - 1; i >= 0; i--) {
    if(obj == *(tracker_menu_mode_buttons[i]))
      break;
  }
  
  return i;
}


// set the tracker/tool editing mode to the given value
void TrackerFormsObj::set_editing_mode(int editMode) {
  register int i, j;

  if(editMode < 0 ||
  	editMode >= (sizeof(tracker_menu_objs) / sizeof(FL_OBJECT ***)))
    return;

  freeze();      
    for(j=0; j < sizeof(tracker_menu_objs) / sizeof(FL_OBJECT ***); j++) {
      fl_set_button(*(tracker_menu_mode_buttons[j]), j == editMode);
      if(j == editMode) {
        for(i=0; i < tracker_menu_obj_size[j]; i++)
          fl_show_object(*(tracker_menu_objs[j][i]));
      } else {
        for(i=0; i < tracker_menu_obj_size[j]; i++)
          fl_hide_object(*(tracker_menu_objs[j][i]));
      }
    }
  unfreeze();
}



// fill the available tracker browser
void TrackerFormsObj::reset_avail_tracker_browser(void) {
  register int i;
  char trbuf[128];
  
  freeze();
    int currline = fl_get_browser(tracker_avail_list);
    fl_clear_browser(tracker_avail_list);
    for(i=0; i < trackerList -> num_avail(); i++) {
      sprintf(trbuf,"%2d. %s",i+1, trackerList->avail_name(i));
      fl_add_browser_line(tracker_avail_list, trbuf);
    }
    if(currline > 0)
      fl_select_browser_line(tracker_avail_list, currline);
    else if(trackerList -> num_avail() > 0)
      fl_select_browser_line(tracker_avail_list, 1);
  unfreeze();
}


// fill the running tracker browser
void TrackerFormsObj::reset_running_tracker_browser(void) {
  register int i;
  char trbuf[128];
  
  freeze();
    int currline = fl_get_browser(tracker_running_list);
    fl_clear_browser(tracker_running_list);
    for(i=0; i < trackerList -> num(); i++) {
      sprintf(trbuf,"%1d. %s %s",i+1, trackerList->name(i),
      	(trackerList->tracker(i)->paused() ? "(paused)" : ""));
      fl_add_browser_line(tracker_running_list, trbuf);
    }
    if(currline > 0)
      fl_select_browser_line(tracker_running_list, currline);
    else if(trackerList -> num() > 0)
      fl_select_browser_line(tracker_running_list, trackerList -> num() + 1);
  unfreeze();
}


// fill the tool browser
void TrackerFormsObj::reset_tool_browser(void) {
  register int i;
  char trbuf[128];
  
  int currline = fl_get_browser(tool_list);
  freeze();
    fl_clear_browser(tool_list);
    for(i=0; i < uiVR->tools.num(); i++) {
      sprintf(trbuf,"%1d. (%1d) %s", i+1, uiVR->tools[i]->sensor(),
      	(uiVR->tools[i]->top_tool())->name);
      fl_add_browser_line(tool_list, trbuf);
    }
    
    select_tool(currline);
  unfreeze();
}


// select the Nth tool
void TrackerFormsObj::select_tool(int tn) {
  if(fl_get_browser_maxline(tool_list) < 1)
    return;
    
  freeze();
  
    // if the new line is undefined, select the first item
    if(tn < 1)
      tn = 1;
    
    // make sure the proper line is selected
    if(fl_get_browser(tool_list) != tn)
      fl_select_browser_line(tool_list, tn);
      
    // set the value of all the controls to the settings for the given tool
    ToolControl *tmptool = uiVR->tools[tn - 1];
    float *offset = tmptool->offset();
    fl_set_counter_value(tool_scale, tmptool->scale());
    fl_set_counter_value(tool_length, tmptool->length());
    fl_set_counter_value(tool_detail, (float)(tmptool->detail()));
    fl_set_counter_value(tool_x_offset, offset[0]);
    fl_set_counter_value(tool_y_offset, offset[1]);
    fl_set_counter_value(tool_z_offset, offset[2]);
  unfreeze();
}


//////////////////////  protected virtual routines  

// routine to check the main form for use
int TrackerFormsObj::check(FL_OBJECT *obj) {
  char selbuf[256];

  if(obj == tracker_avail_list || obj == tracker_running_list) {
    ;		// nothing to do

  } else if(obj == tracker_start) {
    addcommand(new CmdTrackerStart(fl_get_browser(tracker_avail_list) - 1,
    					NULL, id()));

  } else if(obj == tracker_pause) {
    addcommand(new CmdTrackerPause(fl_get_browser(tracker_running_list) - 1,
    					id()));
    
  } else if(obj == tracker_unpause) {
    addcommand(new CmdTrackerUnpause(fl_get_browser(tracker_running_list) - 1,
    					id()));
    
  } else if(obj == tool_list) {
    select_tool(fl_get_browser(obj));

  } else if(obj == tool_scale || obj == tool_length || obj == tool_detail) {
    addcommand(new CmdToolSetSize(fl_get_browser(tool_list) - 1, 
                    fl_get_counter_value(tool_scale),
		    fl_get_counter_value(tool_length),
		    (int)(fl_get_counter_value(tool_detail)), 
		    id()));

  } else if(obj==tool_x_offset || obj==tool_y_offset || obj==tool_z_offset) {
    addcommand(new CmdToolSetOffset(fl_get_browser(tool_list) - 1, 
                    fl_get_counter_value(tool_x_offset),
		    fl_get_counter_value(tool_y_offset),
		    fl_get_counter_value(tool_z_offset), 
		    id()));

  } else if(obj == tool_type) {
    int newtype = fl_get_choice(obj) - 1;
    if(newtype == 1) {
      // just do a 'pop'
      addcommand(new CmdToolPop(fl_get_browser(tool_list) - 1, id()));
    } else if(newtype == 2) {
      // change to a 'grabber'
      addcommand(new CmdToolReplace(fl_get_browser(tool_list) - 1,
      				new GrabTool("grabber", scene), id()));
    }

  } else if(find_editing_mode(obj) >= 0) {
    set_editing_mode(find_editing_mode(obj));

  } else
    return FALSE;

  return TRUE;
}


// do form-specific acting on commands.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int TrackerFormsObj::forms_act_on_command(int type, Command *, int) {

  MSGDEBUG(3,"TrackerFormsObj: acting on command " << type << sendmsg);

  if(type == Command::TRACKER_START || type == Command::TRACKER_PAUSE ||
	type == Command::TRACKER_UNPAUSE) {
    reset_running_tracker_browser();

  } else if(type == Command::TOOL_SIZE || type == Command::TOOL_OFFSET) {
    select_tool(fl_get_browser(tool_list));

  } else if(type == Command::TOOL_PUSH || type == Command::TOOL_POP ||
  	type == Command::TOOL_REPLACE) {
    reset_tool_browser();

  } else
    // unknown command type
    return FALSE;
    
  return TRUE;
}


//////////////////////  public virtual routines  

// initialize the user interface
void TrackerFormsObj::init(void) {
  freeze();

    // set bounds, precision, etc. for counters
    fl_set_counter_bounds(tool_scale, 0.0, 10000.0);
    fl_set_counter_bounds(tool_length, 0.0, 10000.0);
    fl_set_counter_bounds(tool_detail, 1.0, 100.0);
    fl_set_counter_bounds(tool_x_offset, -10000.0, 10000.0);
    fl_set_counter_bounds(tool_y_offset, -10000.0, 10000.0);
    fl_set_counter_bounds(tool_z_offset, -10000.0, 10000.0);

    fl_set_counter_step(tool_scale, 0.1, 1.0);
    fl_set_counter_step(tool_length, 0.1, 1.0);
    fl_set_counter_step(tool_detail, 1.0, 1.0);
    fl_set_counter_step(tool_x_offset, 0.1, 1.0);
    fl_set_counter_step(tool_y_offset, 0.1, 1.0);
    fl_set_counter_step(tool_z_offset, 0.1, 1.0);

    fl_set_counter_precision(tool_scale, 2);
    fl_set_counter_precision(tool_length, 2);
    fl_set_counter_precision(tool_detail, 0);
    fl_set_counter_precision(tool_x_offset, 2);
    fl_set_counter_precision(tool_y_offset, 2);
    fl_set_counter_precision(tool_z_offset, 2);

    fl_set_counter_value(tool_scale, 1.0);
    fl_set_counter_value(tool_length, 1.0);
    fl_set_counter_value(tool_detail, 5.0);
    fl_set_counter_value(tool_x_offset, 0.0);
    fl_set_counter_value(tool_y_offset, 0.0);
    fl_set_counter_value(tool_z_offset, 0.0);

    // initialize menu with settings for the tool type
    fl_clear_choice(tool_type);
    fl_addto_choice(tool_type, "(no change)");
    fl_addto_choice(tool_type, "(previous)");
    fl_addto_choice(tool_type, "grabber");

    // do a reset to set initial values
    reset();

    // set the current editing mode
    set_editing_mode(DEFAULT_TR_EDIT_MODE);

  unfreeze();
}


// reset the user interface
void TrackerFormsObj::reset(void) {
  reset_avail_tracker_browser();
  reset_running_tracker_browser();
  reset_tool_browser();
}
