 

/*****************************************************************************
 *

   tracker.h- include file for tracker library; contains all #defines,
    	    	typedefs, and external declarations for using the 
		tracker library.  There is no tracker-specific information
		in this file-  all such definitions should be put in a
		local include file for that tracker, and then included
		by table.h ONLY.  The main upper-level trackerlib code
		must (by definition) be ignorant of tracker-specific
		knowledge.

   Conventions:
    	- all constants start with T_
    	- complex typedef names start with t_ and end with the word "type"
	    (simple ones start with t_, but may omit the trailing "type")
	- all global variables start with t_
	- all general tracker routines start with t_
	- all distance measurements are in meters


   Revision History:
    
    RCS: $Id: tracker.h,v 2.0 93/06/17 14:57:26 holloway Exp $

   (see tracker.c for main revision history)

   Author	    	Date	  Comments
   ------	    	--------  ----------------------------
   Rich Holloway    	06/13/93  Created one version of trackerlib for
                                   all architectures
   Rich Holloway	01/13/93  Added t_get_unit_offsets()
   Rich Holloway	07/17/92  Added t_get_time, minor fixes to polhemus
    	    	    	    	    driver
   Erik Erikson         06/1/92   Added the CM_EXTERN_FUNCTION prototypes
   Rich Holloway	05/26/92  Major rework: added support for IEEE 488
    	    	    	    	    interface and a layer between comm. 
				    interface and driver code.  Added 
				    'interface' and 'interface_info' keywords
				    for .tracker file.  Fastrak works.
   Rich Holloway	05/13/92  Fixed t_null_read to support 
    	    	    	    	    t_set_unit_offsets()
   Rich Holloway    	05/06/92  Killed signal handler
   Rich Holloway    	12/19/91  Added t_command() for tracker-specific
    	    	    	    	    commands;  handles hemisphere command
   Rich Holloway    	07/08/91  Added interrupt handler to shut down all
    	    	    	    	    trackers on ^C; added t_set_unit_offsets
   Rich Holloway    	06/16/91  Path for T_MASTER_CONFIG_DIR is now
    	    	    	    	    different for each architecture;  added
				    support for TRACKER = "null" for null
				    tracker device
   Russ Taylor		05/29/91  Ported to Iris
   Rich Holloway	04/29/91  Changed t_open to support opening multiple
    	    	    	    	    trackers using TRACKER env var
   Rich Holloway	03/09/91  Cleared any tracker-specific information
    	    	    	    	    from tracker.h;  simplified table.h.
   Rich Holloway	02/26/91  Bird driver should work, byte-swapping code
    	    	    	    	    fixed, localized unix dependencies.
   Rich Holloway	01/31/91  Release 1.0
   Rich Holloway	10/02/90  Initial version for Beta 1.0


   Developed at the University of North Carolina at Chapel Hill, supported
   by the following grants/contracts:
   
     DARPA #DAEA18-90-C-0044
     ONR #N00014-86-K-0680
     NIH #5-R24-RR-02170
   
 *
 *****************************************************************************/


/* handle multiple includes of this file	*/
#ifndef T_TRACKER_INCLUDED
#define T_TRACKER_INCLUDED

/* we use the std definition of NULL (at least) */
#include <stdio.h>

/* for quaternion definitions	*/
#include <quat.h>

/* for the CM_EXTERN_FUNCTION macros */
#include <cm_macros.h>

/*****************************************************************************
 *
    constants
 *
 *****************************************************************************/


/*****************************************************************************
 *
    site-specific constants
 *
 *****************************************************************************/

/*
 * master directory to look for .tracker file if not in cur or home dirs.
 *  should point to some centrally accessible directory for placing .tracker
 * file;  for multiple architectures, make this point to private lib dir
 * so that different machines can connect the same tracker to different
 * /dev's
 */

/* include vmd's config.h file to get location for .tracker file */
#include "config.h"

/* default definition	*/
#ifndef T_MASTER_CONFIG_DIR
#  ifdef VMDDIR
#    define T_MASTER_CONFIG_DIR 	VMDDIR
#  else
#    define T_MASTER_CONFIG_DIR 	"/vr/lib"
#  endif
#else
#  define T_MASTER_CONFIG_DIR 		"/vr/lib"
#endif

/* at unc, we have a library for communicating with serial devices over
 *  the network transparently;  this is not shipped with trackerlib by
 *  default (but is available)
 */
#undef T_USE_SDI

/*****************************************************************************
 *
    communications interface driver information
 *
 *****************************************************************************/

/*
 * In general:  define this iff there is a National 488 board connected to
 *  your machine.  At UNC, we only define this for suns since we don't 
 *  have any 488 boards for the SGI.  If you replace this with:
 *
 *  	#undef T_NATIONAL_488_DRIVER
 *
 *  or just comment it out, then the dummy routines defined in gpibdriver.c
 *  will be compiled instead.  Then make sure that all trackers specified
 *  in the .tracker file have 'interface = serial' specified.
 */
#undef T_NATIONAL_488_DRIVER



/*****************************************************************************
 *
    machine-specific constants
 *
 *****************************************************************************/

/* size of buffer on host for reading serial data.  note that on some
 *  systems (like the Vax) this is the real size of the read buffer;  on
 *  others (like the SGI), there is apparently no real buffer size, so
 *  we just deal in chunks of this size and make it look as if there
 *  is a read buffer this size.  This constant should not be changed unless
 *  the real read buffer size on your machine is smaller than this, in
 *  which case things might work & they might not.
 */
#define T_READ_BUFFER_SIZE	    256

/* extra big buffer for various operations  */
#define T_MAX_READ_BUFFER_SIZE      (2*T_READ_BUFFER_SIZE)

/* default baud rate for communicating with trackers	*/
#define T_DEFAULT_BAUD_RATE 	    19200


/*****************************************************************************
 *
    tracker-specific commands-  in general, these are commands that apply
    	to some subset of all trackers.  when adding new commands, please
	try to keep them as general-purpose as possible.  the convention
	for these is as follows:
	
	- command #defines begin with T_C_
	- the interface routine is t_command, which takes three arguments:
	    
	    - trackerIndex:  tracker table index
	    - command:  which command is being specified
	    - info pointer:  pointer to data area for any arguments for
	    	the command
 *
 *****************************************************************************/

/* for magnetic trackers, such as polhemus and bird, set active hemisphere  */
/* Query for the optical tracker */
/* Force-feedback command for the tracker */
#define T_C_SET_HEMISPHERE	    200
#define	T_C_REVERSE_HEMISPHERE      201
#define	T_C_DATA_NOT_READY	    202
#define	T_C_FORCE_FEEDBACK	    203

/*****************************************************************************
 *
    Data structures passed to the T_C_ commands.
 *
 *****************************************************************************/

/* this is for the UNC force feedback arm   */
typedef	struct {
		int	unit;		/* Unit to apply force to */
		double	F[3];		/* Force to apply */
		double	T[3];		/* Torque to apply */
	} t_force_command_type;


/*****************************************************************************
 *
    general constants
 *
 *****************************************************************************/

#define T_TRUE	1
#define T_FALSE	0

/* i/o stuff	*/
#define T_READ_ONLY	"r"

/* low level i/o    */
/* null file descriptor	*/
#define T_NULL_FD   	(-1)

/* return codes	*/
#define T_OK	    	    	0
#define T_ERROR     	    	(-1)
#define T_NULL_FUNC_PTR	    	(-2)
#define T_UNRECOGNIZED_COMMAND	(-3)

/* i/o function called does not support this tracker's comm. interface    */
#define T_UNSUPPORTED_DEVICE	(-4)

/*
 * t_read return codes
 */

/* useful for joyboxes as trackers, and other devices that can have a
 *  "dead zone"
 */
#define T_NO_MOVEMENT	33


/* this is returned by tracker routines if the driver routine ptr was NULL.
 *  this may or may not signal an error depending on the tracker & context.
 */
#define T_NULL_DRIVER	(-69)

/* string constants */
#define T_STRING_EQUAL	0

/* report format options    */
#define T_XYZ_QUAT  	    	40
#define T_PPHIGS_MATRIX	    	(T_XYZ_QUAT+1)
#define T_XFORM_MATRIX	    	(T_PPHIGS_MATRIX+1)
#define T_DEFAULT_DATA_FORMAT	(T_XFORM_MATRIX)

/* types of interface for communication.  these are used by the low-level
 *  read routines for figuring out which driver functions to call.
 *   for parallel interfaces, will need to be of the form 
 *  	T_<vendor>_<number>;    eg, T_NATIONAL_488,  T_XYTEC_485, etc.
 *  note we need to distinguish between different vendors of the same 
 *  interface standard, since they may have different driver routines
 *  for read/write/open/close.
 * if you add a new type of interface here, add a keyword for it in the 
 *  parser (parser.y,lexconfig.l,config.c,etc), so that it can be parsed
 *  in the .tracker file.
 */
/* boring old serial line   */
#define T_SERIAL  	    110

/* a/d board	*/
#define T_ANALOG_DIGITAL    (T_SERIAL+1)

/* this is the National Instruments IEEE 488 board  */
#define T_NATIONAL_488      (T_ANALOG_DIGITAL+1)


/*****************************************************************************
 *
    these constants determine the size of the tracker table, t_table.
 *
 *****************************************************************************/

/* number of slots in driver table, including null entries. */
#define T_MAX_NUM_TRACKERS    	35



/*****************************************************************************
 *
   frame constants:  used by t_frame to set origin and orientation of
    	    	    	tracker coordinate system.  subsequent tracker
			reports are given relative to the system chosen.

    - T_TRACKER_COORDS_ROOM_ORIENT
	T_TRACKER_COORDS_ROOM_ORIENT means that the
	tracker's origin should be used, but the orientation should
	be the room orientation.  Useful for making all trackers have
	the same orientation while still having the world centered
	on that tracker's origin.  This is the default.

    - T_ROOM_FRAME
	Specifies that the south-west corner of the room is the 
	origin, with x = east, y = north, z = up.  This is useful
	when you want to use a model of the room in your virtual 
	world that coincides with the real room.  In this mode, 
	the user's code is independent of where the tracker is 
	within the room, so that the tracker's origin can be moved 
	to a new location in the room, but no recompiling is 
	necessary (see note on .tracker in trackerlib(3l) for 
	details). 

    - T_TRACKER_FRAME
	Use the tracker's native coordinate system:  the tracker's
	source is the origin, and its coordinate axes are the axes of
	the frame.

 *
 *****************************************************************************/

#define T_TRACKER_COORDS_ROOM_ORIENT    60
#define T_ROOM_FRAME	    	    	T_TRACKER_COORDS_ROOM_ORIENT+1
#define T_TRACKER_FRAME	    	    	T_ROOM_FRAME+1

/* default frame for tracker position information  */
#define T_DEFAULT_FRAME    T_TRACKER_COORDS_ROOM_ORIENT

/* the maximum number of units a tracker can have.  if this changes, 
 *  change the values of T_NULL_UNIT_TABLE and T_DEFAULT_UNIT_XFORMS   */
#define T_MAX_NUM_UNITS	    8

/* this needs to be a large, positive number, since unitsOn[] is sorted	*/
#define T_NULL_UNIT 	    32760

/* this is an initialization table for unitsOn, which is an array of 
 * T_MAX_NUM_UNITS integers
 */
#define T_NULL_UNIT_TABLE { T_NULL_UNIT, T_NULL_UNIT, \
    	    	    	    T_NULL_UNIT, T_NULL_UNIT, \
    	    	    	    T_NULL_UNIT, T_NULL_UNIT, \
			    T_NULL_UNIT, T_NULL_UNIT }

/* default xform for units  */
#define T_DEFAULT_UNIT_XFORMS {	{ T_FALSE, {{0, 0, 0}, {0, 0, 0, 1}} }, \
    	    	    	    	{ T_FALSE, {{0, 0, 0}, {0, 0, 0, 1}} }, \
    	    	    	    	{ T_FALSE, {{0, 0, 0}, {0, 0, 0, 1}} }, \
    	    	    	    	{ T_FALSE, {{0, 0, 0}, {0, 0, 0, 1}} }, \
    	    	    	    	{ T_FALSE, {{0, 0, 0}, {0, 0, 0, 1}} }, \
    	    	    	    	{ T_FALSE, {{0, 0, 0}, {0, 0, 0, 1}} }, \
    	    	    	    	{ T_FALSE, {{0, 0, 0}, {0, 0, 0, 1}} }, \
    	    	    	    	{ T_FALSE, {{0, 0, 0}, {0, 0, 0, 1}} } }

/* max length of TRACKER env var    */
#define T_ENV_STRING_LENGTH 	80

/* this character string contains all valid separators for tracker names
 *  in the list specified by the TRACKER env var
 */
#define T_ENV_STRING_SEPARATORS	    " \t:,"

/* maximum length of various t_table strings	*/
#define T_MAX_TRACKER_NAME_LENGTH   80
#define T_MAX_DEV_NAME_LENGTH	    80
#define T_MAX_FUNC_NAME_LENGTH	    80

/* null index for failed table searches, etc	*/
#define T_NULL_TRACKER	    	(-1)

/* this is a bogus tracker string to signal that we want to use the tracker
 * specified by the environment variable TRACKER
 */
#define T_ENV_TRACKER	    	   ((t_name_type *) NULL)

/* this is the tracker to use if none is specified-  default to first
 * tracker in the tracker table (after t_init has been run)
 */
#define T_DEFAULT_TRACKER_INDEX	    0

/* config file stuff	*/
#define T_CONFIG_FILE_NAME  	".tracker"

/* name for null tracker    */
#define T_NULL_TRACKER_NAME 	"null"

/* the longest name allowed for any config file (including path)  */
#define T_CONFIG_FILE_NAME_LENGTH  	256

/* for t_get_time   */
#define T_MICROSECS_PER_SEC	((t_time_type) 1e6)


/*****************************************************************************
 *
    typedefs
 *
 *****************************************************************************/


typedef int	t_boolean;

/* type of index into tracker table */
typedef int 	t_index;

/* for tracker names	*/
typedef char    t_name_type[T_MAX_TRACKER_NAME_LENGTH];

/* for tracker-specific commands    */
typedef int 	t_command_type;

/* for t_get_time   */
typedef double	t_time_type;

/*
 * trackerlib data formats
 */
/* xyzQuat represents position by x, y, z and orientation by a quaternion. 
 *  for more on quaternions, see Ken Shoemake's 1985 SIGGRAPH paper
 */
typedef q_xyz_quat_type t_xyz_quat_type;


/* standard Foley and van Dam transformation matrix	*/
typedef double  t_matrix_type[4][4];

/* transpose of t_matrix_type, with last row left off.  see PPHIGS
 *  Programmer's Manual for details
 */
typedef float t_pphigs_matrix_type[3][4];
    

/* typedef for report   */
typedef union
    {
    
    t_xyz_quat_type	    xyzQuat;
    t_matrix_type	    matrix;
    t_pphigs_matrix_type    pphigsMatrix;
    
    } t_report_type;


typedef struct
    {
    
    t_boolean	    xformFlag;	/* should this unit's report be xformed?    */
    t_xyz_quat_type xyzQuat;	/* xform values	    	    	    	    */
    
    } t_xform_type;

typedef struct
    {
    int	    type;   /* serial,	parallel (IEEE 488, 485, etc)	*/
    
    int	    info;   /* baud rate for serial interface, unused so far
    	    	     *  for parallel, but may be device address eventually.
    	     	     */

    char    device[T_MAX_DEV_NAME_LENGTH];	/* /dev entry or board 
    	    	    	    	    	    	 * identifier	
						 */
    } t_interface_type;

/* 
 * this is the main typedef for the tracker table, which is an array of
 *  these objects.  the tracker table is initialized in globals.c using
 *  the definitions in table.h, and updated again by the contents of 
 *  the .tracker file.
 */
typedef struct
    {
    t_name_type	    	name;
    
    t_interface_type	interface;    /* communications interface info	*/
    
    double  xOffset, 	    /* offsets from southwest corner of room to	    */
    	    yOffset,	    /*    tracker origin    	    	    	    */
	    zOffset,
	    
	    yawOffset,   /* angle offsets (in deg) between canonical orient */
	    pitchOffset, /*  and tracker's default orientation; these give  */
	    rollOffset;  /*  rotation from desired cs to real orientation   */
	    	    	 /*  these get inverted by code     	    	    */


    /*************************************************************************
     *
    	 private data items- not initialized in the .tracker file	
     *
     *************************************************************************/
    
    /* open/initialize tracker         	    */
    char    openFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*open)( CM_DOTDOTDOT );
                     /* t_table_type *trackerPtr, int numUnits */
        
    /* close/shut down tracker  	       	    */
    char    closeFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*close)( CM_DOTDOTDOT );
    
    /* read 'where' report from tracker    	    */
    char    readFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*read)( CM_DOTDOTDOT );
    
    /* enables a unit	*/
    char    enableFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*enable)( CM_DOTDOTDOT );	    	
    
    /* disables a unit	*/
    char    disableFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*disable)( CM_DOTDOTDOT );	    	
    
    /* tracker-specific commands routine	*/
    char    commandFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*command)( CM_DOTDOTDOT );	    	
    
    /* diagnostic function for setting up tracker   */
    char    diagsFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*diags)( CM_DOTDOTDOT );	    	

    /* user-definable extra driver routines	    */
    char    user1FuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*user1)( CM_DOTDOTDOT );	    	
    
    char    user2FuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*user2)( CM_DOTDOTDOT );	    	
    
    int	    fileDescriptor; 	/* fd for this tracker	*/
    int     numUnits;	    	/* number of tracker reports per poll   */
    
    /* this is a sorted list of active unit numbers.  it's sorted so that
     *  we can linearly zip through the list of active units without 
     *  worrying which ones are active.  this is particularly useful for
     *  the polhemus, since it can have any combination of units on 
     *  & sends a unit/station
     *  number with each report, which we need to use to verify
     *  that the report is ok.  with this, we can index into it from
     *  0 to numUnits very quickly on every read.  it also speeds access
     *  to the unitXforms array for transforming the unit reports.
     */
    int	    unitsOn[T_MAX_NUM_UNITS];
    
    /* xform info for each unit-  this is NOT sorted;  use the unitsOn array
     *  or the real unit number to index into this
     */
    t_xform_type    unitXforms[T_MAX_NUM_UNITS];
    
    int	    reportLength;   	/* length of 1 (non-encoded)
    	    	    	    	 *   report for 1 unit
				 */
    
    int	    totalReportLength;	/* length of the encoded report list for all
    	    	    	    	 *  active units
				 */

    int	    dataFormat;  	/* xyz, angles or quaternions, etc  */
    int	    mode; 	    	/* meaning is tracker-specific.  for polhemus,
    	    	    	    	 * indicates polling vs. continuous mode.
				 */
    int	    frame; 	    	/* frame for x, y, z; room or tracker based */
    q_type  offsetQuat;    	/* angular offsets in quaternion form	*/
    
    } t_table_type;


/* Type for t_tracker_funcs global, which maps tracker types to the
 * functions to call for each tracker.  The nameBase field is
 * compared to the name of the tracker to determine what type of 
 * tracker it is.  
 */

typedef struct {    
    t_name_type    nameBase;
    char    openFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*open)( CM_DOTDOTDOT );
    char    closeFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*close)( CM_DOTDOTDOT );
    char    readFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*read)( CM_DOTDOTDOT );
    char    enableFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*enable)( CM_DOTDOTDOT );	    	
    char    disableFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*disable)( CM_DOTDOTDOT );	    	
    char    commandFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*command)( CM_DOTDOTDOT );	    	
    char    diagsFuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*diags)( CM_DOTDOTDOT );	    	
    char    user1FuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*user1)( CM_DOTDOTDOT );	    	
    char    user2FuncName[T_MAX_FUNC_NAME_LENGTH];
    int	    (*user2)( CM_DOTDOTDOT );	    	
} t_tracker_funcs_type;



/*****************************************************************************
 *
    default tracker values
 *
 *****************************************************************************/

/* maximum possible number of tracker types */
/* (if you update this number, the t_tracker_funcs global will have to
   be updated) */
#define T_MAX_NUM_TRACKER_TYPES 16    

#define T_DEFAULT_NAME              ""
#define T_DEFAULT_INTERFACE_TYPE    T_SERIAL
#define T_DEFAULT_INTERFACE_INFO    19200
#define T_DEFAULT_INTERFACE_DEV	    ""

#define T_DEFAULT_X_OFFSET   	    0.0
#define T_DEFAULT_Y_OFFSET   	    0.0

/* this is the medium height setting 	*/
#define T_DEFAULT_Z_OFFSET   	    0.0

#define T_DEFAULT_ROLL_OFFSET       0.0
#define T_DEFAULT_PITCH_OFFSET      0.0
#define T_DEFAULT_YAW_OFFSET        0.0

#define T_DEFAULT_OPEN_NAME	    ""
#define T_DEFAULT_OPEN_PTR	    NULL

#define T_DEFAULT_CLOSE_NAME	    ""
#define T_DEFAULT_CLOSE_PTR	    NULL

#define T_DEFAULT_READ_NAME	    ""
#define T_DEFAULT_READ_PTR	    NULL

#define T_DEFAULT_ENABLE_NAME	    ""
#define T_DEFAULT_ENABLE_PTR	    NULL

#define T_DEFAULT_DISABLE_NAME      ""
#define T_DEFAULT_DISABLE_PTR	    NULL

#define T_DEFAULT_COMMAND_NAME      ""
#define T_DEFAULT_COMMAND_PTR	    NULL

#define T_DEFAULT_DIAGS_NAME	    ""
#define T_DEFAULT_DIAGS_PTR	    NULL

#define T_DEFAULT_USER1_NAME	    ""
#define T_DEFAULT_USER1_PTR	    NULL

#define T_DEFAULT_USER2_NAME	    ""
#define T_DEFAULT_USER2_PTR	    NULL

#define T_DEFAULT_FILE_DESCRIPTOR 	T_NULL_FD
#define T_DEFAULT_NUM_UNITS	    	0
#define T_DEFAULT_UNITS_ON	    	T_NULL_UNIT_TABLE
#define T_DEFAULT_REPORT_LENGTH   	0
#define T_DEFAULT_TOTAL_REPORT_LENGTH	0

#define T_DEFAULT_REPORT_FORMAT   	T_XFORM_MATRIX
#define T_DEFAULT_MODE		        0
#define T_DEFAULT_OFFSET_QUAT	    	0,0,0,1

/*****************************************************************************
 *
    declarations
 *
 *****************************************************************************/


/*****************************************************************************
 *
 * system globals
 *
 *****************************************************************************/

/* the main tracker table   */
extern t_table_type 	    t_table[T_MAX_NUM_TRACKERS];
extern t_tracker_funcs_type t_tracker_funcs[T_MAX_NUM_TRACKER_TYPES];

extern int  	 t_num_defined_trackers;

/* initialized report structures for first call to read, copied into user's
 *  report list in case of a bad first read-  guarantees that all data
 *  returned by trackerlib is meaningful, if not correct
 */
extern t_report_type t_identity_matrix;
extern t_report_type t_pphigs_identity_matrix;
extern t_report_type t_xyz_quat_init;


/*****************************************************************************
 *
    function declarations
 *
 *****************************************************************************/

/* 
 * user-code-callable routines	
 */

/* returns tracker number on success, T_NULL_TRACKER otherwise	*/
/* if trackerName = T_ENV_TRACKER, get index'th name from TRACKER
 *  environment variable
 */
CM_EXTERN_FUNCTION( t_index t_open, ( t_name_type trackerName[], int index ));

/* closes down tracker;  T_OK on success, T_ERROR otherwise */
CM_EXTERN_FUNCTION( int  t_close, ( t_index whichTracker ));

/* reads reports for all active units of this tracker    */
CM_EXTERN_FUNCTION( int  t_read, ( t_index        whichTracker, 
				   t_report_type *reportListPtr ));

/* turn on/off units of tracker; T_OK on success, T_NULL_DRIVER if no such
 *  driver for this tracker;  T_ERROR on failure
 */
CM_EXTERN_FUNCTION( int  t_enable, ( t_index whichTracker, int whichUnit ));
CM_EXTERN_FUNCTION( int  t_disable, ( t_index whichTracker, int whichUnit ));

/* sets the data type for this tracker, T_OK on success, else T_ERROR 	*/
CM_EXTERN_FUNCTION( int  t_data_format, ( t_index whichTracker, 
					  int     dataFormat ));

/* sets the frame to be as defined by the frame constants defined above	 */
CM_EXTERN_FUNCTION( int  t_frame, ( t_index whichTracker, int whichFrame ));

/* get the room angle & position offsets for this tracker's origin in room
 *  space
 */
CM_EXTERN_FUNCTION( void t_get_offsets, ( t_index 	whichTracker,
					  q_vec_type	posnVec,
					  q_type  	orientQuat ));


/* set xlate & orient offsets for a unit    */
CM_EXTERN_FUNCTION( int  t_set_unit_offsets, ( t_index 	tIndex,
					       int	unitNum,
					       q_type  	orientQuat,
					    q_vec_type	xlateVec ));

/* get xlate & orient offsets for a unit; quat and vec are return parms    */
CM_EXTERN_FUNCTION( int  t_get_unit_offsets, ( t_index 	tIndex,
					       int	unitNum,
					       q_type  	orientQuat,
					    q_vec_type	xlateVec ));

/*
 * occasional use routines
 */

/* return pointer to null-terminated string for name of tracker	*/
CM_EXTERN_FUNCTION( char *t_name, ( t_index trackerIndex ));

/* diagnostics function	for tracker */
CM_EXTERN_FUNCTION( int t_diags, ( int whichTracker ));


/* tracker-specific command routine */
CM_EXTERN_FUNCTION( int t_command, ( t_index  	     whichTracker,
				     t_command_type  command,
				     void    	    *infoPtr ));

/* print out numReports reports, assuming dataFormat   */
CM_EXTERN_FUNCTION( int  t_print_report, ( t_report_type *reportListPtr,
					   int  	  numReports, 
					   int  	  dataFormat ));

/* for printing out reports */
CM_EXTERN_FUNCTION( void  t_print_xform_matrix, ( t_report_type  reportList[], 
						  int	    	 numReports ));

CM_EXTERN_FUNCTION( void  t_print_pphigs_matrix, ( t_report_type reportList[], 
						   int	    	 numReports ));

CM_EXTERN_FUNCTION( void  t_print_xyz_quat, ( t_report_type reportList[], 
					      int	    numReports ));


/* conversion routines-  convert a list of xyz/quat reports to matrix types */
CM_EXTERN_FUNCTION( void t_xyzQuat_to_matrix, ( t_table_type   *trackerPtr,
					        t_report_type  reportList[] ));

CM_EXTERN_FUNCTION( void t_xyzQuat_to_pMatrix, ( t_table_type  *trackerPtr,
						 t_report_type reportList[] ));

/*****************************************************************************
 *
    timing routines
 *
 *****************************************************************************/

/* sleep a random wait of between 0 and max milliseconds   */
CM_EXTERN_FUNCTION( void t_rand_sleep, ( int max ));

/* sleep "wait" milliseconds	*/
CM_EXTERN_FUNCTION( void t_sleep, ( unsigned int wait ));

/* returns time in microseconds       */
CM_EXTERN_FUNCTION( t_time_type t_get_time, ( CM__VOID_ ));


/*****************************************************************************
 *
   service routines, not usually called by user code	
 *
 *****************************************************************************/


/* composes two xyz_quats-  src and dest can be the same    */
CM_EXTERN_FUNCTION( void t_compose_xform, ( t_xyz_quat_type *C_from_A_ptr, 
					    t_xyz_quat_type *C_from_B_ptr, 
					    t_xyz_quat_type *B_from_A_ptr ));

/*
 * communications routines:  these provide interface-independent access
 *  to serial/parallel devices for driver functions.  driver functions
 *  should never call low-level read/write functions directly.
 */

/* open device for read/write	*/
CM_EXTERN_FUNCTION( int  t_open_device, ( t_table_type  *trackerPtr ));

/* close device	*/
CM_EXTERN_FUNCTION( int  t_close_device, ( t_table_type  *trackerPtr ));

/* general read routine;  reads given buffer from device */
CM_EXTERN_FUNCTION( int  t_raw_read, ( t_table_type    *trackerPtr,
				       unsigned char   buffer[],
				       int	       numBytes ));

/* general write routine;  writes given buffer to device  */
CM_EXTERN_FUNCTION( int  t_raw_write, ( t_table_type    *trackerPtr, 
				        unsigned char   buffer[],
				        int	    	numBytes ));


/* clears the serial buffer associated with trackerPtr->dev */
CM_EXTERN_FUNCTION( int  t_flush, ( t_table_type *trackerPtr ));


/* read all available full reports  */
CM_EXTERN_FUNCTION( int  t_read_all_reports, 
		    ( t_table_type  *trackerPtr,
		      unsigned char  buffer[],
		      int	     totalReportLength ));

/* general non-hanging read function	*/
CM_EXTERN_FUNCTION( int  t_nonblock_read, ( t_table_type *trackerPtr,
					    unsigned char buffer[],
					    int	          numBytesRequested ));

/* hanging read function- waits until it gets bytesRequested	*/
CM_EXTERN_FUNCTION( void t_patient_read, ( t_table_type   *trackerPtr,
					   unsigned char  buffer[],
					   int	          bytesRequested ));

/* 
 * swaps the first bufSize bytes of buffer[] 
 *	 number of bytes to be swapped should be an even number, but
 *	    doesn't have to be-  odd byte will be left alone
 */
CM_EXTERN_FUNCTION( void t_swap_bytes, ( unsigned char buffer[], 
					 int           bufSize ));

/* reverse the byte order of words in a buffer	*/
CM_EXTERN_FUNCTION( void t_reverse_bytes, ( unsigned char destBuffer[], 
					    unsigned char srcBuffer[],
					    int	          wordSize, 
					    int           numWords ));

/* find a config file in the following search path:
 *     1) current dir  2) home dir  3) master config dir
 *  (as defined by T_MASTER_CONFIG_DIR) and return a file pointer to the 
 *  first one found, if any.  If no file is found, return NULL.
 */
CM_EXTERN_FUNCTION( FILE *t_open_config_file, ( char baseFileName[], 
					        char mode[] ));

/* find a config file in the following search path:
 *     1) current dir  2) home dir  3) master config dir
 *  (as defined by T_MASTER_CONFIG_DIR) and return a file pointer to the 
 *  first one found, if any.  If no file is found, return NULL.
 */
CM_EXTERN_FUNCTION( int t_open_all_config_files, ( char baseFileName[], 
					        char mode[],
                                                FILE* configFilePtrs[] ));

/* general init routine	*/
CM_EXTERN_FUNCTION( int t_init, ( CM__VOID_ ));

/* initializes tracker table	*/
CM_EXTERN_FUNCTION( int  t_init_table, ( CM__VOID_ ));

/* initializes offsets within room  */
CM_EXTERN_FUNCTION( int  t_init_offsets, ( CM__VOID_ ));

/* given a tracker name, finds its corresponding entry in tracker table	*/
CM_EXTERN_FUNCTION( t_index t_find_table_entry, ( char trackerName[] ));

/* prints out the driver table in format suitable for insertion 
 *  into a .tracker file.
 */
CM_EXTERN_FUNCTION( void t_print_table, ( CM__VOID_ ));

/* gets tracker name from TRACKER environment variable	*/
CM_EXTERN_FUNCTION( void t_name_from_env, ( char trackerName[] ));

/* gets the tracker table index of userTrackerName[index] or TRACKER[index] */
CM_EXTERN_FUNCTION( t_index t_get_index, ( t_name_type userTrackerName[], 
					   int         index ));

/* tracker table access routines */
CM_EXTERN_FUNCTION( int  t_add_unit, ( t_table_type *trackerPtr, 
				       int           whichUnit ));
CM_EXTERN_FUNCTION( int  t_remove_unit, ( t_table_type *trackerPtr,
					  int           whichUnit ));

/* used for putting user terminal in raw mode for diags routine	*/
CM_EXTERN_FUNCTION( int  t_open_raw_term, ( CM__VOID_ ));
CM_EXTERN_FUNCTION( int  t_close_raw_term, ( CM__VOID_ ));

/* initializes the identity reports for each data format;  these are
 *  copied into the report list on the first read call to avoid explosions
 *  in the user code in case of a bad initial report from the tracker
 */
CM_EXTERN_FUNCTION( int  t_init_reports, ( CM__VOID_ ));

/* handle first read call for this tracker;  initializes    
 *  report list in case of bad read call and sets tracker
 *  table to point to read routine appropriate for dataFormat
 */
CM_EXTERN_FUNCTION( int  t_first_read, ( t_table_type   *trackerPtr, 
					 t_report_type   reportList[],
					 char    	 funcName[],
					 int	         (*funcPtr)() ));


/*****************************************************************************
 *
    null tracker driver routines
 *
 *****************************************************************************/


/* pseudo read routine for null tracker.    */
CM_EXTERN_FUNCTION( int  t_null_read, ( t_table_type    *trackerPtr,
				        t_report_type   reportList[] ));

/* turn on a specified station/unit of the null tracker	*/
CM_EXTERN_FUNCTION( int  t_null_enable, ( t_table_type    *trackerPtr, 
					  int	    	   whichUnit ));


/*  turn off a specified station/unit of the null tracker   */
CM_EXTERN_FUNCTION( int  t_null_disable, ( t_table_type    *trackerPtr, 
					   int	    	    whichUnit ));

#endif /* T_TRACKER_INCLUDED */

