/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: AnimateFormsObj.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 95/03/24 18:46:12 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The on-screen menu to control the animation of molecules.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	AnimateFormsObj.C,v $
 * Revision 1.4  95/03/24  18:46:12  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.3  1994/11/11  16:36:37  billh
 * No change.
 *
 * Revision 1.2  94/10/21  03:52:41  billh
 * Now keeps track of what numbers are displayed in frame counter box, to
 * speed display.
 * 
 * Revision 1.1  1994/10/20  01:30:58  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/AnimateFormsObj.C,v 1.4 95/03/24 18:46:12 billh Exp $";
#endif

#include <stdio.h>
#include <stdlib.h>
#include "AnimateFormsObj.h"
#include "CmdAnimate.h"
#include "MoleculeList.h"
#include "Global.h"


// commands we are interested in
static int numCmds = 5;
static int cmdList[5] = { Command::ANIM_DIRECTION,
	Command::ANIM_JUMP,		Command::ANIM_SKIP,
	Command::ANIM_STYLE,		Command::ANIM_SPEED };

/////////////////////////  constructor  
AnimateFormsObj::AnimateFormsObj(UIList *uil, CommandQueue *cq,
	int sh,int bd,int *npos) : FormsObj("animate", uil, cq, bd, npos) {
	
  // set values of common menu elements
  form = animateMenu;
  lightButton = animate_menu_button;
  offButton = animate_menu_off;
  
  // register which commands we are interested in
  for(int i=0; i < numCmds; command_wanted(cmdList[i++]));
  
  // turn on if required
  if(sh)
    On();
    
  buttonMode = (-1);
  currFrame = totFrame = (-2);
}


/////////////////////////  destructor  
AnimateFormsObj::~AnimateFormsObj(void) {
  // nothing to do
}


/////////////////////////  private routines  

// set the state of the anim direction controls based on the top mol
void AnimateFormsObj::reset_dir_controls(void) {
  int newmode;
  if(mol)
    newmode = mol->anim_dir();
  else
    newmode = Animation::PAUSE;
  
  // set which buttons are pressed
  if(newmode != buttonMode) {
    freeze();
    fl_set_button(reverse_button,(newmode == Animation::REVERSE));
    fl_set_button(reverse1_button,(newmode == Animation::REVERSE1));
    fl_set_button(pause_button,(newmode == Animation::PAUSE));
    fl_set_button(forward1_button,(newmode == Animation::FORWARD1));
    fl_set_button(forward_button,(newmode == Animation::FORWARD));
    unfreeze();
    buttonMode = newmode;
  }
}


// set the status of the current frame, based on the top mol
void AnimateFormsObj::reset_frame_counter(void) {
  char txtbuf[64];
  int newcurr, newtot;

  if(mol) {
    newcurr = mol->frame();
    newtot = mol->num();
  } else {
    newcurr = newtot = (-1);
  }
  
  if(newcurr != currFrame || newtot != totFrame) {
    currFrame = newcurr;
    totFrame = newtot;
    freeze();
    if(currFrame >= 0)
      sprintf(txtbuf, "%d / %d", currFrame, totFrame);
    else
      strcpy(txtbuf, "");
    fl_set_object_label(frame_label, txtbuf);
    unfreeze();
  }
}


// set the state of other data items for this menu based on the top mol
void AnimateFormsObj::reset_status_items(void) {
  if(mol) {
    freeze();
    fl_set_counter_value(frame_skip, mol->skip());
    fl_set_choice(anim_style, mol->anim_style() + 1);
    fl_set_slider_value(anim_speed_slider, mol->speed());
    unfreeze();
  }
}

//////////////////////  protected virtual routines  

// routine to check the main form for use
int AnimateFormsObj::check(FL_OBJECT *obj) {

  if(obj == anim_start || obj == anim_end || obj == frame_input) {
    int newframe;
    if(obj == anim_start)
      newframe = SET_FRAME_START;
    else if(obj == anim_end)
      newframe = SET_FRAME_END;
    else
      newframe = atoi(fl_get_input(obj));
    addcommand(new CmdAnimJump(newframe, id()));
    
  } else if(obj == reverse_button || obj == forward_button ||
  	obj == reverse1_button || obj == forward1_button ||
	obj == pause_button) {
    Animation::AnimDir newDir;
    if(obj == reverse_button)
      newDir = Animation::REVERSE;
    else if(obj == forward_button)
      newDir = Animation::FORWARD;
    else if(obj == reverse1_button)
      newDir = Animation::REVERSE1;
    else if(obj == forward1_button)
      newDir = Animation::FORWARD1;
    else
      newDir = Animation::PAUSE;
    addcommand(new CmdAnimDir(newDir, id()));
    
  } else if(obj == frame_skip) {
    addcommand(new CmdAnimSkip((int)(fl_get_counter_value(obj)), id()));
    
  } else if(obj == anim_style) {
    Animation::AnimStyle newStyle = Animation::ONCE;
    int nsint = fl_get_choice(obj) - 1;
    if (nsint == Animation::ONCE)
      newStyle = Animation::ONCE;
    else if(nsint == Animation::ROCK)
      newStyle = Animation::ROCK;
    else if(nsint == Animation::LOOP)
      newStyle = Animation::LOOP;
    addcommand(new CmdAnimStyle(newStyle, id()));
    
  } else if(obj == anim_speed_slider) {
    addcommand(new CmdAnimSpeed(fl_get_slider_value(obj), id()));
  
  } else
    return FALSE;

  return TRUE;
}


// do form-specific acting on commands.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int AnimateFormsObj::forms_act_on_command(int type, Command *, int) {

  MSGDEBUG(3,"AnimateFormsObj: acting on command " << type << sendmsg);

  if(type == Command::ANIM_DIRECTION) {
    reset_dir_controls();
    
  } else if(type == Command::ANIM_JUMP) {
    reset();
    
  } else if(type == Command::ANIM_SKIP || type == Command::ANIM_STYLE ||
  		type == Command::ANIM_SPEED) {
    reset_status_items();

  } else
    // unknown command type
    return FALSE;
    
  return TRUE;
}


//////////////////////  public virtual routines  

// initialize the user interface
void AnimateFormsObj::init(void) {
  freeze();
    // set all the basic values for controllers, etc
    fl_set_counter_bounds(frame_skip, 1.0, 100000.0);
    fl_set_counter_step(frame_skip, 1.0, 1.0);
    fl_set_counter_precision(frame_skip, 0);
    fl_set_input(frame_input, "");
    fl_clear_choice(anim_style);
    for(int i=0; i < Animation::TOTAL_STYLES; i++)
      fl_addto_choice(anim_style, animationStyleName[i]);
   
    // do a reset to set initial values
    reset();

  unfreeze();
}


// reset the user interface
void AnimateFormsObj::reset(void) {
  freeze();
    reset_dir_controls();
    reset_frame_counter();
    reset_status_items();
  unfreeze();
}


// update the user interface after every redraw
void AnimateFormsObj::update(void) {
  if(displayed()) {
    freeze();
      reset_dir_controls();
      reset_frame_counter();
    unfreeze();
  }
}
