/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdDisplay.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 95/03/24 18:47:43 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   These are the Commands that control the various aspects
 * of the display, like, the clipping planes, eye separation, etc.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	CmdDisplay.C,v $
 * Revision 1.3  95/03/24  18:47:43  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.2  1994/11/02  07:31:00  billh
 * Added commands to change display screen size and distance to screen.
 *
 * Revision 1.1  94/10/03  01:42:46  dalke
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/CmdDisplay.C,v 1.3 95/03/24 18:47:43 billh Exp $";
#endif


#include "CmdDisplay.h"
#include "ColorList.h"
#include "LightList.h"
#include "Axes.h"
#include "Stage.h"
#include "Global.h"

// These are the Command:: enums used by this file:
//  DISP_RESHAPE, DISP_STEREO, DISP_EYESEP, DISP_FOCALLEN, 
//  DISP_LIGHT_ON, DISP_LIGHT_HL, DISP_LIGHT_ROT, DISP_MATERIALS_CHANGE,
//  DISP_CLIP, DISP_DEPTHCUE, DISP_ANTIALIAS, DISP_SCRHEIGHT, DISP_SCRDIST,
//  CMD_AXES, CMD_STAGE

// reshape the display
int CmdDisplayReshape::do_execute(void) {
  int retval;
  if(retval = (display != NULL))
    display->reshape();
  return retval;
}
CmdDisplayReshape::CmdDisplayReshape(int newUIid)
  : Command(Command::DISP_RESHAPE, newUIid) {
  *cmdText << "display reshape";
 *cmdText << ends;
}

// set stereo mode of display
int CmdDisplayStereo::do_execute(void) {
  int retval;
  if(retval = (display != NULL
      && stmode >= 0 && stmode < display->num_stereo_modes()))
    display->set_stereo_mode(stmode);
  return retval;
}
CmdDisplayStereo::CmdDisplayStereo(int newstmode, int newUIid)
  : Command(Command::DISP_STEREO, newUIid) {
  stmode = newstmode;
  if(display && stmode >= 0 && stmode < display->num_stereo_modes())
    *cmdText << "display stereo " << display->stereo_name(stmode);
  else
    *cmdText << "display stereo " << stmode;
  *cmdText << ends;
}


// set eye separation of display
int CmdDisplayEyesep::do_execute(void) {
  int retval;
  if(retval = (display != NULL))
    display->set_eyesep(sep);
  return retval;
}
CmdDisplayEyesep::CmdDisplayEyesep(float newsep, int newUIid)
  : Command(Command::DISP_EYESEP, newUIid) {
  sep = newsep;
  *cmdText << "display eyesep " << sep;
  *cmdText << ends;
}


// set focal length of display
int CmdDisplayFocallen::do_execute(void) {
  int retval;
  if(retval = (display != NULL
      && flen != 0.0 && display->eye_dist() != 0.0)) {
    float lx, ly, lz, fl;
    display->where_looking(lx, ly, lz);
    fl = flen / (display->eye_dist());
    lx *= fl;  ly *= fl;  lz *= fl;
    display->change_look(lx, ly, lz);
  }
  return retval;
}
CmdDisplayFocallen::CmdDisplayFocallen(float newlen, int newUIid)
  : Command(Command::DISP_FOCALLEN, newUIid) {
  flen = newlen;
  *cmdText << "display focallength " << flen;
  *cmdText << ends;
}


// set screen height value
int CmdDisplayScreenHeight::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    display->screen_height(val);
  }
  return retval;
}
CmdDisplayScreenHeight::CmdDisplayScreenHeight(float newval, int newUIid)
  : Command(Command::DISP_SCRHEIGHT, newUIid) {
  val = newval;
  *cmdText << "display height " << val;
  *cmdText << ends;
}


// set distance to screen from origin
int CmdDisplayScreenDistance::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    display->distance_to_screen(val);
  }
  return retval;
}
CmdDisplayScreenDistance::CmdDisplayScreenDistance(float newval, int newUIid)
  : Command(Command::DISP_SCRDIST, newUIid) {
  val = newval;
  *cmdText << "display distance " << val;
  *cmdText << ends;
}


// turn on/off the Nth light
int CmdDisplayLightOn::do_execute(void) {
  int retval;
  if(retval = (lights != NULL && n >= 0 && n < DISP_LIGHTS)) {
    lights->set(n);
    if(onoff)
      (lights->current())->on();
    else
      (lights->current())->off();
  }
  return retval;
}
CmdDisplayLightOn::CmdDisplayLightOn(int ln, int turnon, int newUIid)
  : Command(Command::DISP_LIGHT_ON, newUIid) {
  n = ln;  onoff = turnon;;
  *cmdText << "light " << n;
  *cmdText << (onoff ? " on" : " off");
  *cmdText << ends;
}


// highlight the Nth light
int CmdDisplayLightHL::do_execute(void) {
  int retval;
  if(retval = (lights != NULL && n >= 0 && n < DISP_LIGHTS)) {
    lights->set(n);
    (lights->current())->highlight(hl);
  }
  return retval;
}
CmdDisplayLightHL::CmdDisplayLightHL(int ln, int highlt, int newUIid)
  : Command(Command::DISP_LIGHT_HL, newUIid) {
  n = ln;  hl = highlt;
  *cmdText << "light " << n;
  *cmdText << (hl ? " highlight" : " unhighlight");
  *cmdText << ends;
}


// rotate the position of the Nth light
int CmdDisplayLightRot::do_execute(void) {
  int retval;
  if(retval = (lights != NULL && n >= 0 && n < DISP_LIGHTS)) {
    lights->set(n);
    (lights->current())->add_rotation(theta, axis);
  }
  return retval;
}
CmdDisplayLightRot::CmdDisplayLightRot(int ln, float th, char ax,
  int newUIid): Command(Command::DISP_LIGHT_ROT, newUIid) {
  n = ln;  theta = th;  axis = ax;
  *cmdText << "light " << n;
  *cmdText << " rot " << axis << " " << theta; 
  *cmdText << ends;
}


// change material characteristics for the display
int CmdDisplayMaterialsChange::do_execute(void) {
  int retval;
  if(retval = (colors != NULL && n >= 0 && n < MAXCOLORS)) {
    if(ctype == ALPHA_INDEX)
      colors->change_alpha(n, mdata[0]);
    else if(ctype == SHININESS_INDEX)
      colors->change_shininess(n, mdata[0]);
    else if(ctype == COLOR_INDEX)
      colors->change_color(n, mdata[0], mdata[1], mdata[2]);
    else if(ctype == AMBIENT_INDEX)
      colors->change_ambient(n, mdata[0], mdata[1], mdata[2]);
    else if(ctype == DIFFUSE_INDEX)
      colors->change_diffuse(n, mdata[0], mdata[1], mdata[2]);
    else if(ctype == SPECULAR_INDEX)
      colors->change_specular(n, mdata[0], mdata[1], mdata[2]);
  }
  return retval;
}
CmdDisplayMaterialsChange::CmdDisplayMaterialsChange(int c, int typ, 
                                  float *data, int newUIid)
  : Command(Command::DISP_MATERIALS_CHANGE, newUIid) {
  n = c;  ctype = typ;
  *cmdText << "color " << n;
  if(ctype == ALPHA_INDEX) {
    mdata[0] = data[0];
    *cmdText << " alpha " << mdata[0];
  } else if(ctype == SHININESS_INDEX) {
    mdata[0] = data[0];
    *cmdText << " shininess " << mdata[0];
  } else if(ctype == COLOR_INDEX) {
    mdata[0] = data[0]; mdata[1] = data[1]; mdata[2] = data[2];
    *cmdText << " color " << mdata[0] << " " << mdata[1] << " "
            << mdata[2];
  } else if(ctype == AMBIENT_INDEX) {
    mdata[0] = data[0]; mdata[1] = data[1]; mdata[2] = data[2];
    *cmdText << " ambient " << mdata[0] << " " << mdata[1] << " "
            << mdata[2];
  } else if(ctype == DIFFUSE_INDEX) {
    mdata[0] = data[0]; mdata[1] = data[1]; mdata[2] = data[2];
    *cmdText << " diffuse " << mdata[0] << " " << mdata[1] << " "
            << mdata[2];
  } else if(ctype == SPECULAR_INDEX) {
    mdata[0] = data[0]; mdata[1] = data[1]; mdata[2] = data[2];
    *cmdText << " specular " << mdata[0] << " " << mdata[1] << " "
            << mdata[2];
  }
  *cmdText << ends;
}


// turn on/off antialiasing
int CmdDisplayAAOn::do_execute(void) {
  int retval;
  if(retval = (display != NULL && display->aa_available())) {
    if(onoff)
      display->aa_on();
    else
      display->aa_off();
  }
  return retval;
}
CmdDisplayAAOn::CmdDisplayAAOn(int turnon, int newUIid)
  : Command(Command::DISP_ANTIALIAS, newUIid) {
  onoff = turnon;
  *cmdText << "display antialias ";
  *cmdText << (onoff ? "on" : "off");
  *cmdText << ends;
}


// turn on/off depth cueing
int CmdDisplayDepthcueOn::do_execute(void) {
  int retval;
  if(retval = (display != NULL && display->cueing_available())) {
    if(onoff)
      display->cueing_on();
    else
      display->cueing_off();
  }
  return retval;
}
CmdDisplayDepthcueOn::CmdDisplayDepthcueOn(int turnon, int newUIid)
  : Command(Command::DISP_DEPTHCUE, newUIid) {
  onoff = turnon;
  *cmdText << "display depthcue ";
  *cmdText << (onoff ? "on" : "off");
  *cmdText << ends;
}


//  change the axes location
int CmdDisplayAxes::do_execute(void) {
  int retval;
  if(retval=(axes != NULL && newpos >= 0 && newpos < axes->locations()))
    retval = axes->location(newpos);
  return retval;
}
CmdDisplayAxes::CmdDisplayAxes(int npos, int newUIid)
  : Command(Command::CMD_AXES, newUIid) {
  newpos = npos;
  if((axes != NULL) && newpos >= 0 && newpos < axes->locations()) {
    *cmdText << "axes location " << axes->loc_description(newpos);
    *cmdText << ends;
 }
}
    

//  change the stage location
int CmdDisplayStage::do_execute(void) {
  int retval;
  if(settingToChange == LOCATION) {
    if(retval=(stage !=NULL && newpos >= 0 && newpos < stage->locations()))
      retval = stage->location(newpos);
  } else if(settingToChange == PANELS) {
    if(retval=(stage !=NULL))
      retval = stage->panels(newpos);
  } else
    retval = FALSE;
  return retval;
}
CmdDisplayStage::CmdDisplayStage(Settings toset, int npos, int newUIid)
  : Command(Command::CMD_STAGE, newUIid) {
  newpos = npos;
  settingToChange = toset;
  if((stage != NULL)) {
    *cmdText << "stage ";
    if(settingToChange == LOCATION && newpos >= 0 && 
        newpos < stage->locations()) {
      *cmdText << "location " << stage->loc_description(newpos);
    } else if(settingToChange == PANELS) {
      *cmdText << "panels " << newpos;
    }
    *cmdText << ends;
  }
}


// This handles the whole range of clipping plane options
//  There are derived classes so you won't have to have the funky flags
// to change {near,fixed} clipping plane {to a fixed,by a relative} amount
// or not
int CmdDisplayClip::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    if (changenear) {
      if (setval)
        display -> set_near_clip(amount);
      else
        display->addto_near_clip(amount);
    } else {
      if (setval)
        display -> set_far_clip(amount);
      else
        display->addto_far_clip(amount);
    }
  }
  return retval;
}
CmdDisplayClip::CmdDisplayClip(int ischangenear, int issetval, 
                                float newamt, int UIid)
  : Command(Command::DISP_CLIP, UIid) {
  changenear = ischangenear;
  setval = issetval;
  amount = newamt;
  *cmdText << "display ";
  *cmdText << (changenear ? "near" : "far");
  *cmdText << "clip " << (setval ? "set " : "add ");
  *cmdText << amount;
  *cmdText << ends;
}  

 
