/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: CmdLabel.C,v $
 *      $Author: billh $        $Locker:  $                $State: Exp $
 *      $Revision: 1.3 $      $Date: 95/03/24 18:47:48 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Command objects used to create, list, delete, or graph labels for measuring
 * geometries.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	CmdLabel.C,v $
 * Revision 1.3  95/03/24  18:47:48  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.2  1995/03/17  22:26:42  billh
 * Atom name specification now requires segment name instead of residue name;
 * fixed so no longer displays error message when label toggled on/off
 *
 * Revision 1.1  1995/03/04  05:33:08  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/CmdLabel.C,v 1.3 95/03/24 18:47:48 billh Exp $";
#endif

#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include "CmdLabel.h"
#include "GeometryList.h"
#include "GeometryAtom.h"
#include "GeometryBond.h"
#include "GeometryAngle.h"
#include "GeometryDihedral.h"
#include "MoleculeList.h"
#include "Molecule.h"
#include "Global.h"
#include "Inform.h"


// return codes for atom name string
enum AtomNameReturnCodes { ANAME_ILLEGAL, ANAME_BADMOL, ANAME_BADSEG,
	ANAME_BADRES, ANAME_BADATOM, ANAME_OK };

// utility routine: return mol ID and atom index given a string
// representing an atom.
// If returns == ANAME_OK, name was OK and atom index found.
// If returns == ANAME_ILLEGAL, illegal specification;
// If returns == ANAME_BADMOL, unknown molecule ID
// If returns == ANAME_BADSEG, unknown segment specification
// If returns == ANAME_BADRES, unknown residue specification
// If returns == ANAME_BADATOM, unknown atom name
int find_atom_from_name(char *checkstr, int *mid, int *aid) {
  register int i;
  int mol_id, colons = 0, dashes = 0;
  Molecule *mol;
  char *segname, *resid, *namestart;
  char astr[128];

  // make sure there are 1..2 colons, and 1 dash
  if(checkstr) {

    // make a copy of the string, since we'll modify things
    strcpy(astr, checkstr);

    char *tmpstr = astr;
    while(*tmpstr) {
      if(*tmpstr == ':') {
        colons++;
	*tmpstr = '\0';
	if(colons == 1)
	  segname = tmpstr + 1;
	else if(colons == 2)
	  namestart = tmpstr + 1;
      } else if(*tmpstr == '-') {
        dashes++;
	*tmpstr = '\0';
	resid = tmpstr + 1;
      }
      tmpstr++;
    }
  }

  if(dashes != 1 || colons < 1 || colons > 2)
    return ANAME_ILLEGAL;

  // first find the molecule ID, if present
  if(colons == 2 && isdigit(*astr)) {
    mol_id = atoi(astr);
    mol = moleculeList->molecule(moleculeList->mol_index_from_id(mol_id));
    if(!mol)
      return ANAME_BADMOL;
  } else if(colons == 1 && moleculeList->top()) {
    mol = moleculeList->top();
    mol_id = mol->id();
    namestart = segname;
    segname = astr;
  } else 
    return ANAME_BADMOL;
  
  // now find the indices for the atom names
  int segnameindex = (mol->segNames).typecode(segname);
  if(segnameindex < 0)
    return ANAME_BADSEG;
  int residindex = (mol->resIds).typecode(resid);
  if(residindex < 0)
    return ANAME_BADRES;
  int atomindex = (mol->atomNames).typecode(namestart);
  if(atomindex < 0)
    return ANAME_BADATOM;

  // finally search through the molecule for the proper atom
  for(i=(mol->nAtoms - 1); i >= 0; i--) {
    Atom *atom = mol->atom(i);
    if(atom->segnameindex == segnameindex && atom->residindex == residindex &&
    	atom->nameindex == atomindex) {
      *mid = mol_id;
      *aid = i;
      return ANAME_OK;
    }
  }
  
  // if here, no atom matched.  return bad atom error
  return ANAME_BADATOM;
}


// check the given category name, and return its index if OK, -1 otherwise
int check_geom_cat_name(char *geomcat) {
  int retval = geometryList->geom_list_index(geomcat);
  if(retval < 0)
    msgErr << "Unknown geometry category '" << geomcat << "'." << sendmsg;
  return retval;
}


// print out name and value in common format
void display_label_name(int n, Geometry *g) {
  msgInfo << " ";
  if(n >= 0)
    msgInfo << n << ": ";
  msgInfo << g->name();
  if(g->has_value())
    msgInfo << " = " << g->calculate();
}

////////////////////// add a new label 
CmdLabelAdd::CmdLabelAdd(char *geomcat, int n, char **items, int UIid) :
	Command(LABEL_ADD, UIid) {
  // NOTE: this creates the proper Geometry object; if it cannot do this,
  //	error messages are printed, and no Geometry obj is created.

  cat = (-1);
  geom = NULL;

  if(n < 1) {
    msgErr << "No objects specified to be labeled." << sendmsg;
  } else {  
    // find geometry category, if known
    cat = check_geom_cat_name(geomcat);
    if(cat >= 0 && cat < 4) {
      // find molecule id's and atom names
      int *middata = new int[n];
      int *atmdata = new int[n];
      int i;
      for(i=0; i < n; i++) {
        int errcode = find_atom_from_name(items[i], middata+i, atmdata+i);
        char *errmsg = NULL;
        switch(errcode) {
          case ANAME_ILLEGAL: errmsg="Illegal atom name syntax"; break;
	  case ANAME_BADMOL:  errmsg="Unknown molecule in atom name"; break;
	  case ANAME_BADRES:  errmsg="Unknown residue name in atom name"; break;
	  case ANAME_BADATOM: errmsg = "Unknown atom name"; break;
        }
        if(errmsg) {
          msgErr << errmsg << ": '" << items[i] << "'" << sendmsg;
	  break;		// from for loop
        }
      }
    
      if(i == n) {
        if(cat == geometryList->geom_list_index("Atoms") && n == 1)
          geom = new GeometryAtom(middata[0], atmdata[0], moleculeList);
        else if(cat == geometryList->geom_list_index("Bonds") && n == 2)
          geom = new GeometryBond(middata, atmdata, moleculeList);
        else if(cat == geometryList->geom_list_index("Angles") && n == 3)
          geom = new GeometryAngle(middata, atmdata, moleculeList);
        else if(cat == geometryList->geom_list_index("Dihedrals") && n == 4)
          geom = new GeometryDihedral(middata, atmdata, moleculeList);
      }

      delete [] middata;
      delete [] atmdata;
    }
  }

  *cmdText << "label add " << geomcat;
  for(int j=0; j < n; j++)
    *cmdText << " " << items[j];
  *cmdText << ends;
}

int CmdLabelAdd::do_execute(void) {

  if(cat >= 0 && geom != NULL) {
    int retval = geometryList->add_geometry(cat, geom);
    if(retval <= 0) {
      if(retval == 0)
        msgErr << "Could not add new label " << geom->name() << sendmsg;
      delete geom;
    } else {
      msgInfo << "Added new " << geometryList->geom_list_name(cat) << " label";
      display_label_name((-1), geom);
      msgInfo << sendmsg;
    }
    return retval; 
  } else {
    msgErr << "Could not add new label." << sendmsg;
  }
  
  // if here, command failed
  return FALSE;
}


////////////////////// list the labels in a category
CmdLabelList::CmdLabelList(char *geomcat, int UIid) :
	Command(LABEL_LIST, UIid) {

  // find geometry category, if known
  cat = check_geom_cat_name(geomcat);

  *cmdText << "label list " << geomcat << ends;
}

int CmdLabelList::do_execute(void) {

  if(cat >= 0) {
    GeomListPtr glist = geometryList->geom_list(cat);
    int gnum = glist->num();
    msgInfo << "Labels in category '" << geometryList->geom_list_name(cat);
    msgInfo << "': " << gnum << sendmsg;
    for(int i=0; i < gnum; i++) {
      display_label_name(i, (*glist)[i]);
      msgInfo << sendmsg;
    }
    
    return TRUE;
  }
  
  // if here, command failed
  return FALSE;
}


////////////////////// toggle a geometry category on/off
CmdLabelShow::CmdLabelShow(char *geomcat, int s, int n, int UIid) :
	Command(LABEL_SHOW, UIid) {

  // find geometry category, if known
  cat = check_geom_cat_name(geomcat);
  show = s;
  item = n;

  *cmdText << "label " << (show ? "show" : "hide") << " " << geomcat;
  if(n >= 0)
    *cmdText << " " << n;
  *cmdText << ends;
}

int CmdLabelShow::do_execute(void) {

  if(cat >= 0) {
    geometryList->show_geometry(cat, item, show);
    return TRUE;
  }
  
  // if here, command failed
  return FALSE;
}

//////////////////////// delete a label
CmdLabelDelete::CmdLabelDelete(char *geomcat, int n, int UIid) :
	Command(LABEL_DELETE, UIid) {

  // find geometry category, if known
  cat = check_geom_cat_name(geomcat);
  item = n;

  *cmdText << "label delete " << geomcat;
  if(n >= 0)
    *cmdText << " " << n;
  *cmdText << ends;
}

int CmdLabelDelete::do_execute(void) {

  if(cat >= 0) {
    return geometryList->del_geometry(cat, item);
  }

  // if here, command failed
  return FALSE;
}


////////////////////// graph the values for the Nthe label in the given category
CmdLabelGraph::CmdLabelGraph(char *geomcat, int n, char *gcmd, int UIid) :
	Command(LABEL_GRAPH, UIid) {

  // find geometry category, if known
  cat = check_geom_cat_name(geomcat);
  item = n;

  // save graphing command
  graphCmd = stringdup(gcmd);
  
  *cmdText << "label graph " << geomcat << " " << n << " " << graphCmd << ends;
}

int CmdLabelGraph::do_execute(void) {

  if(cat >= 0) {
    // get geometry list
    GeomListPtr glist = geometryList->geom_list(cat);
    int gnum = glist->num();
    if(item < 0 || item >= gnum) {
      msgErr << "Illegal label number " << item << sendmsg;
      return FALSE;
    }
    
    // get geometry object; return error if label has no values
    Geometry *g = (*glist)[item];
    if(g->has_value()) {

      // fill array with list of geom values; return error if cannot calc all
      ResizeArray<float> gValues(1024);
      if(g->calculate_all(gValues)) {

        // print or graph data
        if(! strlen(graphCmd)) {
          // if go graph command, write data to console
          msgInfo << "Values for label";
          display_label_name(item, g);
          msgInfo << sendmsg;
          for(int i=0; i < gValues.num(); i++) {
	    msgInfo << "  "  << gValues[2*i] << "  " << gValues[2*i + 1];
	    msgInfo << sendmsg;
          }
        } else {
          // if a graphing command is given, write data to temp file and run
          // the command via 'system'
      
          // first create a temporary filename
          char filename[128];
          tmpnam(filename);
      
          // now open the file for writing
          FILE *outfile = fopen(filename, "w");
          if (!outfile) {
            msgErr << "Cannot open file for graph output." << sendmsg;
            return FALSE;
          }
      
          // and write the values into it
          int values = (gValues.num() / 2);
          for(int i=0; i < values; i++)
            fprintf(outfile,"%f  %f\n", gValues[2*i], gValues[2*i + 1]);
          fclose(outfile);

          // execute the graphing command, giving the filename as an optionally
          // used argument
          char execstr[128];
          sprintf(execstr, graphCmd, filename);
	  double beforeCmd = time_of_day();
          system(execstr);

          // make sure there is at least three seconds before continuing
	  double afterCmd = time_of_day();
	  if(afterCmd - beforeCmd < 2.0)
            sleep(3 - (int)(afterCmd - beforeCmd));
      
          // finally delete the file
          unlink(filename);
        }
      
        // we were successful
        return TRUE;
      }
    }

    // if here, could not graph properly
    msgErr << "Unable to graph data." << sendmsg;
  }
  
  // but if we're here, we were not successful
  return FALSE;
}
