/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: ColorList.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.16 $	$Date: 1995/03/28 03:37:54 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A collection of colored materials; provides ways to define them and edit
 * them.  Each Displayable must turn on or off material characteristics
 * if needed, and say which color to use.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: ColorList.C,v $
 * Revision 1.16  1995/03/28  03:37:54  billh
 * Added a few more built-in residues.
 * Make black an 'extra' color; the first 16 colors are 'visible' colors,
 * which are used to color molecules such that black is not assigned
 * as a color for an unknown segment or residue name, etc.
 *
 * Revision 1.15  95/03/24  18:48:21  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.14  1995/03/10  07:48:41  dalke
 * corrected end of file
 *
 * Revision 1.13  1995/03/09  00:19:46  dalke
 * added way to get index nearest to an rgb triplet
 *
 * Revision 1.12  1994/12/07  03:19:14  dalke
 * Almost done Blk/Wht scale
 *
 * Revision 1.11  1994/12/06  08:32:25  billh
 * Added reset_color routine, which sets the colors to their default
 * values.
 * Changed ICEBLUE color to BLACK; ICEBLUE objects now CYAN.
 *
 * Revision 1.10  94/12/05  08:25:47  billh
 * Added public member colorNames with string names for the regular colors.
 * Added many options for changing how the color scale is done.
 * 
 * Revision 1.9  94/11/21  14:44:43  billh
 * Reversed colormap; now runs blue->red instead of red->blue.
 * 
 * Revision 1.8  94/11/11  18:25:31  billh
 * Made colormap colors brighter (now in range 0.4 ... 1 instead of 0 ... 1)
 * 
 * Revision 1.7  94/11/09  06:34:08  dalke
 * Now uses 'MULT' instead of 'NONE' for matrix method.
 * 
 * Revision 1.6  1994/11/09  02:49:42  billh
 * Changed colormap definition; now looks more like red ... green ... blue
 *
 * Revision 1.5  1994/11/02  01:34:47  billh
 * alpha = 0.3 by default now (instead of 0.5).
 *
 * Revision 1.4  94/10/01  03:03:59  billh
 * Removed 'update' routine.
 * 
 * Revision 1.3  94/09/24  20:26:32  billh
 * Added color category NameList's, which are used by the AtomColor
 * object to determine the color for each atom, and which are used to
 * change the color for different items.
 * 
 * Revision 1.2  1994/09/12  20:51:41  billh
 * Changed due to new Displayable constructor arguments.
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /tmp_mnt/Home/h2/billh/projects/vmd/src/RCS/ColorList.C,v 1.16 1995/03/28 03:37:54 billh Exp $";
#endif

#include <stdio.h>
#include "ColorList.h"
#include "Displayable.h"
#include "DispCmds.h"
#include "DisplayDevice.h"
#include "NameList.h"
#include "Inform.h"

// globally-accessible strings with methods for creating color scale
char *colorScaleMethod[ColorList::SCALE_TOTAL] = { "RGB", "RWB", "BlkW" };

// globally-accessible strings with names for the regular colors
char *defColorNames[REGCLRS] = { "blue", "red", "gray", "orange", "yellow",
	"tan", "silver", "green", "white", "pink", "cyan", "purple",
	"lime", "mauve", "ochre", "iceblue", "black" };

// static data for default colors (same as diffuse characteristic)
static float defaultColor[REGCLRS][3] = {
        { 0.0, 0.0, 1.00 }, { 1.00, 0.0, 0.0 }, // BLUE, RED
        { 0.45, 0.45, 0.45 }, { .8, 0.5, .2 },  // GREY, ORANGE
        { 0.8, 0.8, 0.0 }, { 0.5, 0.5, 0.2 },   // YELLOW, TAN
        { 0.6, 0.6, 0.6 }, { 0.0, 0.5, 0.0 },   // SILVER, GREEN
        { 1.0, 1.0, 1.0 }, { 1.0, 0.6, 0.6 },   // WHITE, PINK
        { 0.25, 0.75, 0.75 }, { 0.5, 0.2, 0.5 },        // CYAN, PURPLE
        { 0.5, 0.9, 0.4 }, { 0.9, 0.4, 0.7 },   // LIME, MAUVE
        { 0.5, 0.3, 0.0 }, { 0.5, 0.75, 0.75 }, // OCHRE, ICEBLUE
	{ 0.0, 0.0, 0.0 }			// BLACK
};

static float defaultSpecular[3] = { 1.0, 1.0, 1.0 };
static float defaultAmbient[3] = { 0.0, 0.0, 0.0 };
static float defaultShininess = 40.0;
static float defaultAlphaSolid = 1.0;
static float defaultAlphaTrans = 0.3;


// constructor
ColorList::ColorList(Scene *sc) : Displayable3D(MULT, "Color List", sc, 5) {
  int i, j;
  float *rcol, *tcol;
  char cNameNumBuf[4];
  
  dispCmdColorDef = new DispCmdColorDef;

  // initialize flags
  for(i=0; i < MAXCOLORS; needDefine[i++] = TRUE);
  needAnyDefined = TRUE;
  needDefineCleanup = FALSE;
  
  // initialize color data for 'regular' colors
  for(i=0; i < REGCLRS; i++) {
  
    // add color names to colorNames list
    colorNames.add_name(defColorNames[i], i);

    // initialize color data
    reset_color(i);
  }

  // add integer names for colors as well as string names
  for(i=0; i < REGCLRS; i++) {
    sprintf(cNameNumBuf, "%d", i);
    colorNames.add_name(cNameNumBuf, i);
  }

  // initialize some color data for 'colormap' colors
  for(i=0; i < MAPCLRS; i++) {
    rcol = colorData[BEGMAP + i];
    tcol = colorData[BEGTRANMAP + i];
    for(j=0; j < 3; j++) {
      rcol[AMBIENT_INDEX + j]  = tcol[AMBIENT_INDEX + j] = defaultAmbient[j];
      rcol[SPECULAR_INDEX + j] = tcol[SPECULAR_INDEX + j]= defaultSpecular[j];
    }

    // assign values to other components
    rcol[SHININESS_INDEX] 	= tcol[SHININESS_INDEX] = defaultShininess;
    rcol[ALPHA_INDEX]		= defaultAlphaSolid;
    tcol[ALPHA_INDEX]		= defaultAlphaTrans;
  }
  
  // initialize data for rest of color scale
  scaleMethod = RGB;
  scaleMin = 0.1;
  scaleMax = 1.0;
  scaleMidpoint = 0.5;
  create_colorscale();

  // Displayable characteristics
  rot_off();
  scale_off();
  glob_trans_off();
  cent_trans_off();
  
  MSGDEBUG(1,"Creating Color List with " << MAXCOLORS << " colors ...");
  MSGDEBUG(2,sendmsg);
}


// destructor
ColorList::~ColorList(void) {
  // delete all added color categories
  for(int i=0; i < categories.num(); i++) {
    delete (categories.data(i));
    categories.set_data(i,NULL);
  }

  delete dispCmdColorDef;
}


// create the color scale based on current settings
void ColorList::create_colorscale(void) {
  float *rcol, *tcol;

  // make sure settings and other values are OK
  if(scaleMethod != RGB && scaleMethod != RWB && scaleMethod != BLK_W)
    scaleMethod = RGB;
  if(scaleMin < 0 || scaleMin >= scaleMax)
    scaleMin = 0.0;
  if(scaleMidpoint < 0 || scaleMidpoint > 1.0)
    scaleMidpoint = 0.5;

  // find 'middle' color, and sizerange of colors
  float range = scaleMax - scaleMin;
  float midcolor = 0.5 * (scaleMax + scaleMin);

  // precalc the 'quarterpoints'
  float leftmidpt = 0.5 * scaleMidpoint;
  float rightmidpt = 0.5 * (scaleMidpoint + 1.0);

  // precalc some slopes for graphs
  float pslope, nslope, leftslope, rightslope;
  if(scaleMidpoint > 0.0)
    nslope = - (range / scaleMidpoint);
  if(scaleMidpoint < scaleMax)
    pslope = range / (1.0 - scaleMidpoint);
  if(leftmidpt < scaleMidpoint)
    leftslope = range / (scaleMidpoint - leftmidpt);
  if(rightmidpt > scaleMidpoint)
    rightslope = range / (scaleMidpoint - rightmidpt);

  // create the color scale
  // initialize color data for 'colormap' colors
  for(int i=0; i < MAPCLRS; i++) {
    rcol = colorData[BEGMAP + i];
    tcol = colorData[BEGTRANMAP + i];
    
    // define the diffuse color, which determines the color of the material
    // color map goes from blue (small numbers) through green (mid numbers)
    // to red (large numbers)
    float relpos = (float)i / (float)(MAPCLRS - 1);
    float rcolor, gcolor, bcolor;

    if(scaleMethod == RGB) {
      rcolor = gcolor = bcolor = scaleMin;

      // set red component
      if(relpos > scaleMidpoint) 
        rcolor += pslope * (relpos - scaleMidpoint);

      // set green component
      if(relpos > leftmidpt && relpos <= scaleMidpoint) 
        gcolor += leftslope * (relpos - leftmidpt);
      else if(relpos >= scaleMidpoint && relpos < rightmidpt)
        gcolor += range + rightslope * (relpos - scaleMidpoint);

      // set blue component
      if(relpos < scaleMidpoint) 
        bcolor += range + (nslope * relpos);

    } else if(scaleMethod == RWB) {
      rcolor = gcolor = bcolor = midcolor;

      // set red component
      if(relpos < scaleMidpoint)
        rcolor += range + (0.5 * nslope * relpos); 

      // set blue component (green is always midcolor)
      if(relpos > scaleMidpoint)
        bcolor += 0.5 * pslope * (relpos - scaleMidpoint);
   } else if (scaleMethod == BLK_W) {
      if(relpos < scaleMidpoint)  // slope up to half
        rcolor = gcolor = bcolor = 0 + (relpos - 0.0) /
                   (scaleMidpoint - 0.0)*0.5;

      if(relpos >= scaleMidpoint)  // from half to full
        rcolor = gcolor = bcolor = 0.5 + (relpos - scaleMidpoint) /
	           (1 - scaleMidpoint) * 0.5;
   }

    // assign values to components
    rcol[DIFFUSE_INDEX + 0]	= tcol[DIFFUSE_INDEX + 0] = rcolor;
    rcol[DIFFUSE_INDEX + 1]	= tcol[DIFFUSE_INDEX + 1] = gcolor;
    rcol[DIFFUSE_INDEX + 2]	= tcol[DIFFUSE_INDEX + 2] = bcolor;    
  }

  // signal we need these colors redefined
  needAnyDefined = TRUE;
  for(int csn=0; csn < MAPCLRS; csn++)
    needDefine[MAPCOLOR(csn)] = needDefine[MAPTRANSLUCENT(csn)] = TRUE;
}


// add new color category; return it's index (may already have been added)
int ColorList::add_color_category(char *newcat) {
  int oldnum, retval;

  if(!newcat)
    return (-1);
 
  // add new name with NULL category; see what new index is
  oldnum = categories.num();
  retval = categories.add_name(newcat,NULL);
  if(oldnum != categories.num()) {
    // number of categories changed, so we must have added a new one.
    // create a new NameList, and put it in the data value for the new cat.
    categories.set_data(retval, new NameList<int>);

    MSGDEBUG(2,"Added new color category '" << newcat << "', index="<< retval);
    MSGDEBUG(2,sendmsg);
  } else {
    MSGDEBUG(2,"Existing color category '" << newcat << "', index=" << retval);
    MSGDEBUG(2," found." << sendmsg);
  }
  
  return retval;
}


//
// change data for the Nth color
//
  
void ColorList::change_alpha(int n, float a) {
  if(n >= 0 && n < MAXCOLORS) {
    colorData[n][ALPHA_INDEX] = a;
    needDefine[n] = needAnyDefined = TRUE;
  }
}

void ColorList::change_shininess(int n, float s) {
  if(n >= 0 && n < MAXCOLORS) {
    colorData[n][SHININESS_INDEX] = s;
    needDefine[n] = needAnyDefined = TRUE;
  }
}

void ColorList::change_ambient(int n, float r, float g, float b) {
  if(n >= 0 && n < MAXCOLORS) {
    colorData[n][AMBIENT_INDEX] = r;
    colorData[n][AMBIENT_INDEX + 1] = g;
    colorData[n][AMBIENT_INDEX + 2] = b;
    needDefine[n] = needAnyDefined = TRUE;
  }
}

void ColorList::change_specular(int n, float r, float g, float b) {
  if(n >= 0 && n < MAXCOLORS) {
    colorData[n][SPECULAR_INDEX] = r;
    colorData[n][SPECULAR_INDEX + 1] = g;
    colorData[n][SPECULAR_INDEX + 2] = b;
    needDefine[n] = needAnyDefined = TRUE;
  }
}

void ColorList::change_diffuse(int n, float r, float g, float b) {
  if(n >= 0 && n < MAXCOLORS) {
    colorData[n][DIFFUSE_INDEX] = r;
    colorData[n][DIFFUSE_INDEX + 1] = g;
    colorData[n][DIFFUSE_INDEX + 2] = b;
    needDefine[n] = needAnyDefined = TRUE;
  }
}

void ColorList::change_color(int n, float r, float g, float b) {
  change_diffuse(n, r, g, b);
}


// restore the Nth color to it's original form
void ColorList::reset_color(int n) {
  if(n >= BEGREGCLRS && n < (BEGREGCLRS + REGCLRS)) {
    float *rcol = colorData[BEGREGCLRS + n];
    float *tcol = colorData[BEGTRAN + n];
    for(int j=0; j < 3; j++) {
      rcol[AMBIENT_INDEX + j]  = tcol[AMBIENT_INDEX + j] = defaultAmbient[j];
      rcol[SPECULAR_INDEX + j] = tcol[SPECULAR_INDEX + j]= defaultSpecular[j];
      rcol[DIFFUSE_INDEX + j]  = tcol[DIFFUSE_INDEX + j] = defaultColor[n][j];
    }
    rcol[SHININESS_INDEX] = tcol[SHININESS_INDEX] = defaultShininess;
    rcol[ALPHA_INDEX] = defaultAlphaSolid;
    tcol[ALPHA_INDEX] = defaultAlphaTrans;
    
    needDefine[n] = needAnyDefined = TRUE;
  }
}

// given (r,g,b) find the color index in the current list which
// best matches it in the least square sense
int ColorList::nearest_index(float r, float g, float b)
{
   float *rcol = colorData[BEGREGCLRS];  // get the solid colors
   float lsq = r - rcol[DIFFUSE_INDEX]; lsq *= lsq;
   float tmp = g - rcol[DIFFUSE_INDEX+1]; lsq += tmp * tmp;
         tmp = b - rcol[DIFFUSE_INDEX+1]; lsq += tmp * tmp;
   float best = lsq;
   int bestidx = BEGREGCLRS;
   for (int n= BEGREGCLRS+1; n < (BEGREGCLRS + REGCLRS); n++) {
      rcol = colorData[BEGREGCLRS+n];
      lsq = r - rcol[DIFFUSE_INDEX]; lsq *= lsq;
      tmp = g - rcol[DIFFUSE_INDEX+1]; lsq += tmp * tmp;
      tmp = b - rcol[DIFFUSE_INDEX+2]; lsq += tmp * tmp;
      if (lsq < best) {
       best = lsq;
       bestidx = n;
      }
   }
   return bestidx;
}


// create the command list
void ColorList::create_cmdlist(void) {
  // do reset first
  reset_disp_list();
  
  // check if need to define any materials
  if(needAnyDefined) {
    for(int i=0; i < MAXCOLORS; i++) {
      if(needDefine[i]) {
        dispCmdColorDef->putdata(i, colorData[i], this);
	needDefine[i] = FALSE;
      }
    }
  }

  // reset flags
  needDefineCleanup = needAnyDefined;
  needAnyDefined = FALSE;
}


// prepare the colorlist for display ... regenerate command list if necessary
void ColorList::prepare(DisplayDevice *) {
  // if a flag has been set, change the settings
  if(needAnyDefined || needDefineCleanup)
    create_cmdlist();
}
