/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CoorDCD.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $	$Date: 95/03/24 18:48:36 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * DCD file class - subclasses of ICoorFile and OCoorfile; used to read and
 * write DCD files.  Since a DCD file contains multiple coordinates, the
 * user must call the read routine multiple times, until no sets are left.
 * For output files, the user must specify in the constructor the type
 * (DCD_BINARY or DCD_ASCII) of the output file.
 *
 * A BaseMolecule must be supplied to both input and output versions,
 * so that the structure data can be accessed.
 *
 * NOTE: currently, this supports only BINARY format.  ASCII uses that
 * strange 12Z6 fortran format, which is a pain to deal with.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	CoorDCD.C,v $
 * Revision 1.2  95/03/24  18:48:36  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.1  1994/09/17  09:11:36  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/CoorDCD.C,v 1.2 95/03/24 18:48:36 billh Exp $";
#endif

#include <stdio.h>
#include "CoorDCD.h"
#include "ReadDCD.h"
#include "BaseMolecule.h"
#include "Timestep.h"
#include "Inform.h"
#include "Atom.h"

/****************************  ICoorDCD routines  **************************/

// constructor
ICoorDCD::ICoorDCD(BaseMolecule *newmol) {
  // save the molecule pointer
  mol = newmol;

  // save number of atoms expected in file
  nAtoms = mol->nAtoms;
  
  // a DCD file can store many sets, but we don't know how many yet.
  
  // by default, assume in binary format
  fileType = BINARY;
  fileNum = (-1);
  frameStart = frameSkip = 0;
  namnf = 0;
  
  X = Y = Z = NULL;
  freeatoms = NULL;
}


// close the input file; return success
ICoorDCD::~ICoorDCD(void) {
  if(opened()) {
    if(file_type() == ASCII)
      fclose(cf);
    else
      close_dcd_read(fileNum, namnf, freeatoms);
  }
  if(X) delete [] X;
  if(Y) delete [] Y;
  if(Z) delete [] Z;
}


// initializer; reads header info and inits all variables.  Returns if file was
// initialized.
int ICoorDCD::init(char *fn) {
  int errcode;
  int dcdatoms;
  
  // if file is already open, we can continue processing it.
  if(opened())
    return TRUE;

  // initially we assume init failed; also, must do base-class initialization
  Initialized = FALSE;
  ICoorFile::init(fn);

  // open the file
  // only try to open as binary 
  fileNum = open_dcd_read(filename());
  if(fileNum < 0) {
    if(fileNum == DCD_DNE)
      errorNo = DCD_ERR_NOFILE;
    else if(fileNum == DCD_OPENFAILED)
      errorNo = DCD_ERR_BADFORMAT;
    else
      errorNo = DCD_ERR_NOTOPEN;
    return FALSE;
  }
  fileType = BINARY;
  
  errcode = read_dcdheader(fileNum, &dcdatoms, &nFrames, &frameStart,
	&frameSkip, &timeStep, &namnf, &freeatoms);
  if(errcode < 0 || dcdatoms <= 0) {
    if(errcode == DCD_BADFORMAT)
      errorNo = DCD_ERR_BADFORMAT;
    else if(errcode == DCD_BADMALLOC)
      errorNo = DCD_ERR_NOTOPEN;
    else if(dcdatoms <= 0) 
      errorNo = DCD_ERR_NOATOMS;
    else
      errorNo = DCD_ERR_EARLYEOF;
    return FALSE;
  } else if(dcdatoms != nAtoms) {
    errorNo = DCD_ERR_A_NOT_EQ;
    return FALSE;
  }

  MSGDEBUG(1,"Opened DCD input file '" << filename() << "'." << sendmsg);
  MSGDEBUG(1,"   Atoms: " << nAtoms << sendmsg);
  MSGDEBUG(1,"   Start frame: " << frameStart << sendmsg);
  if(namnf != 0)
    MSGDEBUG(1,"   Constrained atoms: " << namnf << sendmsg);

  X = new float[nAtoms];
  Y = new float[nAtoms];
  Z = new float[nAtoms];
  
  return (Initialized = TRUE);
}


// read the next set of coordinates; return list of positions, or NULL if
// error
Timestep* ICoorDCD::read(void) {
  int i,j,k;
  Timestep *pos;
  float *myx = X, *myy = Y, *myz = Z;

  if(!opened() || (nFrames && currFrames == nFrames))
    return NULL;

  // read in next step, if available
  i = read_dcdstep(fileNum, nAtoms, X, Y, Z, namnf,
  	(currFrames == 0), freeatoms);
  if(i < 0) {
    if(i == (-1))
      return NULL;
    else if(i == DCD_BADFORMAT)
      errorNo = DCD_ERR_BADFORMAT;
    else
      errorNo = DCD_ERR_EARLYEOF;
    return NULL;
  }
    
  // create a new position list to store the data
  if(mol->num() > 0) {
    // a frame exists to get extra data from
    pos = new Timestep(nAtoms, timeStep, (mol->item(0))->data);
  } else {
    // this is the first frame; must have Timestep allocate space for data
    pos = new Timestep(nAtoms, timeStep);
  }
  
  // put data in Timestep
  for(i=0, j=0; i < nAtoms; i++) {
    pos->pos[j++] = *(myx++);
    pos->pos[j++] = *(myy++);
    pos->pos[j++] = *(myz++);
  };
  
  // possibly put in extra data
  if(mol->num() == 0) {
    for(i=0, j=0; i < nAtoms; i++) {
      Atom *atom = mol->atom(i);
      for(k=0; k < ATOMEXTRA; k++)
        pos->data[j++] = atom->extra[k];
    }
  }
  
  currFrames++;
  return pos;
}


/****************************  OCoorDCD routines  **************************/
 
// constructor
OCoorDCD::OCoorDCD(BaseMolecule *nm, DCDTYPE typ, int nframes) : OCoorFile() {
  // save the molecule pointer
  mol = nm;

  // save number of atoms expected in file
  nAtoms = mol->nAtoms;

  fileType = typ;
  nFrames = (nframes > 0 ? nframes : 0);
  
  // currently, only binary files can be written or read
  if(fileType != BINARY) {
    msgErr << "Cannot create DCD file, only BINARY format supported.";
    msgErr << sendmsg;
  }
}


// destructor
// close the output file; return success
OCoorDCD::~OCoorDCD(void) {
  if(opened()) {
    if(file_type() == ASCII)
      fclose(cf);
    else
      close_dcd_write(fileNum);
  }
  if(X) delete [] X;
  if(Y) delete [] Y;
  if(Z) delete [] Z;
}


// initializer; writes header info and inits all variables.  Returns TRUE
// if file was initialized.
int OCoorDCD::init(char *fn) {  
  
  // if file is already open, we can continue processing it.
  if(opened())
    return TRUE;

  Initialized = FALSE;

  // if filetype is not binary, error
  if(fileType != BINARY) {
    msgErr << "Cannot create DCD file, only BINARY format supported.";
    msgErr << sendmsg;
    return FALSE;
  }

  // must do base-class initialization
  OCoorFile::init(fn);

  // open the file
  // only try to open as binary 
  fileNum = open_dcd_write((char *)filename());
  if(fileNum < 0) {
    if(fileNum == DCD_OPENFAILED)
      errorNo = DCD_ERR_BADFORMAT;
    else if(fileNum == DCD_FILEEXISTS)
      errorNo = DCD_ERR_FILEEXISTS;
    else
      errorNo = DCD_ERR_NOTOPEN;
    return FALSE;
  }
  fileType = BINARY;
  
  // write the header for the DCD file
  write_dcdheader(fileNum, (char *)filename(), nAtoms, nFrames,
  	frameStart, frameSkip, timeStep);
  
  MSGDEBUG(1,"Opened DCD output file '" << filename() << "'." << sendmsg);
  MSGDEBUG(1,"   Atoms: " << nAtoms << sendmsg);
  MSGDEBUG(1,"   Start frame: " << frameStart << sendmsg);

  X = new float[nAtoms + 1];
  Y = new float[nAtoms + 1];
  Z = new float[nAtoms + 1];
  
  return (Initialized = TRUE);
}


// write the coordinates in the given coordinate set to the file.  Return
// total frames written so far, or (-1) if error.
int OCoorDCD::write(Timestep *pos) {
  int i, j;
  float *myx = X, *myy = Y, *myz = Z;
  
  // check for errors
  if(!opened() || !pos)
    return (-1);

  if(pos->num != nAtoms) {
    errorNo = DCD_ERR_A_NOT_EQ;
    return (-1);
  }
  
  // write the coordinate data
  for(i=0, j=0; i < nAtoms; i++) {
    *(myx++) = pos->pos[j++];
    *(myy++) = pos->pos[j++];
    *(myz++) = pos->pos[j++];
  }
  write_dcdstep(fileNum, nAtoms, X, Y, Z);
  
  return ++currFrames;
}

