/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CoorFile.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 95/03/24 18:48:39 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Base class for all coordinate files, both for reading and writing.
 * Subclasses must provide methods for opening and closing files,
 * and for reading/writing coordinates.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	CoorFile.h,v $
 * Revision 1.3  95/03/24  18:48:39  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.2  1994/11/21  14:45:09  billh
 * No change.
 *
 * Revision 1.1  94/09/17  09:11:36  billh
 * Initial revision
 * 
 ***************************************************************************/
#ifndef COORFILE_H
#define COORFILE_H

#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <pwd.h>
#include "utilities.h"
extern class Timestep;

// max number of comment lines that will be kept
#define MAX_COOR_REM	10

// error flag value if no error
#define NO_COOR_ERR	 0


////////////////////////////// base class for input and output versions
class CoorFile {

private:
  // number of comment lines read and stored, up to MAX_COOR_REM
  int nRemLines;
  char *remLines[MAX_COOR_REM];
  
  // are we reading or writing from this file?
  int isInput;
  
  // name of the coordinate file
  char *fName;
  
  // person using the file
  char *uName;
  
protected:
  // is the file opened properly and ready for I/O?
  int Initialized;
  
  // file pointer for this file
  FILE *cf;
  
  // atoms in this coor file
  int nAtoms;
  
  // number of frames this file contains, or can contain
  int nFrames;

  // frames processed so far (if 0, none read/written yet)
  int currFrames;
  
  // error flag
  int errorNo;
  
  // add a comment line to the stored list
  void add_comment(char *cline) {
    if(nRemLines < MAX_COOR_REM)
      remLines[nRemLines++] = stringdup(cline);
  }
  
public:
  // constructor and destuctor
  // rd = TRUE if file is being read (input), FALSE if being
  //	written (output)
  CoorFile(int); 
  virtual ~CoorFile(void);
  
  // initialization routine.  Must be called before reading/writing any frames.
  // return TRUE if the file was properly initialized.
  // argument: name of file to read/write
  virtual int init(char *);

  //
  // utility functions
  //

  int errorno(void) { return errorNo; }
  int opened(void) { return Initialized; }
  int atoms(void) { return nAtoms; }
  int tot_frames(void) { return nFrames; }
  int curr_frames(void) { return currFrames; }
  int reading(void) { return isInput; }
  int writing(void) { return !isInput; }
  const char *filename(void) { return (const char *)fName; }
  const char *user(void) { return (const char *)uName; }
};


//////////////////////////// base class for input coor file
class ICoorFile : public CoorFile {

protected:
  //
  // virtual protected routines
  //
  
  // read file header
  virtual void read_header(void);
  
public:
  // constructor/destructor
  ICoorFile(void) : CoorFile(TRUE) { }
  
  // read the next set of coordinates; return list of positions, or NULL if
  // error
  virtual Timestep *read(void) = 0;

};


// base class for output coor file

class OCoorFile : public CoorFile {

protected:
  //
  // virtual protected routines
  //
  
  // write file header
  virtual void write_header(void);
  
public:
  // constructor/destructor
  OCoorFile(void) : CoorFile(FALSE) { }
  
  // write the next set of coordinates; given list of positions, returns total
  // frames written so far
  virtual int write(Timestep *) = 0;

};

#endif
