/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Displayable.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.15 $	$Date: 95/03/24 18:49:05 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Base class for all objects which are drawn in the DisplayDevice.
 * Each Displayable is also a linked list of other Displayables, which act
 * as 'children' to the parent, i.e. they get the parent's tranformation as
 * well as their own.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	Displayable.h,v $
 * Revision 1.15  95/03/24  18:49:05  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.14  1995/03/22  08:05:26  dalke
 * fixed insidious bug involving not knowing the initial
 * cmdList length after freeing linked list
 *
 * Revision 1.13  1995/03/17  22:27:25  billh
 * Added virtual function 'pickable_on' to indicate if a Pickable object
 * is currently being displayed.
 *
 * Revision 1.12  1995/02/22  03:57:58  billh
 * Now supports picking objects with a 2D or 3D pointer.
 * Displayable now derived from Pickable; some virtual functions supplied
 * by Displayble, and also unRegistering a Displayable also unRegisters it
 * from the PickList part of Scene.
 *
 * Revision 1.11  95/02/12  07:17:28  dalke
 * added code to clean the DLINKLIST links in the drawing list
 * 
 * Revision 1.10  1994/12/06  08:33:34  billh
 * All Displayable's now derived from ColorUser ... so each displayable can
 * be given a ColorList object to which color categories can be added,
 * and for each color category color objects can be made that hold a
 * colormap index.  The color editing commands can change which index is
 * associated with a name.
 *
 * Revision 1.9  94/10/12  20:08:16  dalke
 * Added "void set_rot(float x, char axis);"
 * 
 * Revision 1.8  1994/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.7  1994/10/01  03:04:32  billh
 * Removed 'update' routine.  Also made 'draw_prepare' call 'done_preparing'
 * in registered Scene's to signal preparations are done.
 *
 * Revision 1.6  94/09/26  18:54:43  billh
 * Lists of scenes and children now ResizeArray objects.
 * 
 * Revision 1.5  1994/09/23  00:45:43  billh
 * Fixed bug with destructor (was not removing children links).
 * draw_prepare and draw_update call corresponding routines in children.
 *
 * Revision 1.4  94/09/17  09:01:32  billh
 * Fixed bug so that children only register with a scene once.
 * 
 * Revision 1.3  94/09/15  02:05:22  billh
 * Children must get copies of parents transformation matrices at the start.
 * Also made 'set_name' a virtual function, so the Molecule class could use
 * the functionality as well.
 * 
 * Revision 1.2  94/09/14  04:11:33  billh
 * Now allows for 'child' Displayable objects for each Displayable; these
 * can have extra transformation effects, and only register their display
 * command list with a Scene.
 * 
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
#ifndef DISPLAYABLE_H
#define DISPLAYABLE_H

#include "Matrix4.h"
#include "ResizeArray.h"
#include "ColorUser.h"
#include "Pickable.h"
#include "utilities.h"
extern class DispCmd;
extern class Displayable;
extern class Scene;
extern class DisplayDevice;


class Displayable : public ColorUser, public Pickable {

public:
  // method for preparing trans matrix
  enum TransMethod { MULT, LOAD, NONE };

  // where to put the next command in the command list
  void *cmdListPos;

  // where the beginning of drawable command are
  // (this starts with the matrix push)
  void *cmdListBeg;

  // the original scene we were created with
  Scene *origScene;

  // all Scene objects with which we have registered
  ResizeArray<Scene *> sceneList;

  // components of the transformation ... centering, rotating, global trans
  Matrix4 centm, rotm, oldrotm, globm, scalem;

  // main transformation matrix (GT* S * R * CT) for this object
  Matrix4 tm;
  
  // name of this object
  char *name;

  // routine to set the name of the object
  virtual void set_name(char *);
  
private:
  // storage for DispCmd commands ... kept as char array, but will contain 
  // other types as well.
  int initial_block_size;
  //  Place to start the next command
  char *cmdList;
  //  Offset in the current data block
  int cmdListSize;
  //  Size of the current data block
  int cmdListBlockSize;

  // are we currently in the middle of entering a command?
  int entering;

  // this object's tranformation method
  TransMethod transMethod;

  // do we need to recalculate the trans matrix in the next call to 'prepare'?
  int needMatrixRecalc;

  // does all the creation work after variables have been initialized
  void do_create(char *, int, int, TransMethod);

protected:
  // which of the following operations will this object listen to
  int doCent, doRot, doGlob, doScale;

  // should the object be displayed?
  int *displayObj;
  
  // is the object free to be affected by rotations/translations/etc?
  int *isFixed;
  
  // dimension of this displayable (2 or 3)
  int *Dim;
  
  // command to put transformation matrix in our commandlist, and for push
  // and pop commands
  DispCmd *matCmd, *pushCmd, *popCmd;

  // recalculate the transformation matrix, and replace matrix in cmdList
  void recalc_mat(void);

  // our parent Displayable; if NULL, this is a parent Displayable
  Displayable *parent;

  // list of all children Displayable's
  ResizeArray<Displayable *> children;


public:
  // parent constructor ... trans matrix action method (mult, load, none),
  // name, dim (2 or 3), the scene to register with, initial size of cmd list
  Displayable(TransMethod, char *, int, Scene *, int);

  // child constructor ... same as above, but instead of Scene we specify
  // the parent Displayable
  Displayable(TransMethod, char *, int, Displayable *, int);

  // destructor ... will unregister with all scene's.
  virtual ~Displayable(void);

  // register/unregister with the given scene object
  void Register(Scene *);
  void unRegister(Scene *);

  // begin a new command in the command list; arg is the code for the command
  // and the number of bytes needed for just the command (not counting info
  // for the code, size, etc...)
  // returns TRUE if successful (ie., there was no memory problem)
  int start_cmd(int ident_code, int size);

  // end a command in the command list
  // this updates the pointers
  void end_cmd(void);

  // free up the links of a display list and give the data space
  // back to the origScene
  void free_linked_list(char *start);
  // reset the display command list; remove all current commands
  void reset_disp_list(void);

  // signal that a reset of the trans matrix is needed next 'prepare' cycle.
  void need_matrix_recalc(void) { needMatrixRecalc = TRUE; }

  //
  // routines for working as a Pickable
  //
  
  // return our list of draw commands with picking draw commands in them
  virtual void *pick_cmd_list(void);

  // return name of pickable
  virtual char *pick_name(void);

  // return whether the pickable object is being displayed
  virtual int pickable_on(void);
      
  //
  // routines for working with the ColorList
  //
  
  // set up to use a specific ColorList.  This stores the list, and calls
  // a function to do specific action when given a new list
  virtual void use_colors(ColorList *);

  // do action due to the fact that a color for the given ColorList for
  // the specified category has changed
  virtual void color_changed(ColorList *, int);

  //
  // deal with child Displayable's
  //

  // return number of children
  int num_children(void) { return children.num(); }
  
  // return the Nth child pointer
  Displayable *child(int N) { return children[N]; }
  
  // return the index of the given child pointer, or (-1) if none.
  int child_index(Displayable *d) { return children.find(d); }

  // is this a child object?
  int is_child(void) const { return (parent != NULL); }

  // add the given Displayable as a child (assuming it is one)
  void add_child(Displayable *);

  // remove the specified child ... does not delete it.  return success.
  int remove_child(Displayable *);
  int remove_child(int N) { return remove_child(child(N)); }

  //
  // get/set flags for this object
  //
  
  // dimension of this Displayable
  int dim(void) const { return *Dim; }

  // object displayed
  int displayed(void) const { return *displayObj; }
  virtual void off(void);
  virtual void on(void);

  // make the object fixed (not responsive to scale/rot/trans commands
  int fixed(void) const { return *isFixed; }
  void fix(void) {
    *isFixed = TRUE;
    // fix all children
    for(int i=0; i < num_children(); i++)
      child(i)->fix();
  }
  void unfix(void) {
    *isFixed = FALSE;
    // unfix all children
    for(int i=0; i < num_children(); i++)
      child(i)->unfix();
  }

  //
  // preparation/update routines
  //

  // specific prepare for drawing ... do any updates needed right before draw.
  // This version called by draw_prepare, and supplied by derived class.
  virtual void prepare(DisplayDevice *);
  
  // prepare for drawing ... do any updates needed right before draw.
  // this is called by the Scene object to prepare all objects
  void draw_prepare(DisplayDevice *);
  
  //
  // command to set whether to be affected by particular trans routines
  //

  void scale_on(void) { doScale = TRUE; }
  void scale_off(void) { doScale = FALSE; }
  int scaling(void) const { return doScale && !fixed(); }
  
  void rot_on(void) { doRot = TRUE; }
  void rot_off(void) { doRot = FALSE; }
  int rotating(void) const { return doRot && !fixed(); }

  void cent_trans_on(void) { doCent = TRUE; }
  void cent_trans_off(void) {  doCent = FALSE; }
  int cent_translating(void) const { return doCent; }

  void glob_trans_on(void) { doGlob = TRUE; }
  void glob_trans_off(void) { doGlob = FALSE; }
  int glob_translating(void) const { return doGlob && !fixed(); }

  //
  // command to change transformation
  //

  // reset to identity the tranformation ... virtual so resets can affect
  // other factors as well.
  virtual void reset_transformation(void);

  // just change the transformation to the one given
  void load_transformation(Matrix4 &);
  
  // scaling
  void set_scale(float s);
  void mult_scale(float s);
  void set_scale(Matrix4 &);
  void mult_scale(Matrix4 &);
  
  // rotating
  void add_rot(float x, char axis);
  void add_rot(Matrix4 &);
  void set_rot(Matrix4 &);
  void set_rot(float x, char axis);

  // global translation
  void set_glob_trans(float, float, float);
  void set_glob_trans(Matrix4 &);
  void add_glob_trans(float, float, float);
  void add_glob_trans(Matrix4 &);

  // centering translation
  void set_cent_trans(float, float, float);
  void set_cent_trans(Matrix4 &);
  void add_cent_trans(float, float, float);
  void add_cent_trans(Matrix4 &);

};


////////////////////////  2D displayable object  

class Displayable2D : public Displayable {

public:
  // constructor/destructor; just specify 2D object, which does not use the
  // display transformation matrix in any way
  Displayable2D(TransMethod t, char *nm, Scene *scn, int sz)
  	: Displayable(t, nm, 2, scn, sz) { }

  Displayable2D(TransMethod t, char *nm, Displayable *par, int sz)
  	: Displayable(t, nm, 2, par, sz) { }
};

////////////////////////  3D displayable object  

class Displayable3D : public Displayable {

public:
  // constructor/destructor; just specify 3D object, which does not use the
  // display transformation matrix in any way
  Displayable3D(TransMethod t, char *nm, Scene *scn, int sz)
  	: Displayable(t, nm, 3, scn, sz) { }

  Displayable3D(TransMethod t, char *nm, Displayable *par, int sz)
  	: Displayable(t, nm, 3, par, sz) { }
};

#endif
