/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Matrix4.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 95/03/24 18:50:25 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * 4 x 4 Matrix, used for a transformation matrix.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	Matrix4.C,v $
 * Revision 1.6  95/03/24  18:50:25  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.5  1995/02/22  04:07:51  billh
 * Added new constructor for just a float * argument - takes the given
 * single-dim array of floats, assumed to be of proper size, and copies
 * it into internal matrix storage as 4x4 matrix.
 *
 * Revision 1.4  1994/12/09  01:20:49  khamer
 * Added inverse.
 *
 * Revision 1.3  1994/09/29  07:11:38  billh
 * Fixed small bug.
 *
 * Revision 1.2  94/09/28  22:09:22  billh
 * changed fsqrt to sqrtf
 * 
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/Matrix4.C,v 1.6 95/03/24 18:50:25 billh Exp $";
#endif

#include <iostream.h>
#include <math.h>
#include <string.h>
#include "Matrix4.h"
#include "Inform.h"

// constructor, for the case when an array of floating-point numbers is given
Matrix4::Matrix4(float *m) {
  memcpy((void *)mat, (void *)m, MATRIX_DIM * MATRIX_DIM * sizeof(float));
}


// print the matrix out to the given ostream
ostream& operator<<(ostream& os, Matrix4 &m) {
  register int i,j;
  
  for(i=0; i < MATRIX_DIM; i++) {
    os << (i ==0 ? "[" : " ");
    for(j=0; j < MATRIX_DIM; j++)
      os << " " << m.mat[i][j];
    os << (i == (MATRIX_DIM-1) ? " " : "\n");
  }
  os << "]";
  return os;
}


// print the matrix out to the given Inform object
Inform& operator<<(Inform& os, Matrix4 &m) {
  register int i,j;
  
  for(i=0; i < MATRIX_DIM; i++) {
    os << (i ==0 ? "[" : " ");
    for(j=0; j < MATRIX_DIM; j++)
      os << " " << m.mat[i][j];
    os << (i == (MATRIX_DIM-1) ? " " : "\n");
  }
  os << "]";
  return os;
}


// multiplies a 3D point (first arg) by the Matrix, returns in second arg
Matrix4& Matrix4::multpoint3d(float opoint[3], float npoint[3]) {
  register int i;
  float tmp[MATRIX_DIM];

  for (i=0;i<MATRIX_DIM;i++) 
    tmp[i] = opoint[0] * mat[0][i] + opoint[1] * mat[1][i]
	+ opoint[2] * mat[2][i] + mat[3][i];

  for (i=0;i<3;i++)
    npoint[i] = tmp[i] / tmp[3];

  return *this;
}


// multiplies a 4D point (first arg) by the Matrix, returns in second arg
Matrix4& Matrix4::multpoint4d(float opoint[4], float npoint[4]) {
  register int i;
  float tmp[MATRIX_DIM];

  for (i=0;i<MATRIX_DIM;i++) 
    tmp[i] = opoint[0] * mat[0][i] + opoint[1] * mat[1][i]
	+ opoint[2] * mat[2][i] + opoint[3] * mat[3][i];

  for (i=0;i<4;i++)
    npoint[i] = tmp[i];

  return *this;
}


// clears the matrix (resets it to identity)
Matrix4& Matrix4::identity(void) {
  register int i,j;
  
  for(i=0; i < MATRIX_DIM; i++)
    for(j=0; j < MATRIX_DIM; j++)
      mat[i][j] = (i == j ? 1.0 : 0.0);
      
  return *this;
}


// sets the matrix so all items are the given constant value
Matrix4& Matrix4::constant(float f) {
  register int i,j;
  
  for(i=0; i < MATRIX_DIM; i++)
    for(j=0; j < MATRIX_DIM; j++)
      mat[i][j] = f;
      
  return *this;
}

// return the inverse of this matrix, that is, 
// the inverse of the rotation, the inverse of the scaling, and 
// the opposite of the translation vector.
#define SWAP(a,b) {temp=(a);(a)=(b);(b)=temp;}
Matrix4& Matrix4::inverse(const Matrix4& m) {

  Matrix4 ident;
  register int i, icol, irow, j, k, l, ll;
  int indxc[4], indxr[4], ipiv[4];
  float big, dum, pivinv, temp;
  
  // Gauss-jordan elimination with full pivoting.  Yes, folks, a 
  // GL Matrix4 is inverted like any other, since the identity is 
  // still the identity.
  
  // from numerical recipies in C second edition, pg 39

  *this = m;                // *this will be replaced by *this ^-1  
  ident.identity();         // in this case, ident will also be *this^-1
  
  for(j=0;j<=3;j++) ipiv[j] = 0;
  for(i=0;i<=3;i++) {
    big=0.0;
    for (j=0;j<=3;j++) {
      if(ipiv[j] != 1) {
        for (k=0;k<=3;k++) {
          if(ipiv[k] == 0) {
            if(fabs(mat[j][k]) >= big) {
              big=fabs(mat[j][k]);
              irow=j;
              icol=k;
            }
          } else if (ipiv[k] > 1) msgErr << "Singular matrix" << sendmsg;
        } 
      }
    }
    ++(ipiv[icol]);
    if (irow != icol) {
      for (l=0;l<=3;l++) SWAP(mat[irow][l],mat[icol][l]);
      for (l=0;l<=3;l++) SWAP(ident.mat[irow][l],ident.mat[icol][l]);
    }
    indxr[i]=irow;
    indxc[i]=icol;
    if(mat[icol][icol] == 0.0) msgErr << "Singular matrix" << sendmsg;
    pivinv=1.0/mat[icol][icol];
    mat[icol][icol]=1.0;
    for (l=0;l<=3;l++) mat[icol][l] *= pivinv;
    for (l=0;l<=3;l++) ident.mat[icol][l] *= pivinv;
    for (ll=0;ll<=3;ll++) {
      if (ll != icol) {
        dum=mat[ll][icol];
        mat[ll][icol]=0.0;
        for (l=0;l<=3;l++) mat[ll][l] -= mat[icol][l]*dum;
        for (l=0;l<=3;l++) ident.mat[ll][l] -= ident.mat[icol][l]*dum;
      }
    }
  }
  for (l=3;l>=0;l--) {
    if (indxr[l] != indxc[l]) {
      for (k=0;k<=3;k++) {
        SWAP(mat[k][indxr[l]],mat[k][indxc[l]]);
      }
    }
  }
  return *this;
}
      
// replaces this matrix with the given one
Matrix4& Matrix4::loadmatrix(const Matrix4& m) {
  register int i,j;

  for(i=0; i < MATRIX_DIM; i++)
    for(j=0; j < MATRIX_DIM; j++)
      mat[i][j] = m.mat[i][j];

  return *this;
}


// premultiply the matrix by the given matrix
Matrix4& Matrix4::multmatrix(const Matrix4& m) {
  register int i,j, k;
  Matrix4 mat1(m);		// make copy of first matrix
  Matrix4 mat2(*this);		// make copy of this matrix

  for (i=0;i<MATRIX_DIM;i++) {
    for (j=0;j<MATRIX_DIM;j++) {
      mat[i][j] = 0.0;
      for (k=0;k<MATRIX_DIM;k++)
        mat[i][j] += mat1.mat[i][k] * mat2.mat[k][j];
    }
  }
  
  return *this;
}


// add the given matrix
Matrix4& Matrix4::addmatrix(const Matrix4& m) {
  register int i,j;

  for(i=0; i < MATRIX_DIM; i++)
    for(j=0; j < MATRIX_DIM; j++)
      mat[i][j] += m.mat[i][j];
  
  return *this;
}


// add a specified constant
Matrix4& Matrix4::addconst(float f) {
  register int i,j;

  for (i=0;i<MATRIX_DIM;i++)
    for (j=0;j<MATRIX_DIM;j++)
      mat[i][j] += f;
  
  return *this;
}


// performs a rotation around an axis (char == 'x', 'y', or 'z')
// angle is in degrees
Matrix4& Matrix4::rot(float a, char axis) {
  Matrix4 m(0.0);		// create matrix with all values = 0
  register double angle;

  angle = (double)DEGTORAD(a);

  if (axis == 'x') {
    m.mat[0][0] = 1.0;
    m.mat[1][1] = (float)cos(angle);
    m.mat[1][2] = (float)sin(angle);
    m.mat[2][1] = -m.mat[1][2];
    m.mat[2][2] = m.mat[1][1];
    m.mat[3][3] = 1.0;
  } else if (axis == 'y') {
    m.mat[0][0] = (float)cos(angle);
    m.mat[0][2] = (float) -sin(angle);
    m.mat[1][1] = 1.0;
    m.mat[2][0] = -m.mat[0][2];
    m.mat[2][2] = m.mat[0][0];
    m.mat[3][3] = 1.0;
  } else if (axis == 'z') {
    m.mat[0][0] = (float)cos(angle);
    m.mat[0][1] = (float)sin(angle);
    m.mat[1][0] = -m.mat[0][1];
    m.mat[1][1] = m.mat[0][0];
    m.mat[2][2] = 1.0;
    m.mat[3][3] = 1.0;
  } else {
    cerr << "Matrix4: Error, illegal axis " << axis << " in rotate." << endl;
    return *this;
  }
  
  return multmatrix(m);
}


// performs a translation
Matrix4& Matrix4::translate(float x, float y, float z) {
  Matrix4 m;		// create identity matrix

  m.mat[3][0] = x;
  m.mat[3][1] = y;
  m.mat[3][2] = z;

  return multmatrix(m);
}


// performs scaling
Matrix4& Matrix4::scale(float x, float y, float z) {
  Matrix4 m(0.0);		// create matrix with all values = 0

  m.mat[0][0] = x;
  m.mat[1][1] = y;
  m.mat[2][2] = z;
  m.mat[3][3] = 1.0;

  return multmatrix(m);
}


// sets this matrix to represent a perspective
Matrix4& Matrix4::perspective(short fovy, float aspect, float near,
			      float far) {
  register double angle = (double)DEGTORAD(((float)fovy / 10.0));

  constant(0.0);		// initialize this matrix to 0
  mat[1][1] = (float) (cos(angle/2.0) / sin(angle/2.0));
  mat[0][0] = mat[1][1] / aspect;
  mat[2][2] = -(far+near) / (far-near);
  mat[2][3] = -1.0;
  mat[3][2] = (float) (-(double)(2.0*far*near) / (double)(far-near));

  return *this;  
}


// sets this matrix to represent a window perspective
Matrix4& Matrix4::window(float left, float right, float bottom, 
			 float top, float near, float far) {

  constant(0.0);		// initialize this matrix to 0
  mat[0][0] = (2.0*near) / (right-left);
  mat[1][1] = (2.0*near) / (top-bottom);
  mat[2][0] = (right+left) / (right-left);
  mat[2][1] = (top+bottom) / (top-bottom);
  mat[2][2] = -(far+near) / (far-near);
  mat[2][3] = -1.0;
  mat[3][2] = -(2.0*far*near) / (far-near);

  return *this;  
}


// sets this matrix to a 3D orthographic matrix
Matrix4& Matrix4::ortho(float left, float right, float bottom,
			float top, float near, float far) {

  constant(0.0);		// initialize this matrix to 0
  mat[0][0] = 2.0 / (right-left);
  mat[1][1] = 2.0 / (top-bottom);
  mat[2][2] = -2.0 / (far-near);
  mat[3][0] = -(right+left) / (right-left);
  mat[3][1] = -(top+bottom) / (top-bottom);
  mat[3][2] = -(far+near) / (far-near);
  mat[3][3] = 1.0;

  return *this;  
}


// sets this matrix to a 2D orthographic matrix
Matrix4& Matrix4::ortho2(float left, float right, float bottom, float top) {

  constant(0.0);		// initialize this matrix to 0
  mat[0][0] = 2.0 / (right-left);
  mat[1][1] = 2.0 / (top-bottom);
  mat[2][2] = -1.0;
  mat[3][0] = -(right+left) / (right-left);
  mat[3][1] = -(top+bottom) / (top-bottom);
  mat[3][3] = 1.0;

  return *this;  
}


/* This subroutine defines a viewing transformation with the eye located in
 * polar coordinates looking at the world origin with twist rotation about
 * the line of site.  Precisely, polarview does:
 * polarview = rot(-azim,z)*rot(-inc,x)*rot(-twist,z)*trans(0.0,0.0,-dist)
 */
Matrix4& Matrix4::polarview(float dist, short azim, short inc, short twist) {

  /* premultiply the stack by trans(0.0,0.0,-dist) */
  translate(0.0,0.0,-dist);

  /* premultiply the stack by rotate(-twist,z) */
  rotate(-twist,'z');

  /* premultiply the stack by rotate(-inc,x) */
  rotate(-inc,'x');

  /* premultiply the stack by rotate(-axim,z) */
  rotate(-azim,'z');

  return *this;
}




/* This subroutine defines a viewing transformation with the eye at the point
 * (vx,vy,vz) looking at the point (px,py,pz).  Twist is the right-hand
 * rotation about this line.  The resultant matrix is multiplied with
 * the top of the transformation stack and then replaces it.  Precisely,
 * lookat does:
 * lookat = trans(-vx,-vy,-vz)*rotate(theta,y)*rotate(phi,x)*rotate(-twist,z)
 */
Matrix4& Matrix4::lookat(float vx, float vy, float vz, float px, float py,
			 float pz, short twist) {
  Matrix4 m(0.0);
/*  register double tmp; */
  register float tmp;

  /* pre multiply stack by rotate(-twist,z) */
  rotate(-twist,'z');
  
  m.mat[0][0] = 1.0;
/*
  tmp = sqrt((double) ((px-vx)*(px-vx) + (py-vy)*(py-vy) + (pz-vz)*(pz-vz)));
  m.mat[1][1] = (float)(sqrt((double) ((px-vx)*(px-vx) + (pz-vz)*(pz-vz))) / tmp );
  m.mat[1][2] = (float)( (double)(vy-py) / tmp );
*/
  tmp = sqrtf((px-vx)*(px-vx) + (py-vy)*(py-vy) + (pz-vz)*(pz-vz));
  m.mat[1][1] = sqrtf((px-vx)*(px-vx) + (pz-vz)*(pz-vz)) / tmp;
  m.mat[1][2] = (vy-py) / tmp;
  m.mat[2][1] = -m.mat[1][2];
  m.mat[2][2] = m.mat[1][1];
  m.mat[3][3] = 1.0;
  multmatrix(m);

  /* premultiply by rotate(theta,y) */
  m.constant(0.0);
/*
  tmp = sqrt((double) ((px-vx)*(px-vx) + (pz-vz)*(pz-vz)));
  m.mat[0][0] = (float) ((double)(vz-pz) / tmp );
  m.mat[0][2] = (float) -((double)(px-vx) / tmp );
*/
  tmp = sqrtf((px-vx)*(px-vx) + (pz-vz)*(pz-vz));
  m.mat[0][0] = (vz-pz) / tmp;
  m.mat[0][2] = -(px-vx) / tmp;
  m.mat[1][1] = 1.0;
  m.mat[2][0] = -m.mat[0][2];
  m.mat[2][2] = m.mat[0][0];
  m.mat[3][3] = 1.0;
  multmatrix(m);

  /* premultiply by trans(-vx,-vy,-vz) */
  translate(-vx,-vy,-vz);

  return *this;
}
