/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: NormalScene.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 95/03/24 18:51:02 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Scene object, which maintains a list of Displayable objects and
 * draws them to a DisplayDevice.
 * Each Scene has a list of Displayable objects, and also a list of display
 * commands.  The command lists are used to draw the objects, the Displayable
 * objects to prepare and update objects for drawing.
 *
 * This is the basic version of the Scene, which just uses normal memory
 * for display lists, etc.
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log:	NormalScene.C,v $
 * Revision 1.3  95/03/24  18:51:02  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.2  1994/10/05  04:38:23  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.1  1994/10/01  03:08:43  billh
 * Initial revision
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /private/auto143000131/vmdsrc/vmd/billh/src/RCS/NormalScene.C,v 1.3 95/03/24 18:51:02 billh Exp $";
#endif

#include "NormalScene.h"
#include "DisplayDevice.h"
#include "Inform.h"
#include "utilities.h"


////////////////////////////  constructor  
NormalScene::NormalScene(void)
	: cmdListArray2D(INITLISTSIZE, 2.0), cmdListArray3D(INITLISTSIZE, 2.0),
	  dispObjList2D(INITLISTSIZE, 2.0), dispObjList3D(INITLISTSIZE, 2.0) {

  MSGDEBUG(1,"Creating a Normal Scene ..." << sendmsg);
}


////////////////////////////  destructor  
NormalScene::~NormalScene(void) {
  MSGDEBUG(1,"Deleting (avoiding) a Normal Scene ..." << sendmsg);

  clear();		// unregisters all displayables
}

//////////////////////////  public virtual functions  

// return Nth 2D Displayable object; NULL if not possible
Displayable *NormalScene::displayable2D(int n) {
  if(n >= 0 && n < num_displayable2D())
    return dispObjList2D[n];
  else
    return NULL;
}
  
// return Nth 3D Displayable object; NULL if not possible
Displayable *NormalScene::displayable3D(int n) {
  if(n >= 0 && n < num_displayable3D())
    return dispObjList3D[n];
  else
    return NULL;
}
  
// return Nth 2D command list pointer; NULL if not possible
char *NormalScene::disp_list2D(int n) {
  if(n >= 0 && n < num_disp_list2D())
    return cmdListArray2D[n];
  else
    return NULL;
}
    
// return Nth 3D command list pointer; NULL if not possible
char *NormalScene::disp_list3D(int n) {
  if(n >= 0 && n < num_disp_list3D())
    return cmdListArray3D[n];
  else
    return NULL;
}


// allocate display list storage space of specified size
char *NormalScene::get_disp_storage(int sz) {
  // for this version of the scene, just malloc it
  return new char[sz];
}


// free display list storage space 
void NormalScene::free_disp_storage(char *displist) {

  // for this version of the scene, just delete it
  if(displist)
    delete [] displist;
}


// register the given Displayable and corresponding display list (of dim d)
// If disp is NULL, just store the displist.
void NormalScene::Register(Displayable *disp, char *displist, int d) {

  // make sure we are not already managing this object
  if(disp) {
    int found = (disp->dim() == 3 ? 
  	  dispObjList3D.find(disp) : dispObjList2D.find(disp));
    if(found >= 0) {
      MSGDEBUG(1,"Scene: attempted to register Displayable '" << disp->name);
      MSGDEBUG(1,"' to scene after already registered." << sendmsg);
      return;
    } else {
      MSGDEBUG(1,"Scene: registering Displayable '" << disp->name << "' ...");
      MSGDEBUG(1,sendmsg);
    }

    // add the Displayable pointer to list
    if(disp->dim() == 3) {
      dispObjList3D.append(disp);
      numDisplayable3D = dispObjList3D.num();
    } else {
      dispObjList2D.append(disp);
      numDisplayable2D = dispObjList2D.num();
    }
    MSGDEBUG(2,"     : Now storing " << num_displayable2D() << " 2D and ");
    MSGDEBUG(2,num_displayable3D() << " 3D Displayable's." << sendmsg);  
  }
  
  if(displist) {
    MSGDEBUG(1,"Scene: registering new display list." << sendmsg);
    MSGDEBUG(3,"     : first three values in display list: ");
    MSGDEBUG(3,((int *)displist)[0] << ", " << ((int *)displist)[1] << ", ");
    MSGDEBUG(3,((int *)displist)[2]);
    MSGDEBUG(3,sendmsg);

    // add the display list to our own lists
    if(d == 3) {
      cmdListArray3D.append(displist);
      numCmdLists3D = cmdListArray3D.num();
    } else {
      cmdListArray2D.append(displist);
      numCmdLists2D = cmdListArray2D.num();
    }
    MSGDEBUG(2,"     : Now storing " << num_disp_list2D() << " 2D and ");
    MSGDEBUG(2, num_disp_list3D() << " 3D command lists." << sendmsg);
  }
}


// unregister the given Displayable and corresponding display list (of dim d).
// if disp is NULL, just unregister the given display list
void NormalScene::unRegister(Displayable *disp, char *displist, int d) {
  int found;

  // make sure we are managing this object
  if(disp) {
    found = (disp->dim() == 3 ? 
  	  dispObjList3D.find(disp) : dispObjList2D.find(disp));
    if(found < 0) {
      MSGDEBUG(1,"Scene: attempted to unregister Displayable '" << disp->name);
      MSGDEBUG(1,"' to scene when it is not in the scene." << sendmsg);
      return;
    }
  
    // remove the pointer from list
    if(disp->dim() == 3) {
      dispObjList3D.remove(found);
      numDisplayable3D = dispObjList3D.num();
    } else {
      dispObjList2D.remove(found);
      numDisplayable2D = dispObjList2D.num();
    }
    MSGDEBUG(1,"Scene: unregistered Displayable " << found << ":'");
    MSGDEBUG(1,disp->name << "' ..." << sendmsg);
  }
  
  if(displist) {
    found = (d == 3 ? cmdListArray3D.find(displist) :
    				 cmdListArray2D.find(displist));
    if(found < 0) {
      MSGDEBUG(1,"Scene: attempted to unregister unknown display list");
      MSGDEBUG(1," from scene when it is not in the scene." << sendmsg);
      return;
    }
      
    // remove the pointer from list
    if(d == 3) {
      cmdListArray3D.remove(found);
      numCmdLists3D = cmdListArray3D.num();
    } else {
      cmdListArray2D.remove(found);
      numCmdLists2D = cmdListArray2D.num();
    }
  
    MSGDEBUG(1,"Scene: unregistered display list " << found << sendmsg);
  }
}
