/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: UIObject.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 1995/03/28 03:46:54 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * User Interface Object base class.  All user interface modules are derived
 * from this; it provides methods for registering with the command processor
 * and 'signing up' for which commands it is interested in, as well as
 * generating commands and events.
 *
 ***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: UIObject.h,v $
 * Revision 1.4  1995/03/28  03:46:54  billh
 * Added 'runcommand' routine, to allow a UIObject to request a command be
 * immediately executed instead of going on the queue.
 *
 * Revision 1.3  95/03/24  18:52:24  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.2  1994/10/19  20:23:08  billh
 * Made 'On' and 'Off' virtual routines in every UIObject, and added virtual
 * routine 'is_on' to test for whether the ui is on or off (used mainly by
 * Mouse to provide options to toggle forms on and off)
 *
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
#ifndef UIOBJECT_H
#define UIOBJECT_H

#include "utilities.h"
#include "Inform.h"
extern class UIList;
extern class CommandQueue;
extern class Command;


class UIObject {

public:
  // ways that this object is interested in commands
  enum match {ALL, NONE};
  
  // name of this object
  char *name;

private:
  // the id code for this ui object, as returned by the command processor
  int myID;
  
  // method for matching which commands to use
  match matchMethod;
  
  // is the ui active?
  int uiOn;

  // flag array for the commands we are interested in.  Starts out all set
  // to false, subclasses set the flags for interesting commands.
  int maxCmds;
  char *doCmd;
  
protected:
  // UIList object we've registered with
  UIList *uiL;
  
  // Command Queue to use for new commands
  CommandQueue *cmdQueue;

  // specifically queue a command for later execution.  Return whether
  // the command could be queued.
  int addcommand(Command *);
  
  // send a command to the command queue.  Return command success, or (-1)
  // if the command was queued for later execution
  int runcommand(Command *);

  // note that we are/are not interested in a command
  void command_wanted(int cmd) {
    if(cmd >= 0 && cmd < maxCmds)
      doCmd[cmd] = TRUE;
  }
  
  void command_not_wanted(int cmd) {
    if(cmd >= 0 && cmd < maxCmds)
      doCmd[cmd] = FALSE;
  }

public:
  // constructor and destructor: name, list to register with, match method
  UIObject(char *, UIList *, CommandQueue *, match = ALL);
  virtual ~UIObject(void);
  
  // return id of this interface
  int id(void) { return myID; }
  
  // return matching method
  match match_method(void) { return matchMethod; }
  
  // is the given command one we're interested in?
  int want_command(int cmd) {
    return !(matchMethod != ALL || cmd >= maxCmds || cmd < 0 || !doCmd[cmd]);
  }
  
  //
  // virtual routines to provide information about the UIObject, and turn
  // it on or off
  //

  // is this a menu object?
  virtual int is_menu(void);
   
  // return whether the UI is active
  virtual int is_on(void);

  // turn on UI (allow it to issue commands)
  virtual void On(void);

  // turn off UI (don't allow it to issue commands)
  virtual void Off(void);

  //
  // virtual routines for UI init/display
  //
   
  // initialize this user interface
  virtual void init(void);
  
  // reset the user interface (force update of all info displays)
  virtual void reset(void);
  
  // update the displays
  virtual void update(void);

  //
  // virtual routines for entering/processing commands and events
  //
  
  // update the display due to a command being executed.  Return whether
  // any action was taken on this command.
  // Arguments are the command type, command object, and the 
  // success of the command (T or F).
  virtual int act_on_command(int, Command *, int);
  
  // check for an event, and queue it if found.  Return TRUE if an event
  // was generated.
  virtual int check_event(void);
  
};

#endif
