/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

#include <math.h>
#include "VMDTitle.h"
#include "DisplayDevice.h"

// draws the fancy title for VMD

// functions which put V, M, and D on the display list
// The max height is y+1 (and the min is y+0, since they
// are all capital letters).  The leftmost (on x)
// graphics starts at x.  However, different letters
// are different widths.
//  The width of a stroke is STROKE_WIDTH.
#define STROKE_WIDTH (0.2 + 0.0)


// given the front surface coordinates(4) in counter clockwise direction
// draw the solid box that goes around the box with that
// surface, and a depth of 0.25 deep
// The optional field requests that the ends also be covered
void draw_outsides(float c[8][3], Displayable *disp, int draw_ends = TRUE)
{
  float zoffset[3] = { 0, 0, 0.25}; // make the other 4 coords
  for (int i=0; i<4; i++)
    subtract(c[i+4], c[i], zoffset);
    
  DispCmdTriangle tri;
    // draw the front parallegram
  tri.putdata(c[0], c[1], c[2], disp);
  tri.putdata(c[2], c[3], c[0], disp);
    // draw the rear parallegram
  tri.putdata(c[5], c[4], c[6], disp);
  tri.putdata(c[6], c[4], c[7], disp);
  
    // draw the left face
  tri.putdata(c[0], c[3], c[7], disp);
  tri.putdata(c[7], c[4], c[0], disp);
    // draw the right face
  tri.putdata(c[1], c[5], c[6], disp);
  tri.putdata(c[6], c[2], c[1], disp);

  if (!draw_ends)
    return;
    
    // draw the top face
  tri.putdata(c[0], c[5], c[4], disp);
  tri.putdata(c[5], c[0], c[1], disp);
  
    // draw the bottom face
  tri.putdata(c[2], c[7], c[3], disp);
  tri.putdata(c[7], c[2], c[6], disp);
  
}


// draw a parallelpiped from start to end
//  the top and bottoms are flat
//  the front parallegram marks out the regions
//   (startx-STROKE_WIDTH/2, starty) to 
//   (startx+STROKE_WIDTH/2, starty) to 
//   (endx-STROKE_WIDTH/2, endy) to 
//   (endx+STROKE_WIDTH/2, endy) and back
void draw_stroke(float startx, float starty, float endx, float endy,
                  Displayable *disp)
{
  float c[8][3];
  c[0][0] = startx - (STROKE_WIDTH+0.0)/(2.0);
  c[0][1] = starty; 
  c[0][2] = 0;
  c[1][0] = startx + (STROKE_WIDTH+0.0)/(2.0);
  c[1][1] = starty; 
  c[1][2] = 0;
  c[2][0] = endx + (STROKE_WIDTH+0.0)/(2.0);
  c[2][1] = endy; 
  c[2][2] = 0;
  c[3][0] = endx - (STROKE_WIDTH+0.0)/(2.0);
  c[3][1] = endy; 
  c[3][2] = 0;
  draw_outsides(c, disp, TRUE);
}
void draw_letter_V(float x, float y, Displayable *disp)
{
  // draw the down stroke (note the order, this is because of the 
  //  way I define the coordinates in the draw_stroke routine for width)
  draw_stroke(x+STROKE_WIDTH*(1.5), y+0, x+STROKE_WIDTH/2.0, y+1.0, disp);
  // and the up stroke   
  draw_stroke(x+STROKE_WIDTH*(1.5), y+0, x+STROKE_WIDTH*(2.5), y+1, disp);
}
float letter_V_width(void)
{
 return STROKE_WIDTH*2;
}

void draw_letter_M(float x, float y, Displayable *disp)
{
  // up
  draw_stroke(x+STROKE_WIDTH/2.0, y+0, x+STROKE_WIDTH*(1.5), y+1, disp);
  // down
  draw_stroke(x+STROKE_WIDTH*(2.5), y+0, x+STROKE_WIDTH*(1.5), y+1, disp);
  // up
  draw_stroke(x+STROKE_WIDTH*(2.5), y+0, 
              x+STROKE_WIDTH*(3.5), y+1, disp);
  // down
  draw_stroke(x+STROKE_WIDTH*(4.5), y+0,
              x+STROKE_WIDTH*(3.5), y+1, disp);
}
float letter_M_width(void)
{
 return STROKE_WIDTH*5;
}

// given the angle and the major/minor radii for the ellipse
//   angle = 0 => along major axis
//   increasing angle is in the clockwise direction
// find the x, y point of the axis at that angle (with z=0)
void find_ellipse_coords(float angle, float major, float minor, float *data)
{
  data[0] = minor * sinf(angle);
  data[1] = major * cosf(angle);
  data[2] = 0;
}


// This one is tricky
void draw_letter_D(float x, float y, Displayable *disp)
{
  // up
  draw_stroke(x+STROKE_WIDTH/2.0, y+0, x+STROKE_WIDTH/2.0, y+1, disp);
  // and the curve
  float stepsize = M_PI/15;
  float offset[3] = {STROKE_WIDTH, 0.5, 0};
  offset[0] += x;
  offset[1] += y;
  float c[8][3];
  int i, j;
  for (i=0; i<15; i++) {
     // inside coords
    find_ellipse_coords(stepsize * i, 0.5 - STROKE_WIDTH, STROKE_WIDTH,
                        c[1]);
    find_ellipse_coords(stepsize * (i+1), 0.5 - STROKE_WIDTH, STROKE_WIDTH,
                        c[2]);
     // outside coords
    find_ellipse_coords(stepsize * i, 0.5 , STROKE_WIDTH * 2,
                        c[0]);
    find_ellipse_coords(stepsize * (i+1), 0.5 , STROKE_WIDTH * 2,
                        c[3]);
    // add the offsets
    for (j=0; j<4; j++)
      add(c[j], c[j], offset);
    draw_outsides(c, disp, FALSE);
  }
}

float letter_D_width(void)
{
 return STROKE_WIDTH + STROKE_WIDTH * 1.5;
}

/// And now the class definition
VMDTitle::VMDTitle(Scene *sc) : Displayable3D(MULT, "VMD Title", sc, 15),
   txt1("Theoretical Biophysics"),
   txt2("University of Illinois"),
   txt3("by   Andrew Dalke"),
   txt4("     Bill Humphrey"),
   txt5("     Rick Kufrin (NCSA)")
{
  // displayable characteristics
  rot_off();
  scale_off();
  glob_trans_off();
  cent_trans_off();
  redraw_list();
  
  glob_trans_on();
  set_glob_trans(0, 0.5, 0);
  glob_trans_off();
  timer.start();
}

void VMDTitle::redraw_list(void)
{
  reset_disp_list();
  float x;  // offset to center VMD
  x = 0 + letter_V_width() + STROKE_WIDTH/2.0 + letter_V_width() + 
          STROKE_WIDTH/2.0 + letter_M_width();
  x=x/2;
  materials.putdata(TRUE, this);
  color.putdata(REGRED, this);
  draw_letter_V(-x, -0.5, this);
  color.putdata(REGGREEN, this);
  draw_letter_M(-x + STROKE_WIDTH/2.0 + letter_V_width(),-0.5, this);
  color.putdata(REGBLUE, this);
  draw_letter_D(-x + STROKE_WIDTH/2.0 + letter_V_width() +
                     STROKE_WIDTH/2.0 + letter_M_width(),-0.5, this);
  materials.putdata(FALSE, this);
  if (letterson) {
    color.putdata(REGWHITE, this);
    float pos[3];
    pos[0] = -x*.6; pos[1] = -0.8; pos[2] = 0;
    txtpos.putdata(pos, this);
    txt1.put(this);
    pos[1] -= 0.1;txtpos.putdata(pos, this);
    txt2.put(this);
    pos[1] -= 0.1;txtpos.putdata(pos, this);
    txt3.put(this);
    pos[1] -= 0.1;txtpos.putdata(pos, this);
    txt4.put(this);
    pos[1] -= 0.1;txtpos.putdata(pos, this);
    txt5.put(this);
  }
}

// simply fits a 2rd degree polynomial so that v(d=0)=0=v(d=t)
// and x(d=0)=start and x(d=t)=end
static float solve_position( float d, float t, float start, float end)
{
  float a = 6.*(end-start)/(t*t*t);
  return a*d*d*(t/2-d/3)+start;
}
void VMDTitle::prepare(DisplayDevice *display)
{
  float elapsed = timer.clock_time();
  float delta;
  if (elapsed < 5 + 3) {  // just sit there
    if (!letterson) {
      letterson = TRUE;
      redraw_list();
    }
    return;
  }
  elapsed -= 3;
  if (letterson) {
    letterson = FALSE;
    redraw_list();
  }

  if (elapsed < 30) { // do just the rotations
    delta = elapsed - 5;
    rot_on();
    set_rot( solve_position( delta, 25, 0, 360*8), 'y');
    rot_off();
  }
  if (elapsed < 15) { 
    delta = elapsed - 5;
    scale_on();
    set_scale( 1/(1+delta/3)); // and getting smaller
    scale_off();
    glob_trans_on();
    set_glob_trans(0, 0.5, solve_position(delta, 10, 0, 0.5)); // and moving up
    glob_trans_off();
    return;
  }
  if (elapsed < 20) {
    return;
  }
  // I am at          ( 0  ,  0.5, 0.5)
  // I want to get to ( -.7  ,  0.9  , 0.5) in 10 secs
  if (elapsed < 30) {
    delta = elapsed - 20;
    glob_trans_on();
    set_glob_trans(
       solve_position( delta, 10, 0, -.7 * display->aspect()),
       solve_position( delta, 10, 0.5, 0.9),
       solve_position( delta, 10, 0.5, 0.5));
    glob_trans_off();
    scale_on();
    set_scale( solve_position( delta, 10, 1/(1+10./3.), 0.25));
    scale_off();
    return;
  }
  if (elapsed < 35) return;
  // just rotate
  delta = elapsed - 35;
  rot_on();
  set_rot(delta * 360. / 6., 'y');  
  rot_off();
}
