/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: AtomColor.h,v $
 *      $Author: billh $        $Locker:  $                $State: Exp $
 *      $Revision: 1.7 $      $Date: 1995/05/11 21:36:14 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Parse and maintain the data for how a molecule should be colored.
 *
 ***************************************************************************/
#ifndef ATOMCOLOR_H
#define ATOMCOLOR_H

#include "utilities.h"
class MoleculeList;
class DrawMolecule;
class ColorList;

// default atom coloring method, and max cmd string size
#define DEFAULT_ATOMCOLOR	AtomColor::NAME
#define MAX_ATOMCOLOR_CMD	255

class AtomColor {

public:
  // different methods for coloring atoms
  enum ColorMethod { NAME, TYPE, RESNAME, RESTYPE, RESID, SEGNAME, MOLECULE,
  	INDEX, BETA, OCCUP, MASS, CHARGE, POS, VEL, CHAIN, BACKBONE, TOTAL };
	
  // initial string with representation command
  char cmdStr[MAX_ATOMCOLOR_CMD + 1];
  
  // color determined for each atom
  int *color;

  // number of atoms these colors are for
  int nAtoms;

private:
  // results of command ... how to represent atoms, and some parameters
  ColorMethod colorMethod;
  int colIndex;			// index, if method = INDEX
  int transpar;			// use transparent color?
  
  // color list object this object should use
  ColorList *colList;

  // the list of molecules to use for data ... may be NULL
  MoleculeList *molList;

  // molecule used to base the selection on
  DrawMolecule *mol;

  // parse the given command, and store results.  Return success.
  int parse_cmd(char *);

public:
  AtomColor(MoleculeList *, ColorList *);
  AtomColor(AtomColor &);
  ~AtomColor(void);
  
  // equal operator, to change the current settings.
  AtomColor& operator=(const AtomColor &);

  // return whether the given colorlist and color category index are the
  // settings for the current coloring method
  int current_color_use(ColorList *, int);

  // find the color index for the atoms of the given molecule. Return success.
  int find(DrawMolecule *);

  // provide new settings; does a 'find' at the end if a mol has
  // been previously provided.
  int change(char *newcmd) {
    int retval = parse_cmd(newcmd);
    if(retval && mol)
      retval = find(mol);
    return retval;
  }

  //
  // info about current settings
  //

  // return representation method
  int method(void) { return colorMethod; }
  
  // return color index (may not be applicable)
  int color_index(void) { return colIndex; }
  
  // return if color should be transparent
  int transparent(void) { return transpar; }

};

// a string array with text descriptions of representation methods
extern char *AtomColorName[AtomColor::TOTAL];

#endif

/* REVISION HISTORY:********************************************************
 *
 * $Log: AtomColor.h,v $
 * Revision 1.7  1995/05/11  21:36:14  billh
 * Moved log message to end of file.
 *
 * Revision 1.6  95/03/24  18:47:01  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.5  1994/12/06  08:26:00  billh
 * Added current_color_use routine, which reports if the given ColorList
 * object and color category are the one currently in use.
 *
 * Revision 1.4  94/11/12  09:56:30  billh
 * Added BACKBONE coloring method.
 * 
 * Revision 1.3  1994/10/26  23:19:44  billh
 * Removed 'ok' member;removed initial string in constructor;added 'cmd_parse'
 * and operator= routines, as well as 'change' routine.
 *
 * Revision 1.2  94/09/24  20:24:42  billh
 * Added 'find' routine so colors are pre-computed; added code to compute
 * the different coloring schemes.
 * 
 * Revision 1.1  94/09/23  06:01:39  billh
 * Initial revision
 * 
 ***************************************************************************/
