/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: BabelConvert.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 1995/05/11 21:36:23 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * A set of routines for manipulating Babel to convert file formats
 * into PDB format
 *
 * Babel is available via anonymous ftp to joplin.biosci.arizona.edu
 ***************************************************************************/

#include <unistd.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <ctype.h>

#include "Inform.h"
#include "Molecule.h"
#include "MoleculeFile.h"
#include "Global.h" // for VMDtempFile

#define BABEL_BIN vmdBabelBin        // something like /usr/local/babel/babel
#define BABEL_TMPDIR VMDtempFile("") // something like /tmp

// given the filename for the original file (like "file.xyz"),
// the frame number (from 0 to num-1) and the knowledge of if there
// are one or many files, 
// return the correct filename.  I need to know the number of
// files since babel converts multi-frame data formats to
// "file00001.pdb", ....
// but single frame formats to "file.pdb"
//  Also, if has_multi==-1, this returns the string needed to delete
// (or find) all files that match this file (eg, "/tmp/vmdbabel*.file.xyz")
//  If has_multi==-2, this converts it into a very special format needed for
// sscanf to parse the dir list to check for the number of files made
const char *babel_file(const char *filename, int idx, int has_multi)
{
   // temp space to save the filename or glob
   static char tempspace[513];

   int i=0;
   const char *s;
   for (s = filename; *s != 0; s++) {  // find the filename
      if (*s == '/') i = s-filename+1;
   }
   // so filename+i points to the actual name

   // if there are multiple files in the conversion, then the output
   // looks like "test0041.extensions".  If there was a single file,
   // the output looks like "test.extensions"
   if (has_multi == -1) {
      sprintf(tempspace, "%svmdbabel*.u%d.%s", BABEL_TMPDIR,
	      getuid(), filename + i);
   } else if (has_multi == -2) {
      char *reallytemp = new char[strlen(filename+i)+1];
      strcpy(reallytemp, filename+i);
      *(reallytemp+strlen(reallytemp)-1) = 0;
      sprintf(tempspace, "vmdbabel%%[0-9.]u%d.%s%%c", 
	      getuid(), reallytemp);
      delete [] reallytemp;
   } else if (has_multi == 0) {
      sprintf(tempspace, "%svmdbabel.u%d.%s", BABEL_TMPDIR,
	      getuid(), filename + i);
   } else {
      sprintf(tempspace, "%svmdbabel%04d.u%d.%s", BABEL_TMPDIR, idx+1, 
	      getuid(), filename + i);
   }
   for (char *ptr = tempspace; *ptr; ptr++) {  // babel makes them lowercase!
      *ptr = tolower(*ptr);                    // grrrrrrr
   }
   return tempspace;
}

// deletes all files that could have been made by "filename"
void babel_delete_all(const char *filename)
{
   const char *s = babel_file(filename, 0, -1); // puts a '*' in number field
   char *t = new char[strlen(s) + strlen("rm -f ") + 5];
   sprintf(t, "/bin/rm -f %s", s);
   system(t);
   delete [] t;
}

// convert the file of the given type to PDB and return the
// number of files which need to be read.  Returns 0 if there
// was an error
int babel_convert(int moltype, const char *filename) {
   char *t;
   // map the type to the proper babel conversion string
//   msgInfo << "Type is : " << moltype << "  for " << filename << sendmsg;
   switch(moltype) {
      case MoleculeFile::T : t = "t"; break;
      case MoleculeFile::PREP : t = "prep"; break;
      case MoleculeFile::BS : t = "bs"; break;
      case MoleculeFile::CAR : t = "car"; break;
      case MoleculeFile::BOOG : t = "boog"; break;
      case MoleculeFile::CACCRT : t = "caccrt"; break;
      case MoleculeFile::CADPAC : t = "cadpac"; break;
      case MoleculeFile::CHARMM : t = "charmm"; break;
      case MoleculeFile::C3D1 : t = "c3d1"; break;
      case MoleculeFile::C3D2 : t = "c3d2"; break;
      case MoleculeFile::CSSR : t = "cssr"; break;
      case MoleculeFile::FDAT : t = "fdat"; break;
      case MoleculeFile::C : t = "c"; break;
      case MoleculeFile::FEAT : t = "feat"; break;
      case MoleculeFile::F : t = "f"; break;
      case MoleculeFile::GAMOUT : t = "gamout"; break;
      case MoleculeFile::G : t = "g"; break;
      case MoleculeFile::GAUOUT : t = "gauout"; break;
      case MoleculeFile::HIN : t = "hin"; break;
      case MoleculeFile::ISIS : t = "isis"; break;
      case MoleculeFile::MACMOL : t = "macmol"; break;
      case MoleculeFile::K : t = "k"; break;
      case MoleculeFile::MICRO : t = "micro"; break;
      case MoleculeFile::MI : t = "mi"; break;
      case MoleculeFile::MO : t = "mo"; break;
      case MoleculeFile::MM3 : t = "mm3"; break;
      case MoleculeFile::MMADS : t = "mmads"; break;
      case MoleculeFile::MDL : t = "mdl"; break;
      case MoleculeFile::MOLEN : t = "molen"; break;
      case MoleculeFile::AC : t = "ac"; break;
      case MoleculeFile::AI : t = "ai"; break;
      case MoleculeFile::AO : t = "ao"; break;
      case MoleculeFile::PC : t = "pc"; break;
      case MoleculeFile::QUANTA : t = "quanta"; break;
      case MoleculeFile::SHELX : t = "shelx"; break;
      case MoleculeFile::SPAR : t = "spar"; break;
      case MoleculeFile::SEMI : t = "semi"; break;
      case MoleculeFile::SPMM : t = "spmm"; break;
      case MoleculeFile::MOL : t = "mol"; break;
      case MoleculeFile::MOL2 : t = "mol2"; break;
      case MoleculeFile::CON : t = "con"; break;
      case MoleculeFile::MACCS2 : t = "maccs2"; break;
      case MoleculeFile::MACCS3 : t = "maccs3"; break;
      case MoleculeFile::UNIXYZ : t = "unixyz"; break;
      case MoleculeFile::X : t = "x"; break;
      case MoleculeFile::XED : t = "xed"; break;
    default:
      return 0;
   }
//   msgInfo << "extension is: " << t << sendmsg;
   char *s = new char[strlen(BABEL_BIN) + 
		      strlen(" -i    -op ") +
		      strlen(filename) +
		      strlen(babel_file(filename, 0, 1)) +
//		      strlen(" > /dev/null") +
		      10];
   sprintf(s, "%s -i%s %s all -op %s",
	   BABEL_BIN, t, filename, babel_file(filename, 0, 0));
//   msgInfo << "Calling: " << s << sendmsg;

   babel_delete_all(filename);
   system(s);
   delete [] s;

   // now find how many frames were printed
   const char *fmt = babel_file(filename, 0, -2);
   DIR *dirp = opendir(BABEL_TMPDIR);
   if (dirp == NULL) {
      return 0;
   }
   struct direct *dp;
   char temps[100];
   char tempc;
   char lastc = *(filename + strlen(filename) -1);
   int count = 0;
   while ((dp = readdir(dirp)) != NULL) {
      if (sscanf(dp->d_name, fmt, temps, &tempc) > 1 && lastc == tempc) {
	 count ++;
//	 msgInfo << dp->d_name << " was GOOD:" << fmt << "## " << temps << sendmsg;
//      } else {
//	 msgInfo << dp->d_name << " was no good:" << fmt << "** " << temps << sendmsg;
      }
//      msgInfo << "last c == " << lastc << "    " << tempc << sendmsg;
   }
   closedir(dirp);
//   msgInfo << "Found ##### " << count << sendmsg;
   return count;
}

/* REVISION HISTORY:********************************************************
 *
 * $Log: BabelConvert.C,v $
 * Revision 1.4  1995/05/11  21:36:23  billh
 * Moved log message to end of file.
 *
 * Revision 1.3  95/04/07  22:32:06  dalke
 * added RCS header
 * 
 ***************************************************************************/
