/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Buttons.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.5 $	$Date: 1995/05/11 21:37:25 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Provides an interface to request a set of buttons of various types to
 * have a common method of query.  A Button object is created by providing
 * a DisplayDevice which knows how to read the state of specific buttons.
 * Each Button object is told which 'bank' of buttons to use, which then
 * appear as buttons 0, 1, and 2 for the user.  There are several different
 * button 'banks':
 *      POINTER_BANK_0 : the mouse buttons (0=left,1=middle,2=right)
 *      POINTER_BANK_1 : an alternate bank of mouse buttons; if the device
 *                       only has one set of mouse buttons, this is the
 *                       same as POINTER_BANK_0.
 *      KEY_BANK_0 : functions keys set #1 (0=F1, 1=F2, 2=F3)
 *      KEY_BANK_1 : functions keys set #2 (0=F5, 1=F6, 2=F7)
 *      KEY_BANK_2 : functions keys set #3 (0=F9, 1=F10, 2=F11)
 *      ESC_BANK : the escape key ... all three buttons are the same (ESC)
 ***************************************************************************/
#ifndef BUTTONS_H
#define BUTTONS_H

#include "utilities.h"
class DisplayDevice;

#define NUM_BUTTONS     3


// class for the different types of button inputs
class Buttons {

public:
  // different button banks we can use
  enum ButtonBanks { POINTER_BANK_0, POINTER_BANK_1, KEY_BANK_0, KEY_BANK_1,
			KEY_BANK_2, ESC_BANK, TOTAL_BUTTON_BANKS };

private:
  // device to ask about button states
  DisplayDevice *dispDev;

  // which bank of buttons to use, (F1, F2, F3) , (F5, ...)
  int myBank;  

  // the first 'real' button code for the current bank
  int bankStart;

  // place to store the button info so I don't constantly repoll the same data
  int state[NUM_BUTTONS];      

  // not FALSE indicated that that button changed since the last polling
  int changes[NUM_BUTTONS];    

  // has anything changed since the last time?  Here for speed
  int has_changed;

public:
  // constructor: device to check for events, which bank of Buttons to use
  Buttons(DisplayDevice *, int = POINTER_BANK_0);

  // number of buttons
  int num(void) { return NUM_BUTTONS; }
  
  // return which bank is being used, or change which bank
  int bank(void) { return myBank; }
  void bank(int);

  // return state of a button -- TRUE is on, FALSE is off
  int button(int buttonnum) { 
    return 0<= buttonnum && buttonnum < num() ?
             state[buttonnum] : FALSE;
  }
  
  // return the value by which a button has changed
  int change(int buttonnum) {
     return 0<= buttonnum && buttonnum < num() ?
              changes[buttonnum] : 0;
  }

  // for speed, I want to know if anything has changed since the last
  // polling w/o actually checking all the state variables
  int changed(void) { return has_changed; }
  
  // get the current button state.  If it changed, generate a Command
  void check_event(void);
};

#endif

/* REVISION HISTORY:********************************************************
 *
 * $Log: Buttons.h,v $
 * Revision 1.5  1995/05/11  21:37:25  billh
 * Moved log message to end of file.
 * Redone entirely; now uses the routine 'button_down' in DisplayDevice
 * to make Buttons completely independent of the windowing system.
 *
 * Revision 1.4  95/03/24  18:47:18  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.3  1994/11/08  09:12:15  dalke
 * set up CaveButtons correctly
 *
 * Revision 1.2  1994/09/17  10:53:18  dalke
 * added RCS header
 *
 ***************************************************************************/
