/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DisplayDevice.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.11 $	$Date: 1995/05/11 22:36:18 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * DisplayDevice - abstract base class for all particular objects which
 *	can process a list of drawing commands and render the drawing
 *	to some device (screen, file, preprocessing script, etc.)
 *
 ***************************************************************************/

#include <math.h>
#include "DisplayDevice.h"
#include "Inform.h"
#include "DispCmds.h"
#include "ColorList.h"
#include "utilities.h"

// static data for this object
static char *stereoNameStr[1] = { "Stereo Off" };

/////////////////////////  constructor and destructor  
DisplayDevice::DisplayDevice(char *nm) : transMat(16) {
  int i;

  // save name
  name = stringdup(nm);

  MSGDEBUG(1,"Creating new DisplayDevice '" << name << "'." << sendmsg);

  // set drawing characteristics default values
  lineStyle = ::SOLIDLINE;
  lineWidth = 1;
  sphereRes = 3;
  sphereMode = ::SOLIDSPHERE;
  backColor[0] = backColor[1] = backColor[2] = 0.0;

  // set scalar values
  has2D = TRUE;
  Dim = 3;
  aaAvailable = cueingAvailable = FALSE;
  aaEnabled = cueingEnabled = FALSE;
  xOrig = yOrig = xSize = ySize = 0;
  pickRegion = 0.01;
  colorCat = (-1);

  // set viewing geometry ... looking from z-axis in negative direction,
  // with 90-degree field of view and assuming the origin is in the
  // center of the viewer's 'screen'
  nearClip = 1.0;
  farClip = 10.0;
  eyePos[0] = eyePos[1] = 0.0;  eyePos[2] = 2.0;
  set_screen_pos(2.0 * eyePos[2], 0.0, 4.0/3.0);

  // set stereo options ... while there is no stereo mode by default,
  // set up normal values for stereo data
  inStereo = 0;
  stereoModes = 1;
  stereoNames = stereoNameStr;
  eyeSep = 0.15;
  eyeDist = eyePos[2];		// assumes viewer on pos z-axis
  eyeDir[0] = -eyePos[0];  eyeDir[1] = -eyePos[1];  eyeDir[2] = -eyePos[2];
  upDir[0] = upDir[2] = 0.0;  upDir[1] = 1.0;
  calc_eyedir();		// determines eye separation direction

  // set internal flags for lights
  for(i=0; i < DISP_LIGHTS; i++)
    lightDefined[i] = lightOn[i] = FALSE;
  for(i=0; i < MAXCOLORS; i++)
    matDefined[i] = FALSE;
  materialOn = (-1);
    
  // load identity matrix onto top of transformation matrix stack
  Matrix4 temp_ident;
  transMat.push(temp_ident);
}


// destructor
DisplayDevice::~DisplayDevice(void) {

  set_stereo_mode(0);		// return to non-stereo, if necessary
  delete [] name;
}


/////////////////////////  protected nonvirtual routines  
// calculate the position of the near frustrum plane, based on current values
// of Aspect, vSize, zDist, nearClip and eyePosition
// Assumes the viewer is looking toward the xy-plane
void DisplayDevice::calc_frustrum(void) {
  float d, halfvsize = 0.5 * vSize;
  float halfhsize = Aspect * halfvsize;

  if(eyePos[2] - zDist != 0.0) {
    d = nearClip / (eyePos[2] - zDist);
    cpRight = d * halfhsize;
    cpLeft = -cpRight;
    cpUp = d * halfvsize;
    cpDown = -cpUp;
  }
}


// calculate eyeSepDir, based on up vector and look vector
// eyeSepDir = 1/2 * eyeSep * (lookdir x updir) / mag(lookdir x updir)
void DisplayDevice::calc_eyedir(void) {
  float *L = eyeDir;
  float *U = upDir;
  float m, A = 0.5 * eyeSep;
  eyeSepDir[0] = L[1] * U[2] - L[2] * U[1];
  eyeSepDir[1] = L[2] * U[0] - L[0] * U[2];
  eyeSepDir[2] = L[0] * U[1] - L[1] * U[0];
  m = sqrtf(eyeSepDir[0] * eyeSepDir[0] + eyeSepDir[1] * eyeSepDir[1] +
            eyeSepDir[2] * eyeSepDir[2]);
  if(m > 0.0)
    A /= m;
  else
    A = 0.0;
  eyeSepDir[0] *= A;
  eyeSepDir[1] *= A;
  eyeSepDir[2] *= A;
}

/////////////////////////  protected virtual routines (default values)
// virtual routines to deal with light sources and materials at device level
// these just return TRUE, but do nothing
int DisplayDevice::do_define_light(int, float *, float *) {
  return TRUE;
}

int DisplayDevice::do_activate_light(int, int) {
  return TRUE;
}

int DisplayDevice::do_define_material(int, float *) {
  return TRUE;
}

int DisplayDevice::do_activate_material(int, int) {
  return TRUE;
}

// virtual function which is called when a new colorlist is provided.
// This should NOT have to be changed by derived classes.
void DisplayDevice::do_use_colors(void) {

  // create a display color category ... if it already exists, we'll get
  // it here
  colorCat = colorList->add_color_category("Display");
  
  // add colors to the category, if they do not yet exist
  if((colorList->color_category(colorCat))->typecode("Background") < 0)
    (colorList->color_category(colorCat))->add_name("Background", REGBLACK);

  // update the colors by indicating it has changed
  do_color_changed(colorList, colorCat);
}

// do action due to the fact that a color for the given ColorList for
// the specified category has changed
// This should NOT have to be changed by derived classes.
void DisplayDevice::do_color_changed(ColorList *collist, int ccat) {
  if(collist == colorList && ccat == ccat) {
    // must be changing the background, since this is the only color here
    float *bgcolor = colorList->color(
    	(colorList->color_category(colorCat))->data("Background"));
    set_background(bgcolor[0], bgcolor[1], bgcolor[2]);
  }
}


/////////////////////////  public nonvirtual routines  

// Copy all relevant properties from one DisplayDevice to another
DisplayDevice& DisplayDevice::operator=( DisplayDevice &display) {
  int i,j;
  
  colorCat = display.colorCat;
  
  xOrig = display.xOrig;
  yOrig = display.yOrig;
  xSize = display.xSize;
  ySize = display.ySize;
  
  // Do something about the stack.  For the moment, only copy the top
  // item on the stack.
  transMat.push( (display.transMat).top() );
  
  for(i=0; i<3; i++) {
      eyePos[i] = display.eyePos[i];
      eyeDir[i] = display.eyeDir[i];
      upDir[i] = display.upDir[i];
      eyeSepDir[i] = display.eyeSepDir[i];
      backColor[i] = display.backColor[i];
  }
  
  nearClip = display.nearClip;
  farClip = display.farClip;
  vSize = display.vSize;
  zDist = display.zDist;
  Aspect = display.Aspect;
  cpUp = display.cpUp;
  cpDown = display.cpDown;
  cpLeft = display.cpLeft;
  cpRight = display.cpRight;
  inStereo = display.inStereo;
  eyeSep = display.eyeSep;
  eyeDist = display.eyeDist;
  lineStyle = display.lineStyle;
  lineWidth = display.lineWidth;
  
  for(i=0;i<DISP_LIGHTS;i++) {
      lightDefined[i] = display.lightDefined[i];
      lightOn[i] = display.lightOn[i];
      for(j=0;j<3;j++) {
          lightColor[i][j] = display.lightColor[i][j];
          lightPos[i][j] = display.lightPos[i][j];
      }
  }
  
  for(i=0;i<MAXCOLORS;i++) {
      matDefined[i] = display.matDefined[i];
      for(j=0;j<COLOR_ITEMS;j++) {
          matData[i][j] = display.matData[i][j];
      }
  }
  
  materialOn = display.materialOn;
  materialsActive = display.materialsActive;
  
  return *this;
}


// return relative 2D screen coordinates, given 2D or 3D world coordinates.
void DisplayDevice::rel_screen_loc_3D(float *wloc, float *sloc) {
  // first, find absolute screen coordinates
  long abscoords[2];
  abs_screen_loc_3D(wloc, abscoords);
  
  // now convert these to relative coordss
  sloc[0] = (float) (abscoords[0]);
  sloc[1] = (float) (abscoords[1]);
  rel_screen_pos(sloc[0], sloc[1]);
}

void DisplayDevice::rel_screen_loc_2D(float *wloc, float *sloc) {
  // first, find absolute screen coordinates
  long abscoords[2];
  abs_screen_loc_2D(wloc, abscoords);
  
  // now convert these to relative coordss
  sloc[0] = (float) (abscoords[0]);
  sloc[1] = (float) (abscoords[1]);
  rel_screen_pos(sloc[0], sloc[1]);
}


//
// routines to deal with light sources
//
  
void DisplayDevice::define_light(int n, float *color, float *position) {
  int i;
  
  if(n < 0 || n >= DISP_LIGHTS)
    return;
  
  // save data for future reference
  for(i=0; i < 3; i++) {
    lightColor[n][i] = color[i];
    lightPos[n][i] = position[i];
  }
  
  // have derived class actually do the action
  lightDefined[n] = do_define_light(n, color, position);
}

void DisplayDevice::activate_light(int n, int turnon) {
  if(n < 0 || n >= DISP_LIGHTS || !lightDefined[n])
    return;
    
  if(do_activate_light(n, turnon))
    lightOn[n] = turnon;
}


//
// routines to deal with material definitions
//

void DisplayDevice::define_material(int n, float *data) {

  if(n < 0 || n >= MAXCOLORS)
    return;
    
  // save data for future reference
  for(int i=0; i < COLOR_ITEMS; i++)
    matData[n][i] = data[i];

  // have derived class actually do the action
  matDefined[n] = do_define_material(n, data);
}

// turn on the specified material
void DisplayDevice::activate_material(int n) {

  if(n < 0 || n >= MAXCOLORS)
    return;

  if(do_activate_material(n, TRUE)) {
    materialOn = n;
    materialsActive = TRUE;
  }
}

// turn on the current material (if any)
void DisplayDevice::activate_material(void) {
  int n = materialOn;

  if(n < 0 || n >= MAXCOLORS || !matDefined[n])
    return;

  if(do_activate_material(n, TRUE))
    materialsActive = TRUE;
}

// turn off the use of materials
void DisplayDevice::deactivate_materials(void) {

  if(!materialsActive)
    return;
    
  if(do_activate_material(materialOn, FALSE))
    materialsActive = FALSE;
}

/////////////////////////  public virtual routines

//
// event handling routines
//

// queue the standard events (need only be called once ... but this is
// not done automatically by the window because it may not be necessary or
// even wanted)
void DisplayDevice::queue_events(void) { return; }

// test if there is an event ready
int DisplayDevice::test_events(void) { return FALSE; }

// read the next event ... returns an event type (one of the above ones),
// and a value.  Returns success, and sets arguments.
int DisplayDevice::read_event(long &, long &) { return WIN_NOEVENT; }


//
// get the current state of the device's pointer (i.e. cursor if it has one)
//

// abs pos of cursor from lower-left corner
int DisplayDevice::x(void) { return 0; }

// same, for y direction
int DisplayDevice::y(void) { return 0; }

// whether a button is currently pressed
int DisplayDevice::button_down(int) { return FALSE; }

// allow the user to define a special shape for the cursor ... the data
// is assumed to consist of a single-dim array of 32 unsigned shorts; the
// first 16 define a 16x16-bit (row-major) pattern, the last 16 are a
// "second layer" which may be drawn in a different color
// args: which shape (>0, with 0 the "default" case which cannot be changed)
// the bitmap data, and the "hot spot" from the lower-left corner
void DisplayDevice::change_cursor_shape(int, unsigned short *, int, int) { }

// set the Nth cursor shape as the current one.  If no arg given, the
// default shape (n=0) is used.
void DisplayDevice::set_cursor(int) { }


//
// virtual routines to create and activate a popup (possibly pull-down) menu 
//

// given a PopupMenu definition, create it in the window (but do not
// activate it yet).  Return success.
int DisplayDevice::menu_create(PopupMenu *) { return TRUE; }

// activate a previously-created menu.  If the windowing system has no
// current menu, or cannot do this operation, returns (-1); otherwise,
// returns the return code for the select (-1 if none selected).
int DisplayDevice::menu_activate(void) { return NULL; }

// delete the given menu from the display.
// If no argument is given, deletes the current menu (if any).
// Returns success.
int DisplayDevice::menu_delete(PopupMenu *) { return TRUE; }

// return the eye position to it's original position
void DisplayDevice::reset_eye(void) {
  move_eye(0.0, 0.0, 2.0);
  change_up(0.0, 1.0, 0.0);
  change_look(0.0, 0.0, - eye_dist());
}


// virtual functions to turn on/off depth cuing and antialiasing
void DisplayDevice::aa_on(void) { }
void DisplayDevice::aa_off(void) { }
void DisplayDevice::cueing_on(void) { }
void DisplayDevice::cueing_off(void) { }


// return absolute 2D screen coordinates, given 2D or 3D world coordinates.
void DisplayDevice::abs_screen_loc_3D(float *wloc, long *sloc) {
  // just return world coords
  for(int i=0; i < 2; i++)
    sloc[i] = (long)(wloc[i]);
}

void DisplayDevice::abs_screen_loc_2D(float *wloc, long *sloc) {
  // just return world coords
  for(int i=0; i < 2; i++)
    sloc[i] = (long)(wloc[i]);
}

// size of physical display object
void DisplayDevice::screen_size_mm(long& x, long& y) {
  x = y = 1000;		// just say it's a meter
}

void DisplayDevice::screen_size_pixel(long& x, long& y) {
  x = y = 1000;		// just say 1000 pixels
}


// Given a 3D point (pos A),
// and a 2D rel screen pos point (for pos B), computes the 3D point
// which goes with the second 2D point at pos B.  Result returned in 3rd arg.
void DisplayDevice::find_3D_from_2D(float *, float *, float *) { }
/*
// Given a 3D point (pos A),
// and a 2D rel screen pos point (for pos B), computes the 3D point
// which goes with the second 2D point at pos B.  Result returned in B3D.
void DisplayDevice::find_3D_from_2D(float *A2D, float *A3D,
					float *B2D, float *B3D) {
  float tmpB2D[2], trans2D[3], trans2Dtmp[3], trans3Dtmp[3];
  
  // Add trans to A3D in X&Y dir equal to difference between given 2D points
  trans2D[0] = B2D[0] - A2D[0];
  trans2D[1] = B2D[1] - A2D[1];
  trans2D[2] = 0.0;
  add(B3D, A3D, trans2D);
  
  // now convert the translated 3D point to rel screen coords
  rel_screen_loc_3D(B3D, tmpB2D);
  
  // compute difference between new 2D point and starting point
  trans2Dtmp[0] = tmpB2D[0] - A2D[0];
  trans2Dtmp[1] = tmpB2D[1] - A2D[1];
  trans2Dtmp[2] = 0.0;

  // from ratio of 2D differences, times change in 3D tmp difference, get
  // change in 3D for wanted coordinates
  subtract(trans3Dtmp, B3D, A3D);
  B3D[0] = (trans2Dtmp[0] != 0.0 ?
  	A3D[0] + (trans3Dtmp[0] / trans2Dtmp[0]) * trans2D[0] :  A3D[0]);
  B3D[1] = (trans2Dtmp[1] != 0.0 ?
  	A3D[1] + (trans3Dtmp[1] / trans2Dtmp[1]) * trans2D[1] :  A3D[1]);
  B3D[2] = A3D[2];
}
*/




// process list of draw cmds
void DisplayDevice::render(void *) {
  MSGDEBUG(2, "Default Display - skipping rendering of draw commands.");
  MSGDEBUG(2, sendmsg);
}


// change to a different stereo mode (0 means 'off')
void DisplayDevice::set_stereo_mode(int sm) {
  if(sm != 0) {
    msgErr << "DisplayDevice: Illegal stereo mode " << sm << " specified."
           << sendmsg;
  } else {
    inStereo = sm;
  }
}

// save the current transformation matrix ... by default, a copy is
// kept in this object, but derived classes may handle the trans matrix
// differently
void DisplayDevice::push(void) {
  transMat.push();
}

// restore the current transformation matrix
void DisplayDevice::pop(void) {
  transMat.pop();
}

// replace the current trans matrix with the given one
void DisplayDevice::loadmatrix(Matrix4 &m) {
  (transMat.top()).loadmatrix(m);
}

// multiply the current trans matrix with the given one
void DisplayDevice::multmatrix(Matrix4 &m) {
  (transMat.top()).multmatrix(m);
}


//
// virtual routines for preparing to draw, drawing, and finishing drawing
//
  
void DisplayDevice::prepare2D(int) { }		// ready to draw 2D
void DisplayDevice::prepare3D(int) { }		// ready to draw 3D
void DisplayDevice::clear(void) { }		// erase the device
void DisplayDevice::left(void) { }		// ready to draw left eye
void DisplayDevice::right(void) { }		// ready to draw right eye
void DisplayDevice::normal(void) { }		// ready to draw non-stereo
void DisplayDevice::update(int) { }		// finish up after drawing
void DisplayDevice::reshape(void) { }		// refresh device after change
  

//
//*******************  the picking routine  *********************
//
// This scans the given command list until the end, finding which item is
// closest to the given pointer position.
//
// arguments are dimension of picking (2 or 3), position of pointer,
// draw command list, and returned distance from object to eye position.
// Returns ID code ('tag') for item closest to pointer, or (-1) if no pick.
// If an object is picked, the eye distance argument is set to the distance
// from the display's eye position to the object (after its position has been
// found from the transformation matrix).  If the value of the argument when
// 'pick' is called is <= 0, a pick will be generated if any item is near the
// pointer.  If the value of the argument is > 0, a pick will be generated
// only if an item is closer to the eye position than the value of the
// argument.
// For 2D picking, coordinates are relative position in window from
//	lower-left corner (both in range 0 ... 1)
// For 3D picking, coordinates are the world coords of the pointer.  They
//	are the coords of the pointer after its transformation matrix has been
//	applied, and these coordinates are compared to the coords of the objects
//	when their transformation matrices are applied.

// but first, a macro for returning the distance^2 from the eyepos to the
// given position
#define DTOEYE(x,y,z) ( (x-eyePos[0])*(x-eyePos[0]) + \
			(y-eyePos[1])*(y-eyePos[1]) + \
			(z-eyePos[2])*(z-eyePos[2]) )

int DisplayDevice::pick(int dim, float *pos, void *cmdList, float &eyedist) {
  char *cmdptr = (char *)cmdList;
  int tok, cmdsize;
  float newEyeDist, currEyeDist = eyedist;
  int tag = (-1), inRegion, currTag, minX, minY, maxX, maxY;
  float fminX, fminY, fminZ, fmaxX, fmaxY, fmaxZ;
  float pntpos[3];
  long cpos[2];

  MSGDEBUG(3, "DisplayDevice: checking for a picked object." << sendmsg);
  
  if(!cmdList)
    return (-1);
  
  // initialize picking: find screen region to look for object
  if(dim == 2) {
    fminX = pos[0] - pick_region();
    fmaxX = pos[0] + pick_region();
    fminY = pos[1] - pick_region();
    fmaxY = pos[1] + pick_region();
    abs_screen_pos(fminX, fminY);
    abs_screen_pos(fmaxX, fmaxY);
    minX = (int)fminX;
    maxX = (int)fmaxX;
    minY = (int)fminY;
    maxY = (int)fmaxY;
  } else {
    fminX = pos[0] - 5.0 * pick_region();
    fmaxX = pos[0] + 5.0 * pick_region();
    fminY = pos[1] - 5.0 * pick_region();
    fmaxY = pos[1] + 5.0 * pick_region();
    fminZ = pos[2] - 5.0 * pick_region();
    fmaxZ = pos[2] + 5.0 * pick_region();
  }

  // make sure we do not disturb the regular transformation matrix
  transMat.push();

  // scan through the list, getting each command and executing it, until
  // the end of commands token is found
  float *dataBlock = NULL;
  while((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
    if (tok == DLINKLIST) {
       cmdptr += sizeof(int);       // step forward one (no length here)
       cmdptr = *((char **) cmdptr);// and go to the next link
       tok = ((int *)cmdptr)[0];    // this is guarenteed to be neither
    }                               // a DLINKLIST nor DLASTCOMMAND
       
    cmdsize = ((int *)cmdptr)[1];
    cmdptr += 2*sizeof(int);

    MSGDEBUG(4, "DisplayDevice: doing pick command " << tok);
    MSGDEBUG(4, " of size " << cmdsize << sendmsg);

    if(tok == DPICKPOINT || tok == DPICKPOINT_I) {
    
      // calculate the transformed position of the point
      if(tok == DPICKPOINT) {
        (transMat.top()).multpoint3d((float *)cmdptr, pntpos);
	currTag = (int) ( ((float *)cmdptr)[3]);
      } else {
        (transMat.top()).multpoint3d(dataBlock + ((int *)cmdptr)[0], pntpos);
 	currTag = ((int *)cmdptr)[1];
      }

      // check if in picking region ... different for 2D and 3D
      if(dim == 2) {
        // convert the 3D world coordinate position to 2D absolute screen coord
	abs_screen_loc_3D(pntpos, cpos);
      
        // check to see if the position falls in our picking region
        inRegion = (cpos[0] >= minX && cpos[0] <= maxX &&
			cpos[1] >= minY && cpos[1] <= maxY);
      } else {
        // just check to see if the position is in a box centered on our
	// pointer.  The pointer position should already have been transformed.
	inRegion = (pntpos[0] >= fminX && pntpos[0] <= fmaxX &&
			pntpos[1] >= fminY && pntpos[1] <= fmaxY &&
			pntpos[2] >= fminZ && pntpos[2] <= fmaxZ);
      }
      
      // has a hit occurred?
      if(inRegion) {
        // yes, see if it is closer to the eye position than earlier objects
	newEyeDist = DTOEYE(pntpos[0], pntpos[1], pntpos[2]);
	if(currEyeDist < 0.0 || newEyeDist < currEyeDist) {
	  currEyeDist = newEyeDist;
	  tag = currTag;
	}
      }

    } else if(tok == DPICKLINE || tok == DPICKLINE_I
		|| tok == DPICKBOX || tok == DPICKBOX_I) {
      ;		// do nothing, these are as-yet unsupported picking tokens

    } else if(tok == DPUSH) {
      // push the current trans matrix
      transMat.push();

    } else if(tok == DPOP) {
      // pop the current trans matrix
      transMat.pop();
      
    } else if(tok == DLOAD) {
      // load the given trans matrix onto the stack
      Matrix4 tmpmat((float *)cmdptr);
      (transMat.top()).loadmatrix(tmpmat);

    } else if(tok == DMULT) {
      // multiply the given trans matrix
      Matrix4 tmpmat((float *)cmdptr);
      (transMat.top()).multmatrix(tmpmat);

    } else if(tok == DTRANS) {
      // do a translation
      (transMat.top()).translate(((float *)cmdptr)[0], ((float *)cmdptr)[1],
      		((float *)cmdptr)[2]);
      
    } else if(tok == DSCALE) {
      // do a scale
      (transMat.top()).scale(((float *)cmdptr)[0], ((float *)cmdptr)[1],
      		((float *)cmdptr)[2]);

    } else if(tok == DROT) {
      // do a rotation about a given axis
      (transMat.top()).rot(((float *)cmdptr)[0],
      		'x' + (int)(((float *)cmdptr)[1]));

    } else if(tok == DMOREDATA || tok == DNEWDATA) {
      // set the current drawing data block
#ifdef VMDCAVE
      dataBlock = (float *)cmdptr;      // point to current position in list
#else
      dataBlock = ((float **)cmdptr)[0];        // point to given memory loc
#endif

    }
    
    // update position in array
    cmdptr += cmdsize;
  }

  // make sure we do not disturb the regular transformation matrix
  transMat.pop();

  // return result; if tag >= 0, we found something
  eyedist = currEyeDist;
  return tag;
}

/***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: DisplayDevice.C,v $
 * Revision 1.11  1995/05/11  22:36:18  billh
 * Moved log messages to the end of the file.
 * Added virtual functions to queue and query events, as well as to test
 * the state of buttons and to return the position of the cursor.
 *
 * Revision 1.10  95/03/24  18:48:57  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.9  1995/03/04  05:08:58  billh
 * Fixed error with picking points using indexed positions.
 *
 * Revision 1.8  1995/02/22  04:03:10  billh
 * Added virtual routine pick, which checks the given draw list for points
 * which can be selected.  Also added routines to convert absolute screen
 * coords <--> relative scaled screen coords.
 * Added virtual routine find_3D_from_2D, which takes a 3D point at pos A,
 * and 2D rel screen position at B, and returns the 3D point corresponding
 * to the 2D point at B in the same plane as point A.
 *
 * Revision 1.7  95/01/09  08:51:10  billh
 * Changed default far clipping plane distance to 10.0.
 * 
 * Revision 1.6  1994/12/06  08:24:25  billh
 * Added ability to create color categories, due to being now derived from
 * ColorUser.
 * Added routines to have a new color list specified, and to check for when
 * a color in a relevant category is changed.
 *
 * Revision 1.5  94/11/09  03:45:28  billh
 * Provided default versions for all virtual functions, so an 'empty' display
 * can be created that does essentially nothing (for testing purposes).
 * 
 * Revision 1.4  1994/11/02  07:31:36  billh
 * Added routines to set screen height and distance from origin.
 *
 * Revision 1.3  1994/10/05  04:37:15  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.2  1994/09/28  22:11:55  billh
 * changed fsqrt to sqrtf
 *
 * Revision 1.1  1994/08/24  03:10:37  billh
 * Initial revision
 *
 ***************************************************************************/
