/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DrawPatch.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 1995/05/11 22:38:12 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Another Child Displayable component for a remote molecule; this displays
 * the configuration of any patches which have been stored for the current
 * timestep.  If no patch data is available, draws nothing.
 *
 ***************************************************************************/

#include "DrawPatch.h"
#include "DrawMolecule.h"
#include "ColorList.h"
#include "Inform.h"


// string descriptions for patch display methods
char *patchDisplayMethodName[DrawPatch::TOTAL_PATCH_DISPLAY] = {
	"off", "byatom", "byload", "bynode" };


////////////////////////////  constructor  

DrawPatch::DrawPatch(DrawMolecule *mr)
	: Displayable3D(MULT, mr->name, mr, mr->nAtoms/2 + 2) {

  MSGDEBUG(1,"Creating new patch representation, for parent molecule:\n");
  MSGDEBUG(1," ==> '" << mr->name << "'" << sendmsg);

  // save data
  mol = mr;
  
  // initialize variables
  displayMethod = BYNONE;
  needRegenerate = TRUE;
}


//////////////////////////////// private routines 

// regenerate the command list
void DrawPatch::create_cmdlist(void) {
  float *pdata, pmin, pmax, scalefac;
  int i, j, pnum, indx, clr;
  float bx[3], by[3], bz[3], bxy[3], tx[3], ty[3], txy[3];

  // do we need to recreate everything?
  if(needRegenerate) {

    MSGDEBUG(2, "Regenerating display list for patches '" << name << "' ...");
    MSGDEBUG(2, sendmsg);

    // regenerate both data block and display commands
    needRegenerate = FALSE;
    reset_disp_list();

    // only put in commands if there is a current frame
    if(displayMethod != BYNONE && mol->frame() >= 0) {
      // get current patch data
      pnum = (mol->current())->numPatches;
      
      // only need to do anything if # patches > 0
      if(pnum > 0) {
        pdata = (mol->current())->patchData;
	
	// get data for calculating colors
	if(displayMethod == BYATOM)
	  indx = TSP_ATOMS;
	else if(displayMethod == BYLOAD)
	  indx = TSP_LOAD;
	else
	  indx = TSP_NODE;
	pmin = (mol->current())->minpatch[indx];
	pmax = (mol->current())->maxpatch[indx];
	
	if(indx == TSP_NODE) {
	  if(pmax == pmin)
	    clr = REGWHITE;
	  else
	    scalefac = (float)(REGCLRS-1) / (pmax - pmin);
	} else {
	  if(pmax == pmin)
	    clr = MAPCLRS/2;
	  else
	    scalefac = (float)(MAPCLRS-1) / (pmax - pmin);
        }

        // turn off material characteristics, set line style
        cmdMaterials.putdata(FALSE,this);
        cmdLineType.putdata(SOLIDLINE, this);
        cmdLineWidth.putdata(3, this);

	// for all patches with atoms > 0, draw box
	for(i=0; i < pnum; i++) {
	 if(pdata[TSP_ATOMS] > 0) {
	  // first set the color
          if(pmax != pmin)
	    clr = (int)(scalefac * (pdata[indx] - pmin));
	  MSGDEBUG(1,"Patch " << i << ": color= " << clr << sendmsg);
	  if(indx == TSP_NODE)
	    cmdColorIndex.putdata(REGCOLOR(clr), this);
	  else
	    cmdColorIndex.putdata(MAPCOLOR(clr), this);

	  // find corners of box
	  float xlen = pdata[TSP_XLEN] * 0.98;
	  float ylen = pdata[TSP_YLEN] * 0.98;
	  float zlen = pdata[TSP_ZLEN] * 0.98;
	  for(j=0; j < 3; j++)
	    bx[j] = by[j] = bxy[j] = pdata[j];
	  bx[0] += xlen;
	  by[1] += ylen;
	  bxy[0] += xlen;	bxy[1] += ylen;
	  for(j=0; j < 2; j++) {
	    bz[j] = pdata[j];
	    tx[j] = bx[j];
	    ty[j] = by[j];
	    txy[j] = bxy[j];
	  }
	  bz[2] = pdata[2] + zlen;
	  tx[2] = bx[2] + zlen;
	  ty[2] = by[2] + zlen;
	  txy[2] = bxy[2] + zlen;
	  
	  // draw lines
          cmdLine.putdata(pdata, bx, this);
          cmdLine.putdata(pdata, by, this);
          cmdLine.putdata(bx, bxy, this);
          cmdLine.putdata(by, bxy, this);

          cmdLine.putdata(pdata, bz, this);
          cmdLine.putdata(bx, tx, this);
          cmdLine.putdata(by, ty, this);
          cmdLine.putdata(bxy, txy, this);

          cmdLine.putdata(bz, tx, this);
          cmdLine.putdata(bz, ty, this);
          cmdLine.putdata(tx, txy, this);
          cmdLine.putdata(ty, txy, this);
	 }
	 pdata += TSPATCHDATA;
        }
      }
    }
  }
}


//////////////////////////////// public routines 
// prepare for drawing ... do any updates needed right before draw.
void DrawPatch::prepare(DisplayDevice *) {

  // see if the frame has changed
  if(mol->has_frame_changed())
    needRegenerate = TRUE;
  
  create_cmdlist();
}


// change the method for displaying patches  Return new value.
int DrawPatch::display_method(int newval) {

  // if legal, change setting
  if(newval >= 0 && newval < TOTAL_PATCH_DISPLAY) {
    displayMethod = newval;
    needRegenerate = TRUE;
  }
  
  return displayMethod;
}

/* REVISION HISTORY:********************************************************
 *
 * $Log: DrawPatch.C,v $
 * Revision 1.6  1995/05/11  22:38:12  billh
 * Moved log messages to the end of the file.
 *
 * Revision 1.5  95/03/24  18:49:13  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.4  1994/11/21  14:45:23  billh
 * Patches colored by node use regular colors,not colormap.
 *
 * Revision 1.3  94/11/11  16:35:11  billh
 * No change.
 * 
 * Revision 1.2  94/11/10  19:25:39  billh
 * Fixed problems with displaying patches.  Colors properly now, and only
 * displays patches which have # atoms > 0.
 * 
 * Revision 1.1  94/11/09  03:23:05  billh
 * Initial revision
 * 
 ***************************************************************************/
