/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Molecule.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.7 $	$Date: 1995/05/11 23:18:06 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Main Molecule objects, which contains all the capabilities necessary to
 * store, draw, and manipulate a molecule.  This adds to the functions of
 * DrawMolecule and BaseMolecule by adding routines to read and write various
 * coordinate files.
 *
 * Note: Other capabilities, such as writing structure files, mutations, etc.
 * should go into this level.
 *
 ***************************************************************************/

#include "Molecule.h"
#include "Inform.h"
#include "DisplayDevice.h"


///////////////////////////  constructor  
Molecule::Molecule(char *src, Scene *sc)
  : DrawMolecule(sc), coorIOFiles(8) {
  source = stringdup(src);
}

Molecule::Molecule(char *src, Displayable *dp) 
  : DrawMolecule(dp), coorIOFiles(8) {
  source = stringdup(src);
}

///////////////////////////  destructor  
Molecule::~Molecule(void) {
  if(source)  delete [] source;
  
  while(coorIOFiles.num() > 0) {
    delete coorIOFiles[0];
    coorIOFiles.remove(0);
  }
}

///////////////////////////  public routines  

// ask to read in a coordinate file of the specified type
// arguments: filename, type, beginning frame, ending frame, frame skip
// (if any of the last three are < 0, default values for that arg used).
// Return success.
int Molecule::read_coor_file(char *fn, int t, int begFrame, int endFrame,
	int frameSkip) {

  if(t < 0 || t >= CoorFileData::COORTYPES || !fn)
    return FALSE;

  // create new file, make sure it was opened correctly
  CoorFileData *cfdata = 
  	new CoorFileData(this, fn, t, TRUE, begFrame, endFrame, frameSkip);

  if(cfdata->opened) {
    MSGDEBUG(2,"Added request to read file '" << fn << "', of type " << t);
    MSGDEBUG(2,"\nReading from frame " << cfdata->begFrame << " to ");
    if(cfdata->endFrame < 0) {
      MSGDEBUG(2,"end");
    } else {
      MSGDEBUG(2,cfdata->endFrame);
    }
    MSGDEBUG(2,", skipping frames in steps of " << cfdata->frameSkip);
    MSGDEBUG(2,sendmsg);

    // add to list
    coorIOFiles.append(cfdata);
    return TRUE;
  } else {
    delete cfdata;
    return FALSE;
  }
}


// ask to write out a coordinate file of the specified type
// arguments: filename, type, beginning frame, ending frame, frame skip
// (if any of the last three are < 0, default values for that arg used).
// Return success.
int Molecule::write_coor_file(char *fn, int t, int begFrame, int endFrame,
	int frameSkip) {

  if(t < 0 || t >= CoorFileData::COORTYPES || !fn)
    return FALSE;

  // create new file, make sure it was opened correctly
  CoorFileData *cfdata = 
  	new CoorFileData(this, fn, t, FALSE, begFrame, endFrame, frameSkip);

  if(cfdata->opened) {
    MSGDEBUG(2,"Added request to write file '" << fn << "', of type " << t);
    MSGDEBUG(2,"\nWriting frame " << cfdata->begFrame << " to ");
    if(cfdata->endFrame < 0) {
      MSGDEBUG(2,"end");
    } else {
      MSGDEBUG(2,cfdata->endFrame);
    }
    MSGDEBUG(2,", skipping frames in steps of " << cfdata->frameSkip);
    MSGDEBUG(2,sendmsg);

    // add to list
    coorIOFiles.append(cfdata);
    return TRUE;
  } else {
    delete cfdata;
    return FALSE;
  }
}


// prepare for drawing ... can do one or more of the following:
//	- open a new file and start reading
//	- continue reading an already open file
//	- finish reading a file and close it
// when done, this then 'prepares' the parent class
void Molecule::prepare(DisplayDevice *d) {
  
  // are we already reading/writing?
  if(coorIOFiles.num() > 0) {
    // r/w'ing file ... get new frame, until NULL is returned (then close)
    Timestep *newstep = coorIOFiles[0]->next();

    if(!newstep) {
      // no new frame found; remove from list
      msgInfo << "Finished with coordinate file " << coorIOFiles[0]->name;
      msgInfo << "." << sendmsg;
      delete coorIOFiles[0];
      coorIOFiles.remove(0);
    }
  }

  // do prepare for parent class
  DrawMolecule::prepare(d);
}

/* REVISION HISTORY:********************************************************
 *
 * $Log: Molecule.C,v $
 * Revision 1.7  1995/05/11  23:18:06  billh
 * Moved log messages to end of file.
 *
 * Revision 1.6  95/03/24  18:50:33  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.5  1994/11/22  02:34:02  billh
 * Now reads and writes coordinate files using CoorFileData class, and
 * can use a begframe, endframe, frameskip pattern for both read and write.
 *
 * Revision 1.4  94/10/05  21:45:01  billh
 * Changed to use a ResizeArray instead of linked list.
 * 
 * Revision 1.3  1994/10/05  04:38:23  billh
 * Took out double backslash from text, even in comments.
 *
 * Revision 1.2  1994/09/30  05:36:18  billh
 * Added externally-available string arrays for file type extensions.
 *
 * Revision 1.1  1994/09/17  09:11:36  billh
 * Initial revision
 *
 ***************************************************************************/
