/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeList.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.12 $	$Date: 1995/05/11 23:04:48 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The MoleculeList class, which is a list of the molecules being displayed.
 * This is a Displayable object, where each molecule is a child Displayable.
 *
 ***************************************************************************/
#ifndef MOLECULELIST_H
#define MOLECULELIST_H

#include "Displayable.h"
#include "Molecule.h"
#include "AtomColor.h"
#include "AtomRep.h"
#include "AtomSel.h"
#include "ResizeArray.h"
#include "utilities.h"
class MolAction;

// number of color categories, and where they're found in the table
#define MLCAT_NAMES		0
#define MLCAT_TYPES		1
#define MLCAT_RESNAMES		2
#define MLCAT_RESTYPES		3
#define MLCAT_SEGNAMES		4
#define MLCAT_MOLECULES		5
#define MLCAT_SPECIAL		6
#define NUM_MLCAT		7


class MoleculeList : public Displayable3D {

public:
  // enum for how actions are done: to ALL molecules, all ACTIVE, all
  // DISPLAYED, TOP, or NONE
  enum ActionTarget { ALL, TOP, ACTIVE, DISPLAYED, NONE };

private:
  // the 'top' molecule, which determines what the centering and scaling of
  // the molecules should be
  Molecule *topMol;
  
  // molecules in this list.
  ResizeArray<Molecule *> molList;

  // do we need to reset the transformation if the topMol has any frames?
  // Used only when only one molecule is loaded, and does not have any
  // frames yet.
  int needTopMolReset;

  // current atom selection, representation, and coloring methods
  AtomColor *currAtomColor;
  AtomRep *currAtomRep;
  AtomSel *currAtomSel;

  // set the given Molecule as the top molecule.
  void set_top_molecule(Molecule *);

protected:
  // do action when a new color list is provided
  virtual void do_use_colors(void);

public:
  // mapping of residue names <--> residue types (hydrophobic, neutral, etc.)
  // residue names are the 'name' key, and the index into the residue type
  // color list is the 'data' value.
  NameList<int> resTypes;

  // color category indices
  int colorCatIndex[NUM_MLCAT];

  // put new names from given molecule into color lists
  void add_color_names(int);

public:
  // constructor and destructor
  MoleculeList(Scene *);
  virtual ~MoleculeList(void);

  // return the number of molecules in this list
  int num(void) { return molList.num(); }

  // return the Nth molecule (index runs 0 ... (count-1))
  Molecule *molecule(int n) {
    Molecule *retval = NULL;
    if(n >= 0 && n < num())
      retval = molList[n];
    return retval;
  }
  
  // return the index of the molecule with given ID (-1 if error)
  int mol_index_from_id(int id) {
    for(int i=0; i < num(); i++) {
      if(id == (molList[i])->id())
        return i;
    }
    return (-1);
  }

  // add a new molecule; return it's position in molList, or (-1) if error
  int add_molecule(Molecule *);

  // remove the Nth molecule from the list, i.e. delete it.  Return success.
  int del_molecule(int);

  //
  // routines to get/set characteristics for new graphics representations
  //
  
  // get/set current atom coloring method
  int set_color(char *);
  char *color(void) { return currAtomColor->cmdStr; }
  
  // get/set current atom representation method
  int set_representation(char *);
  char *representation(void) { return currAtomRep->cmdStr; }
  
  // get/set current atom selection command
  int set_selection(char *);
  char *selection(void) { return currAtomSel->cmdStr; }
  
  // add a new graphics representation to the specified molecule.
  // uses the specified coloring, representation, and selection settings.
  // if n < 0, do so for all active molecules.
  int add_rep(int, AtomColor *, AtomRep *, AtomSel *);
  
  // add a new graphics representation to the specified molecule.
  // uses the 'current' coloring, representation, and selection settings.
  // if n < 0, do so for all active molecules.
  int add_rep(int n) {
    return add_rep(n, currAtomColor, currAtomRep, currAtomSel);
  }
  
  // change the graphics representation m, for the specified molecule n, to
  // the new settings.  Return success.
  int change_rep(int m, int n, AtomColor *, AtomRep *, AtomSel *);

  // change the graphics representation m, for the specified molecule n.
  // uses the 'current' coloring, representation, and selection settings.
  int change_rep(int m, int n) {
    return change_rep(m, n, currAtomColor, currAtomRep, currAtomSel);
  }

  // change just the coloring method for the mth rep in the nth molecule.
  int change_repcolor(int m, int n, char *);
  
  // change just the representation for the mth rep in the nth molecule.
  int change_repmethod(int m, int n, char *);
  
  // change just the selection for the mth rep in the nth molecule.
  int change_repsel(int m, int n, char *);

  // delete a graphics representation m, for the specified molecule n.
  // return success.
  int del_rep(int m, int n);

  //
  //
  // routines to get/set top, active, displayed, fixed molecules
  //
  
  // query or set the top molecule
  Molecule *top(void) { return topMol; }
  int is_top(int n) { return (molecule(n) == topMol); }
  int is_top(Molecule *m) { return topMol == m; }
  void make_top(int n) { make_top(molecule(n)); }
  void make_top(Molecule *m);

  // query/set active status of Nth molecule
  int active(int n) { return molecule(n)->active; }
  int active(Molecule *m) { return (m && m->active); }
  void activate(int n) { molecule(n)->active = TRUE; }
  void inactivate(int n) { molecule(n)->active = FALSE; }
  
  // query/set displayed status of Nth molecule
  int displayed(int n) { return molecule(n)->displayed(); }
  int displayed(Molecule *m) { return (m && m->displayed()); }
  void show(int n) { molecule(n)->on(); }
  void hide(int n) { molecule(n)->off(); }
  
  // query/set fixed status of Nth molecule
  int fixed(int n) { return molecule(n)->fixed(); }
  int fixed(Molecule *m) { return (m && m->fixed()); }
  void fix(int n) { molecule(n)->fix(); }
  void unfix(int n) { molecule(n)->unfix(); }

  //
  // functions that affect all molecules
  //

  // apply an action to the list of active molecules
  void act(MolAction&, ActionTarget = ACTIVE);

  // apply an actino to the Nth molecule
  void act(MolAction&, int);

  //
  // Displayable virtual functions
  //

  // reset to identity the tranformation ... virtual so resets can affect
  // other factors as well.
  // This version calls the parent version, but also sets all molecule's
  // tranformations to center and scale properly based on the top molecule.
  virtual void reset_transformation(void);

  // prepare for drawing
  virtual void prepare(DisplayDevice *);

};

#endif

/* REVISION HISTORY:********************************************************
 *
 * $Log: MoleculeList.h,v $
 * Revision 1.12  1995/05/11  23:04:48  billh
 * Moved log message to end of file; fixed ambiguity with setting top mol.
 *
 * Revision 1.11  95/03/28  03:45:32  billh
 * Added more built-in residue colors; fixed setup for coloring by residue
 * type; changing the top molecule no longer does a reset, this is only
 * done when a new molecule is loaded.
 * 
 * Revision 1.10  95/03/24  18:50:53  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.9  1994/12/06  08:28:23  billh
 * MoleculeList no longer includes Global.h; instead, uses the new
 * ability to be derived from ColorUser so that right after creation,
 * the routine use_colors() must be called to tell this object which
 * ColorList to use.
 *
 * Revision 1.8  94/12/05  04:23:59  billh
 * No change.
 * 
 * Revision 1.7  94/10/26  23:22:30  billh
 * Added routines to allow commands to change the settings for the
 * current molecule representation options.
 * 
 * Revision 1.6  94/10/20  01:31:56  billh
 * Added routines to perform a MolAction on the molecules in the list.
 * 
 * Revision 1.5  1994/09/29  07:28:22  billh
 * Attempt to fix problem with getting residue type color.
 * Added routine to return relative index of molecule with given id.
 *
 * Revision 1.4  94/09/26  18:57:24  billh
 * Added separate list for molecules (instead of recasting child pointers).
 * Added del_rep, add_molecule routines.
 * 
 * Revision 1.3  94/09/24  20:26:06  billh
 * Added support for color; added routines to setup color index lists,
 * default atom/residue names, and initialization routines.
 * 
 * Revision 1.2  94/09/23  00:54:06  billh
 * Added ability to add new representations to molecules.
 * Put in initial version of 'prepare', which just checks to see if
 * the top molecule has changed.
 * 
 * Revision 1.1  94/09/17  09:11:36  billh
 * Initial revision
 * 
 ***************************************************************************/
