/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeSigma.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.2 $	$Date: 1995/05/11 23:08:04 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A MoleculeSigma is inherited from MoleculeFile, and supplies additional
 * routines to link the Molecule with the SIgMA MD code.
 *
 ***************************************************************************/

#include <math.h>
#include <string.h>
#include <ctype.h>
#include <sys/param.h>

#include <mdsocket.h>
#include <mdbuffer.h>
#include <Communication.h>
#include <Graphics.h>
#include <Id.h>

#include "MoleculeSigma.h"
#include "PickModeDrag.h"
#include "Atom.h"
#include "ReadPDB.h"
#include "ReadPSF.h"
#include "Scene.h"
#include "DisplayDevice.h"
#include "Mouse.h"
#include "Global.h"
#include "utilities.h"

// Pick mode for dragging; -1 => pick mode needs to be initialized
int MoleculeSigma::pickDragMode = -1;

// identifying code passed to add_pick_mode and returned to
//  create_pick_mode
static const int pickModeId = 42;

/////////////////////////////////////////////////////////////////////
// class constructor ... specifies the type of file, the filename,
// a Scene or Displayable, and a possibly uesful cutoff distance to use.
// The cutoff distance is only used when a PDB is used for the structure.
MoleculeSigma::MoleculeSigma(char *fname, Scene *sc)
	: MoleculeFilePSF(fname, sc) {
    msgInfo << "MoleculeSigma: creating from PSF " << fname << sendmsg;
    init_sigma();
}

// this version uses a Displayable instead of Scene, so that it becomes a
// child Displayable to some parent.
MoleculeSigma::MoleculeSigma(char *fname, Displayable *dp)
	 : MoleculeFilePSF(fname, dp) {
    msgInfo << "MoleculeSigma: creating from PSF " << fname << sendmsg;
    init_sigma();
}

MoleculeSigma::~MoleculeSigma(void) {
    MSGDEBUG(1,"Deleting MoleculeSigma ..." << sendmsg);

    // Send termination message
    buffer_send_event(MD_EXIT);
    send_buffer();
    sleep(2);
    disconnect();
}

// Initialize connection to SIgMA, and set up our special pick mode
void MoleculeSigma::init_sigma() {
    connect();
    // Send initialization message. Just guess at allowed Id range
    //	until we construct a mapping from Ids to atoms.
    graphics_ready(1,1000);

    // Register with the picklist
    register_with_picklist(this->origScene);

    if (pickDragMode == -1) {
	pickDragMode = add_pick_mode("Drag", pickModeId);
	if (pickDragMode == -1)
	   msgErr << "MoleculeSigma: Can't add_pick_mode(Drag)" << sendmsg;
	else
	   MSGDEBUG(1,"MoleculeSigma: pickDragMode = " << pickDragMode << sendmsg);
    }
}

// Connect to a SIgMA process (user must start it up by hand at present)
void MoleculeSigma::connect() {
    sock = -1;

    char host[MAXHOSTNAMELEN];

    if (gethostname(host, sizeof(host)) < 0) {
       msgErr << "MoleculeSigma::connect: can't get host name" << sendmsg;
       return;
    }

    // Listen for an incoming connection
    int port = 0,
	listenfd = socket_connect(host, port, 1, &port);

    if (listenfd < 0) {
	msgErr << "MoleculeSigma::connect: can't establish listening socket" << sendmsg;
	return;
    }

    msgInfo << "Connect SIgMA to: " << host << " " << port << sendmsg;

    // Wait for a SIgMA process to connect to the listener
    sock = socket_accept(listenfd);
    close(listenfd);

    if (sock < 0) {
	msgErr << "MoleculeSigma::connect: accept on listening socket failed" << sendmsg;
	return;
    }

    buffer_set_socket(sock);
    if (buffer_handshake() < 0) {
	msgErr << "MoleculeSigma::connect: setup failed" << sendmsg;
	sock = -1;
	return;
    }

    // Don't block on reads from socket
    socket_setblocking(sock, 0);
}

// Disconnect from a SIgMA process
void MoleculeSigma::disconnect() {
    if (sock < 0)
	return;

    close(sock);
    sock = -1;
}

// create the molecule structure from a PSF file.  Returns success.
int MoleculeSigma::create(void) {
    MSGDEBUG(2,"MoleculeSigma: executing create ..." << sendmsg);

    return MoleculeFilePSF::create();
}

// prepare to draw the molecule; handle messages on the communications
//  socket to Sigma.
void MoleculeSigma::prepare(DisplayDevice *d) {
    int event;
    char *buf;

    while (get_event(&event, &buf, TRUE) != -1) {
	switch (event) {
	    case MD_SET_COORDINATES:
		read_coords(buf);
		break;
	    case MD_ADD_MENU:
		read_menu(buf);
		break;
	    default:
		msgErr << "MoleculeSigma::prepare(): unknown event "
		       << md_message_name(event) << sendmsg;
		break;
	}
    }

    Molecule::prepare(d);
}

// Read user menu definition from Sigma and put it in the UI
void MoleculeSigma::read_menu(char *bufptr) {
    int items;
    char *title;
    char cmdbuf[64];

    bufptr = buffer_read_int(&items, bufptr);
    bufptr = buffer_read_string(&title, bufptr);

    mouse->create_user_menu(title);

    for (int i = 0; i < items; i++) {
	char *menu;
	bufptr = buffer_read_string(&menu, bufptr);
        sprintf(cmdbuf, "sigma %d %d", MD_MENU_PICK, i);
	mouse->add_user_submenu_item(title, menu, cmdbuf);
	delete [] menu;
    }

    delete [] title;
}

// Read molecular coordinates from Sigma & update the model
void MoleculeSigma::read_coords(char *bufptr) {
    int ranges;

    bufptr = buffer_read_int(&ranges, bufptr);
//    msgDebug << "MoleculeSigma::read_coords: ranges = " << ranges << sendmsg;
    int *from = new int[ranges],
	*to = new int[ranges],
	num_coords = 0;

    for (int i = 0; i < ranges; i++) {
	bufptr = buffer_read_int(&from[i], bufptr);
	bufptr = buffer_read_int(&to[i], bufptr);
	num_coords += to[i] - from[i] + 1;
//	  msgDebug << "\trange " << i << " from = " << from[i]
//		  << " to = " << to[i] << sendmsg;
    }
//    msgDebug << "# coordinates = " << num_coords << sendmsg;

    // See Animation.C and MoleculeRemote.C for example of Timestep usage...
    Timestep *newts = item(num() - 1);
    if (newts == NULL) {
	msgErr << "read_coords: no Timestep to update!" << sendmsg;
	return;
    }
    // could create a new timestep instead of reusing:
    //	newts = new Timestep(nAtoms, 0.0);

    // Pointer to data buffer in Timestep
    float *newts_xyz = newts->pos;

    // Loop over each range reading all the coordinates in that range
    //	and updating the model. VMD indices are 0-based, Sculpt
    //	protocol is 1-based, so offset indices appropriately
    for (i = 0; i < ranges; i++) {
	Point3 pos;

	for (int anum = from[i]; anum <= to[i]; anum++) {
	    Atom *aptr = atom(anum-1);

	    bufptr = buffer_read_point3(&pos, bufptr);
	    aptr->pos[0] = pos.x;
	    aptr->pos[1] = pos.y;
	    aptr->pos[2] = pos.z;

	    // Get address of (anum-1)'th point in Timestep buffer
	    float *dptr = newts_xyz + (anum-1) * 3;
	    dptr[0] = pos.x;
	    dptr[1] = pos.y;
	    dptr[2] = pos.z;
	}
    }

    newts->init();	// Recalculate values for the timestep (meaning what?)
    curr_frame_changed();   // Indicate frame data has changed

    delete [] from;
    delete [] to;
}
#if 0

	if(num() == 0) {
	    // this is the first frame, put the positions in the Atom objects
	    atm = atom(i);
	    atm->pos[0] = *tsx;
	    atm->pos[1] = *tsy;
	    atm->pos[2] = *tsz;
	}
	*(pos++) = *(tsx++);
	*(pos++) = *(tsy++);
	*(pos++) = *(tsz++);
    }

    // now append the new frame, if necessary
    if(saveFrame) {
	append_frame(newts);		// append the new frame
    } else {
	newts->init();			// recalc values for the timestep
	curr_frame_changed();		// signal the current frame changed
    }
#endif

// Determine if we're interested in this pick mode
int MoleculeSigma::want_pick_mode(int mode) {
    msgDebug << "MoleculeSigma::want_pick_mode mode = " << mode
	    << " pickDragMode = " << pickDragMode << sendmsg;
    if (mode == pickDragMode)
	return TRUE;
    else
	return MoleculeFilePSF::want_pick_mode(mode);
}

PickMode *MoleculeSigma::create_pick_mode(int id) {
    msgDebug << "MoleculeSigma::create_pick_mode id = " << id << sendmsg;

    if (id == pickModeId) {
	msgDebug << "\tcreating new PickModeDrag" << sendmsg;
	return new PickModeDrag;
    } else
	return MoleculeFilePSF::create_pick_mode(id);
}

// called when a pick is begun:
//    args = display to use, obj picked, button, mode, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
void MoleculeSigma::pick_start(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    msgDebug << "MS pick_start tag = " << tag << " dim = " << dim
	    << " button = " << b << " mode = " << m << sendmsg;
    msgDebug << "\tpos = " << pos[0] << "\t" << pos[1]
	    << "\t" << pos[2] << sendmsg;
    MoleculeFilePSF::pick_start(d,p,b,m,tag,dim,pos);

    // Start specifying a restraint
    Id id(0,tag);   // Id to send is <anything,atom #>
    buffer_send_event(MD_MOUSE_DOWN, &id);
}

// called when a pick moves:
void MoleculeSigma::pick_move(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    msgDebug << "MS pick_move tag = " << tag << " dim = " << dim << " pos = "
	    << pos[0] << "\t" << pos[1] << "\t" << pos[2] << sendmsg;

    if (p != this) {
	// Should we care about this?
	//msgErr << "MS pick_move: Pickable p != this (continuing)" << sendmsg;
    }
    MoleculeFilePSF::pick_move(d,p,b,m,tag,dim,pos);

    Timestep *newts = item(num() - 1);
    if (newts == NULL) {
	msgErr << "MS::pick_move: no Timestep to update!" << sendmsg;
	return;
    }
    // Pointer to data buffer in Timestep; tag = atom #
    float *newts_xyz = newts->pos;
    float *oldpos = newts_xyz + tag * 3;

    msgDebug << "\told object space pos  = " << oldpos[0] << " " << oldpos[1]
	     << " " << oldpos[2] << sendmsg;

    // If a 2D pick, map mouse position to 3D and then into
    //	object space; otherwise, just map into object space.
    float *newpos, newworldpos[3], newobjectpos[3];
    if (dim == 2) {
	float oldworldpos[3];

	// Transform atom to world coordinates
	tm.multpoint3d(oldpos, oldworldpos);

	msgDebug << "\told world space pos   = " << oldworldpos[0] << " "
		<< oldworldpos[1] << " " << oldworldpos[2] << sendmsg;

	// Map mouse position to same depth as old atom position
	d->find_3D_from_2D(oldworldpos, pos, newworldpos);
	msgDebug << "\tnew world space pos   = " << newworldpos[0] << " "
		<< newworldpos[1] << " " << newworldpos[2] << sendmsg;

	// Transform mapped mouse position back to object space,
	//  giving new atomic coordinate.
	Matrix4 tminv(tm);

	tminv.inverse();
	tminv.multpoint3d(newworldpos, newobjectpos);

	msgDebug << "\tnew object space pos  = " << newobjectpos[0] << " "
		<< newobjectpos[1] << " " << newobjectpos[2] << sendmsg;

	newpos = newobjectpos;
    } else {
	// This probably needs to be mapped into object space
	newpos = pos;
    }

    msgDebug << "\trestraint position set to " << newpos[0] << " " << newpos[1]
	    << " " << newpos[2] << sendmsg;

    // Update restraint position in SIgMA
    Point3 restraint;
    restraint.x = newpos[0];
    restraint.y = newpos[1];
    restraint.z = newpos[2];
    buffer_send_event(MD_MOUSE_MOVE, &restraint);
}

// called when a pick ends:
void MoleculeSigma::pick_end(DisplayDevice *d, Pickable *p,
				int b, int m, int tag, int dim, float *pos) {
    msgDebug << "MS pick_end tag = " << tag << " dim = " << dim << " pos = "
	    << pos[0] << "\t" << pos[1] << "\t" << pos[2] << sendmsg;
    MoleculeFilePSF::pick_end(d,p,b,m,tag,dim,pos);

    // End specifying this restraint
    buffer_send_event(MD_MOUSE_UP);
}

/* REVISION HISTORY:********************************************************
 *
 * $Log: MoleculeSigma.C,v $
 * Revision 1.2  1995/05/11  23:08:04  billh
 * Moved log message to the end of the file.
 * Updated to use new mouse menu addition routines.
 *
 * Revision 1.1  95/04/05  20:45:59  billh
 * Initial revision
 * 
 ***************************************************************************/
