/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Remote.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.7 $	$Date: 1995/05/11 23:42:33 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Remote object, which maintains a connection to a remote computer that
 * provides data and structures.
 *
 ***************************************************************************/
#ifndef REMOTE_H
#define REMOTE_H

#include "mdcomm.h"
#include "mdc_errno.h"
class MoleculeRemote;

// variables for accessing the error variables in mdcomm
extern "C" {
  extern int mdc_errno;
  extern char *mdc_errlist[];
};


class Remote {

// make the MoleculeRemote, which uses this class, a friend
friend class MoleculeRemote;

public:
  // settings which can be modified for a running sim
  enum Settings { TRANSFER_RATE, TIMESTEP, TEMPERATURE, SAVEFRAME_RATE,
  	TOTAL_SETTINGS };

  // status level of this object
  enum Status { NO_CONNECT, SELECTING_APP, EDITING_PARAMS, CONNECTED,
  	 CONNECT_ERROR, READ_ERROR, TOTAL_STATUS };

private:
  // have we run mdc_init yet?
  static int run_mdc_init;

protected:
  // are we trying to start a new app (T), or connecting to a running one?
  int startingNewApp;

  // are we currently editing parameters?
  int editingParams;

  // have we made a connection?
  int madeConnection;

  // current connection status description
  Status remoteStatus;
  
  // most recent error value
  int returnValue;
  
  // name of user on remote machine (if NULL, not yet set)
  char *username;
  
  // name of remote computer (if NULL, not yet set)
  char *computer;
  
  // which application we're working with (either from 'proglist' list,
  // or 'joblist' list, depending on whether we're running a new program,
  // or attaching to an already-running one)
  int application;
  
  // possible applications to run
  struct mdc_proglist *proglist;
  int availApps;
  
  // available jobs running on the remote machine, to which we can connect
  struct mdc_joblist *joblist;
  int availJobs;

  // option list (if NULL, not yet set)
  struct md_optlist *optlist;
  int availOptions;
  
  // connection structure (if NULL, not yet set)
  struct md_arena *consumer;
  
  // structure returned in connection (if NULL, not yet set)
  struct md_struct *md_struct;
  
  // 'state' of the connection (used for shared memory access)
  struct dyndState state;
  
  //
  // modifiable parameters
  //

  // rate that new frames are transferred here from remote connection
  int transferRate;
  
  // current timestep
  float timeStep;
  
  // current temperature
  float Temperature;
  
  // frequency with which we save a frame ... every Nth frame received from
  // the remote connection.  If < 1, do not save any.
  int saveFrameRate;

  // return index of for given setting, -1 if error
  int setting_index(char *);

public:
  // constructor/destructor: arguments are username, hostname.
  Remote(char *, char *);
  ~Remote(void);
  
  //
  // general information
  //

  // return username to use on remote machine
  char *user(void) { return username; }
  
  // return host for remote connection
  char *host(void) { return computer; }
  
  // return application for remote connection
  char *app(void);

  // current status of connection (not the same as the error status string)
  char *status(void);
  
  // name of molecule to simulate (taken from STR keyword, if available)
  char *mol_name(void);

  // are the parameters being edited currently?
  int editing_parameters(void) { return editingParams; }

  // are we running a simulation properly?
  int running_simulation(void) { return madeConnection; }

  //
  // routines to change remote object state, i.e. make connection, etc.
  //
  
  // NOTE: either 'get_parameters' or 'attach_to_job' should be run, but ONLY
  // ONE of them (they are mutually-exclusive).

  // get parameters, for the Nth available app.  Return success.
  int get_parameters(int);
  
  // attach to the Nth running job.  Return success.
  int attach_to_job(int);
  
  // if a new simulation is being run, and get_parameters has been called with
  // the parameters all set up, this makes a connection and runs the sim.
  // return success.
  int start_new_simulation(void);

  // check for a new timestep; return TRUE if available.
  int next_ts_available();
  
  // gets timestep data, if available.  returns TRUE if successful.
  int get_next_ts(void);
  
  // signal a new timestep can be sent.  Should be called after get_next_ts
  // has returned true, and the data has been fetched properly.
  void done_with_ts(void);
  
  // close: finishes the connection
  // if arg = TRUE, and an application is running, this will disconnect from
  // the job but let it keep running.  Otherwise, this will kill the remote
  // application.
  void remote_close(int);
  
  //
  // routines to query/change the status of running simulation
  // strings with setting name, and type (i, s, f) are declared as
  // external strings at the bottom of this file.
  //

  // return number of settings available
  int avail_settings(void) { return TOTAL_SETTINGS; }
  
  // change the value of the specified setting.  return success.
  int change_setting(char *, char *);

  // write the value of the specified setting into the given string storage.
  // return success.
  int get_setting(char *, char *);
  int get_setting(int, char *);
  
  //
  // routines to work with parameter data files
  //

  // process list of options of the form "<keyword> = <value>" in a file.
  // return success.
  int read_param_file(FILE *);
  
  // write current parameter list to file, in form "<keyword> = <value>"
  // return success.
  int write_param_file(FILE *);

  //
  // error status routines
  //

  // query latest return status
  int error_status(void) { return returnValue; }
  
  // query return status description
  char *error_status_desc(int st);
  char *error_status_desc(void);

  //
  // query about available already-running jobs
  //

  // number of jobs to which we can connect
  int running_jobs(void) { return availJobs; }
  
  // return pid and uid of the Nth job
  void running_job_ids(int, int &, int &);
  
  // return name of Nth available job's program
  char *running_job_name(int);

  //
  // query about available apps on the remote computer
  //

  // number of apps on the remote machine
  int avail_apps(void) { return availApps; }

  // name of Nth available app
  char *avail_app_name(int);

  //
  // query about / change parameters
  //

  // number of options in optlist
  int options(void) { return availOptions; }

  // complete string description of Nth option, including keyword, value,
  // required, and type
  char *option_string(int);

  // type of nth option; if < 0, error
  int option_type(int n) {
    int retval = (-1);
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->type;
    return retval;
  }
  
  // keyword of nth option; if NULL, error
  char *option_keyword(int n) {
    char *retval = NULL;
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->keyword;
    return retval;
  }
  
  // text description of nth option
  char *option_desc(int n) {
    char *retval = NULL;
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->desc;
    return retval;
  }
  
  // is nth option required?; if < 0, error
  int option_req(int n) {
    int retval = (-1);
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->required;
    return retval;
  }
  
  // default value of nth option; if NULL, error
  char *option_default(int n) {
    char *retval = NULL;
    if(optlist && n >= 0 && n < optlist->count)
      retval = optlist->opt[n]->def;
    return retval;
  }
  
  // change value of nth option; return success
  int set_option(int n, char *val) {
    int retval = FALSE;
    if(optlist && val && n >= 0 && n < optlist->count)
      retval = (mdc_setopt(optlist, option_keyword(n), val) == MDC_EOK);
    return retval;
  }
  
  // change value of option with given keyword; return success
  int set_option(char *key, char *val) {
    int retval = FALSE;
    if(optlist && key && val)
      retval = (mdc_setopt(optlist, key, val) == MDC_EOK);
    return retval;
  }

};


// string descriptions, and types, for available simulation settings
extern char *remoteSettingName[Remote::TOTAL_SETTINGS];
extern char *remoteSettingKeyword[Remote::TOTAL_SETTINGS];
extern char  remoteSettingType[Remote::TOTAL_SETTINGS];

#endif

/* REVISION HISTORY:********************************************************
 *
 * $Log: Remote.h,v $
 * Revision 1.7  1995/05/11  23:42:33  billh
 * Moved log messages to end of file.
 *
 * Revision 1.6  95/03/24  18:51:33  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.5  1994/11/10  17:19:25  billh
 * Added option to save every Nth transferred frame.
 *
 * Revision 1.4  1994/11/02  01:36:40  billh
 * When receiving a timestep, must call 'get_next_ts' in Remote first, get
 * all the data, and when done, call 'done_with_ts' in Remote.
 *
 * Revision 1.3  94/10/21  03:51:20  billh
 * Several bug fixes, and added routines to get and change settings.
 * 
 * Revision 1.2  1994/10/01  11:01:37  billh
 * Several bug fixes due to putting in MoleculeRemote code.
 *
 * Revision 1.1  94/10/01  03:08:43  billh
 * Initial revision
 *  
 ***************************************************************************/
