/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Tracker.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 1995/05/11 23:46:21 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A Tracker is a device for measuring the 6D spatial position of a device.
 * This is an abstract base class which requires a particular subclass to
 * define how that data is acquired.
 *
 ***************************************************************************/
#ifndef TRACKER_H
#define TRACKER_H

#include "utilities.h"
#include "Buttons.h"

#define DEF_MAX_FILTER_SIZE	3

// various tracker error states
//  only the first two are used in UNCTracker
//  all four are used in PVMTracker
#define TRACKER_OK           0
#define TRACKER_NOK         -1
#define TRACKER_NOPVM       -2
#define TRACKER_NOREMOTE    -3


class Tracker {

protected:
  // number of sensors available
  int sensors;

  // if the sensor is working TRACKER_OK (status=0) or not (status<0)
   // (not to be confused with the pause state)
  int status;

  // whether the tracker is acquiring new data, or paused.
  int trackerPaused;

  // 3D position of sensor
  float *position;
  
  // orientation of sensor
  float *orientation;
  
  // availability of sensors
  // used in derived classes to reduce the number of polls
  int *sensorAvailable;

  // number of elements stored for boxcar-average filter, and max we store
  int *filterSize, maxFilterSize;

  // where in the filter storage the next values go
  int *filterSlot;

  // data stored for the filtering ... up to filterSize reads.
  // then, position and orientation hold the average of the last
  // filterSize reads.
  float **filterPos, **filterOrient;  

  // the filtering cuts out when the sum of the absolute value
  // of the difference of the current position and the previous
  // average exceeds the value of filterPositionThreshold
  float filterPositionThreshold; 

  // scale factor applied to position data read from sensors ... used
  // to convert from 'tracker' space to 'world' space.
  float scaleFactor;

  // virtual routine which reads data from sensors, if available
  virtual int my_get_data(int, float *, float *, float *, float *, int) = 0;
  
  // keep track of a "Button" device which we assume is physically
  // associated with this Tracker/sensor pair
  Buttons **buttons;

public:
  // constructors and destructors
  //	Note: the position and orientation must be initialized by
  //		derived classes.
  Tracker(int num_sensors = 2);
  virtual ~Tracker(void);

  // return number of sensors this tracker has
  int numSensors(void) { return sensors; }

  // return the startup status
  int startup_status(void) { return status; }

  // is the tracker paused?
  int paused(void) { return trackerPaused; }

  // pause the tracker
  void pause(void) { trackerPaused = TRUE; }

  // resume the tracker
  void unpause(void) { trackerPaused = FALSE; }

  // function to determine if the given sensor is available
  int sensor_available(int s) {
     return(s >= 0 && s < sensors && sensorAvailable[s]);
  }

  // function to update the sensor positions
  void update_sensors(void);
     
  // read position of sensor
  void get_position(int station, float& x, float& y, float& z, 
                    int do_update = FALSE);

  // read orientation of sensor
  void get_orientation(int station, float *orient, int do_update = FALSE);

  // get the current position threshold value
  float get_position_threshold(void) { return filterPositionThreshold; }

  // change the current position threshold value for all the sensors
  void change_position_threshold(float newThreshold);

  // return current scaling factor
  float scale(void) { return scaleFactor; }

  // change the current scaling factor
  void set_scale(float newScale);

  // assign a Button device to a sensor
  void assign_buttons(int sensor, Buttons *buttons);
  
  Buttons *get_buttons(int sensor);
};  // end of class

#endif

/* REVISION HISTORY:********************************************************
 *
 * $Log: Tracker.h,v $
 * Revision 1.6  1995/05/11  23:46:21  billh
 * Updated due to changes in Buttons; moved log message to end of file.
 *
 * Revision 1.5  95/03/24  18:52:09  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.4  1994/09/24  21:27:56  dalke
 * changed a function name
 *
 * Revision 1.3  1994/09/24  20:58:31  dalke
 * Now I can get the button associated with a sensor
 *
 * Revision 1.2  1994/09/17  09:41:44  dalke
 * Made sensors able to have an associated button input device
 *
 * Revision 1.1  1994/09/03  11:18:13  dalke
 * Initial revision
 *
 ***************************************************************************/
