/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: UIList.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 1995/05/12 00:10:51 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 *  This is the object that maintains the list of all User Interfaces.
 *  The idea is that we create all the UIs (forms, 3D, text, ...) at the
 *  beginning of the program and "register_UI".  Then, during the main
 *  event loop, these UIs are told, first, to check for any events that
 *  occured and put commands on the command queue.  After that step, the
 *  commands are executed and the UIs told about the results of each.
 ***************************************************************************/

#include "UIList.h"
#include "UIObject.h"
#include "Inform.h"
#include "Global.h"


/////////////////////////////////  constructor
UIList::UIList( void) : uilist(16) {
  MSGDEBUG(1,"Creating UI List ..." << sendmsg);
}

/////////////////////////////////  destructor
UIList::~UIList( void) {
  MSGDEBUG(1,"Deleting UI List ..." << sendmsg);
}


//////////////////////////////////  member functions  /////////////////////////
// return the first UI with given name; NULL if no match found
UIObject *UIList::item(char *n) {
  UIObject *ui = NULL;
  for(int i=0; i < uilist.num(); i++) {
    ui = uilist[i];
    if(!strupncmp(n, ui->name, CMDLEN))
      break;
    else
      ui = NULL;
  }
  return ui;
}


// return the Nth item
UIObject *UIList::item(int n) {
  UIObject *retval = NULL;
  if(n >= 0 && n < num())
    retval = uilist[n];
  return retval;
}



// Stick a new UI in the list and return index (unique, since we do not
// ever unregister UIObjects)
int UIList::register_UI(UIObject *ui) {
  uilist.append(ui);		// put at end of list

  MSGDEBUG(2,"UIList: registered UI '" << ui->name << "' as UI ");
  MSGDEBUG(2,uilist.num() - 1 << sendmsg);

  return (uilist.num() - 1);	// since we know it went at end of list
}


// send a Command to the UIs after it has been executed
void UIList::act_on_command_UI(Command::Cmdtype cmdtype, Command *cmd,
	int result) {
  UIObject *uiobj; 
  for(int i=0; i < uilist.num(); i++) {
    uiobj = uilist[i];
    if(uiobj->want_command(cmdtype)) {

      MSGDEBUG(3,"UI '" << uiobj->name << "' acting on command " << cmdtype);
      MSGDEBUG(3," ..." << sendmsg);
	
      uiobj->act_on_command(cmdtype, cmd, result);
    } else {
      MSGDEBUG(3,"UI '" << uiobj->name << "' ignoring command " << cmdtype);
      MSGDEBUG(3," ..." << sendmsg);
    }
  }
}


// the simpler (and slower) interface
void UIList::act_on_command(Command *cmd) { 
  act_on_command_UI(cmd -> gettype(), cmd, TRUE);
}


  // make all registered UI's reset     
void UIList::reset_UI(void) {
    MSGDEBUG(3,"UIList: resetting all UI's ..." << sendmsg);
    for(int i=0; i < uilist.num(); i++)
      (uilist[i])->reset();
  }

  // make all registered UI's init     
void UIList::init_UI(void) {
    MSGDEBUG(3,"UIList: initializing all UI's ..." << sendmsg);
    for(int i=0; i < uilist.num(); i++)
      (uilist[i])->init();
  }

  // make all registered UI's update     
void UIList::update_UI(void) {
    MSGDEBUG(3,"UIList: updating all UI's ..." << sendmsg);
    for(int i=0; i < uilist.num(); i++)
      (uilist[i])->update();
  }

  // make all registered UI's check for an event     
void UIList::check_event_UI(void) {
    MSGDEBUG(3,"UIList: checking for events in all UI's ..." << sendmsg);
    for(int i=0; i < uilist.num(); i++)
      (uilist[i])->check_event();
  }

/* REVISION HISTORY:********************************************************
 *
 * $Log: UIList.C,v $
 * Revision 1.6  1995/05/12  00:10:51  billh
 * Moved log messages to end of file.
 *
 * Revision 1.5  95/03/24  18:52:20  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.4  1994/10/05  07:05:23  billh
 * moved some routines to .C file.
 *
 * Revision 1.3  1994/10/05  06:58:16  billh
 * Converted to use a ResizeArray instead of a DLinkList in the command queue
 * and ui list.
 *
 * Revision 1.2  1994/09/12  20:52:57  billh
 * Made some debugging messages L3 instead of L2 (to reduce L2 verbosity).
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 * 
 ***************************************************************************/
