/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: UIText.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.37 $	$Date: 1995/05/13 01:40:18 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * This is the User Interface for text commands.  It reads characters from
 * the console, and executes the commands.
 *
 * This will use the Tcl library for general script interpretation, which
 * allows for general script capabilities such as variable substitution,
 * loops, etc.  If Tcl cannot be used, text commands will still be available
 * in the program, but the general script capabilities will be lost.
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#if defined(ARCH_IRIX4) || defined(ARCH_IRIX5) || defined(ARCH_IRIX6)
#include <bstring.h>
#endif
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <ctype.h>
#include <fcntl.h>

#include "UIText.h"
#include "CommandQueue.h"
#include "Inform.h"
#include "config.h"
#include "utilities.h"

// standard VMD command objects
#include "CmdAnimate.h"
#include "CmdColor.h"
#include "CmdDisplay.h"
#include "CmdLabel.h"
#include "CmdMenu.h"
#include "CmdMol.h"
#include "CmdMouse.h"
#include "CmdRender.h"
#include "CmdTool.h"
#include "CmdTracker.h"
#include "CmdTrans.h"
#include "CmdUser.h"
#include "CmdUtil.h"

// optional VMD command objects
#ifdef VMDEXTERNAL
#include "CmdExternal.h"
#endif

#ifdef VMDREMOTE
#include "CmdRemote.h"
#endif

#ifdef VMDSIGMA
#include "CmdSigma.h"
#endif

///////////////////  Tcl variables and functions  //////////////////
#ifdef VMDTCL

#include <tcl.h>
#include "Global.h"

// the Tcl interpreter; if it is NULL, it is not available
static Tcl_Interp *tclInterp = NULL;

// number of TextEvent's which are using tclInterp; used to determine when
// to create and destroy the interpreter
static int numUsingTcl = 0;


// function called when a VMD command is encountered; this will then call
// the VMD callback function, and set the return codes, etc.
int UIText_process_Tcl(ClientData clientData, Tcl_Interp *,
			int argc, char *argv[]) {
  // clientData holds the pointer to the UIText object that invoked this
  UIText *uitxt = (UIText *)clientData;

  // now call the text processor callback function for the word
  // and return proper code
  if(uitxt->process_command(argc,argv) == 0)
    // update the screen
    VMDupdate(FALSE);

  // since we print out an error message if necessary we just always return OK
  return TCL_OK;
}

#endif


///////////////////  special text callback functions  //////////////////

// text callback routine for commands which are optional, and have not been
// included in this executable
int text_cmd_unsupported(int, char **argv, CommandQueue *, int) {
  msgWarn << "'" << argv[0] << "' commands are not supported in this version";
  msgWarn << " of the program.  Please recompile with the option enabled.";
  msgWarn << sendmsg;
  return TRUE;
}


////////////////////////////  TextEvent routines  ///////////////////////////

// constructor
TextEvent::TextEvent(char *str, int delcmd, int newUIid)
	: Command(Command::TEXT_EVENT, newUIid) {
  deleteCmd = delcmd;
  cmd = str;
  // get rid of newline at end, if any
  if(cmd[strlen(cmd) - 1] == '\n')
    cmd[strlen(cmd) - 1] = '\0';
}

// destructor; frees up command string memory if necessary
TextEvent::~TextEvent(void) {
  if(deleteCmd && cmd)
    delete [] cmd;
}


////////////////////////////  UIText routines  ///////////////////////////

// constructor
UIText::UIText(UIList *uil, CommandQueue *cq)
  : UIObject("Text Console",uil, cq), input_files(64), txtMsgEcho("Echo") {

  MSGDEBUG(1,"Creating UIText ..." << sendmsg);

  // record which commands we want
  command_wanted(Command::TEXT_EVENT);

  // display system prompt
  need_prompt();
  
  // don't wait for anything
  delay = 0;
  mytimer.clear();

  // echoing is off by default
  doEcho = FALSE;

  // initialize Tcl, if necessary
#ifdef VMDTCL
  if(!(numUsingTcl++))
    tclInterp = Tcl_CreateInterp();
#endif

  // register all the text command here.  Enter them with the 'add_command'
  // routine, in alphabetical order.
  // NOTE: for command which are optional, either add the command with the
  //	the proper callback function, or add it with the callback function
  //	'text_cmd_unsupported', via an ifdef/else/endif sequence.

  add_command("animate",	text_cmd_animate);
  add_command("axes",		text_cmd_axes);
  add_command("color",		text_cmd_color);
  add_command("debug",		text_cmd_debug);
  add_command("display",	text_cmd_display);
  add_command("echo",		text_cmd_echo);
#ifdef VMDEXTERNAL
  add_command("external",	text_cmd_external);
#else
  add_command("external",	text_cmd_unsupported);
#endif
  add_command("help",		text_cmd_help);
  add_command("journal",	text_cmd_log);
  add_command("label",		text_cmd_label);
  add_command("light",		text_cmd_light);
  add_command("log",		text_cmd_log);
  add_command("menu",		text_cmd_menu);
  add_command("mol",		text_cmd_mol);
  add_command("molecule",	text_cmd_mol);
  add_command("mouse",		text_cmd_mouse);
  add_command("play",		text_cmd_play);
  add_command("run",		text_cmd_play);
  add_command("render",		text_cmd_render);
  add_command("rock",		text_cmd_rock);
  add_command("rot",		text_cmd_rotate);
  add_command("rotate",		text_cmd_rotate);
  add_command("rotation",	text_cmd_rotate);
#ifdef VMDREMOTE
  add_command("remote",		text_cmd_remote);
  add_command("sim",		text_cmd_sim);
  add_command("simulation",	text_cmd_sim);
#else
  add_command("remote",		text_cmd_unsupported);
  add_command("sim",		text_cmd_unsupported);
  add_command("simulation",	text_cmd_unsupported);
#endif
  add_command("scale",		text_cmd_scale);
#ifdef VMDSIGMA
  add_command("sigma",		text_cmd_sigma);
#else
  add_command("sigma",		text_cmd_unsupported);
#endif
  add_command("stage",		text_cmd_stage);
  add_command("tool",		text_cmd_tool);
  add_command("tracker",	text_cmd_tracker);
  add_command("trans",		text_cmd_translate);
  add_command("translate",	text_cmd_translate);
  add_command("user",		text_cmd_user);
  add_command("wait",		text_cmd_wait);

  add_command("quit",		text_cmd_quit);
  add_command("exit",		text_cmd_quit);
  add_command("stop",		text_cmd_quit);
}


// destructor
UIText::~UIText(void) {
#ifdef VMDTCL
  if(--numUsingTcl < 1) {
    Tcl_DeleteInterp(tclInterp);
    tclInterp = NULL;
  }
#endif
}


// display the prompt for the user
void UIText::prompt(void) {

  printf(VMD_CMD_PROMPT,myName);
  fflush(stdout);
  
  // tell Message objects they need a newline before next output message
  msgInfo.need_newline(TRUE);
  msgWarn.need_newline(TRUE);
  msgErr.need_newline(TRUE);
  msgDebug.need_newline(TRUE);
}


// return number of text commands currently understood
int UIText::num_commands(void) { return textProcessors.num(); }


// return the Nth word understood; NULL if error
char *UIText::word(int n) {
  if(n < 0 || n >= num_commands())
    return NULL;
  else
    return textProcessors.name(n);
}


// add a new text command
void UIText::add_command(char *nm, TextCallback *cb) {
  textProcessors.add_name(nm, cb);

  // if using Tcl, register Tcl callback as well
#ifdef VMDTCL
  Tcl_CreateCommand(tclInterp, nm, UIText_process_Tcl,
	(ClientData)this, (Tcl_CmdDeleteProc *) NULL);
#endif
}


// process an argc, argv command, and return < 0 if the command is not
// known, 0 if it was successful, or > 0 if an error occcurred
int UIText::process_command(int argc, char **argv) {

  // make sure we have at least one word
  if(argc < 1)
    return 1;

  // search for the word, and when found execute it, otherwise an error
  int wordindx = textProcessors.typecode(argv[0], CMDLEN);
  if(wordindx < 0) {
    msgErr << "Unknown command '" << argv[0] << "'."
           << sendmsg;
    return (-1);
  }

  // now call the text processor callback function for the word
  int retval=(*(textProcessors.data(wordindx))) (argc, argv, cmdQueue, id());

  // print if an error occurred
  if(retval)
    msgErr << "Invalid format for command '" << argv[0] << "'." << sendmsg;

  return retval;
}


// specify new file to read commands from
void UIText::read_from_file(char *fname) {
  FILE *f;
  if((f = fopen(fname,"r")) != NULL) {
    msgInfo << "Reading commands from '" << fname << "'." << sendmsg;
    input_files.push(f);
  } else {
    msgErr << "Cannot open file '" << fname << "' for reading." << sendmsg;
  }
}


// set the text processor to wait for the given number of seconds before
// reading another text command
void UIText::wait(float wd) {
  mytimer.stop();
  mytimer.reset();
  delay = wd;
  if (delay > 0.0)
    mytimer.start();
  else
    delay = 0.0;
}


// check for an event; return TRUE if we found an event; FALSE otherwise
int UIText::check_event(void) {
  static char cmdbuffer[1024];
  fd_set readvec;
  int ret, stdin_fd;
  struct timeval timeout;

  // check to see if I've waited long enough
  if (delay > 0.0)
    if (delay < mytimer.clock_time())
      wait(-1.0);
    else
      return FALSE;  // gotta wait some more

  // check for text from input device
  if(input_files.stack_size() < 1) {
    if(needPrompt) {
      prompt();
      needPrompt = FALSE;
    }
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    stdin_fd = 0;
    FD_ZERO(&readvec);
    FD_SET(stdin_fd, &readvec);
#if defined(ARCH_IRIX4) || defined(ARCH_IRIX5) || defined(ARCH_IRIX6)
    ret = select(16,&readvec,NULL,NULL,&timeout);
#else
    ret = select(16,(int *)(&readvec),NULL,NULL,&timeout);
#endif
  
    if (ret == -1) {
      // got an error
      msgErr << "Error from select, while attempting to read text input."
             << sendmsg;
      return FALSE;
    } else if (ret == 0) {
      // time out
      return FALSE;
    }

    if (fgets(cmdbuffer,1024,stdin) == NULL) {
      addcommand(new CmdQuit(FALSE,id()));
      return FALSE;
    }
    
  } else {
    // just read in next line from current file, if available
    if(!fgets(cmdbuffer, 1024, input_files.top())) {
      msgInfo << "EOF encountered for current input file." << sendmsg;
      input_files.pop();	// now go back to reading previous file
      return FALSE;
    }
  }

  // strip off trailing newline
  int cmdstrlen = strlen(cmdbuffer);
  if(cmdstrlen > 0 && cmdbuffer[cmdstrlen - 1] == '\n')
    cmdbuffer[cmdstrlen - 1] = '\0';

  MSGDEBUG(3, "Read command: " << cmdbuffer << sendmsg);

  // create a text event, and queue it; the event object must deallocate the
  // string space allocated to hold the text command.
  addcommand(new TextEvent(stringdup(cmdbuffer), TRUE, id()));

  return TRUE;
}


// update the display due to a command being executed.  Return whether
// any action was taken on this command.
// Arguments are the command type, command object, and the 
// success of the command (T or F).
int UIText::act_on_command(int type, Command *cmd, int) {

  MSGDEBUG(3,"UIText: acting on command " << type << sendmsg);

  // check all the possible commands that we look for ...
  if(type == Command::TEXT_EVENT) {

    // echo the command to the console, if necessary
    if(doEcho)
      txtMsgEcho << ((TextEvent *)cmd)->cmd << sendmsg;

#ifdef VMDTCL
    // we're using Tcl, so ask the interpreter to parse it and possibly
    // call a callback function when a VMD command is encountered
    if(Tcl_Eval(tclInterp, ((TextEvent *)cmd)->cmd) == TCL_OK) {
      // command went OK, print out result if necessary
      if(*(tclInterp->result) != NULL)
        msgInfo << *(tclInterp->result) << sendmsg;
    }
#else
    // it's a text event, and we're not using Tcl, so tokenize and process
    // the command
    int argc;
    char *argv[256];

    if(!str_tokenize(((TextEvent *)cmd)->cmd, &argc, argv)) {
      need_prompt();
      return FALSE;
    }

    // make sure there is at least ONE token in the command, and skip comments
    if(argc < 1 || argv[0][0] == '#')
      ;		// we processed it, but did nothing 
    else
      // search for the word, and when found execute it, otherwise an error
      process_command(argc, argv);
#endif

    // display the prompt again, if the command was read from the text
    if(cmd->getUIid() == id())
      need_prompt();
    
  } else
    // unknown command type
    return FALSE;

  return TRUE;
}


/* REVISION HISTORY:********************************************************
 *
 * $Log: UIText.C,v $
 * Revision 1.37  1995/05/13  01:40:18  billh
 * Added accessor functions for the text commands; changed to have
 * the screen updated after every Tcl command is executed.
 *
 * Revision 1.36  95/05/12  18:25:23  billh
 * Added Tcl support.
 * 
 * Revision 1.35  95/05/12  00:00:19  billh
 * Using general text command callback function system - UIText maintains
 * a NameList of functions for all the commands it understands.  When a
 * text command is entered, the first word is checked in the list; if found,
 * the text callback function is called to parse and operate on the
 * command.  Thus all the text processing operations have been removed
 * from UIText, and UIText2.C  is no more.  Several new functions have been
 * added to UIText to allow external access to echoing, logging, and other
 * options.  New commands can be added by any other part of VMD by simply
 * providing a word and a callback function.
 * 
 * Revision 1.34  95/04/19  20:12:17  billh
 * Events no longer create a (commented) text version.
 * Fixed some text command bugs (discrepancies between Cmd and UIText).
 * 
 * Revision 1.33  1995/04/13  17:13:51  billh
 * Added changes to allow Sigma connection and addition of forces.
 * Added ability to specify words like 'top' and 'fixed' when issuing a
 * command to affect the molecules (i.e. mol list displayed)
 *
 * Revision 1.32  95/04/05  02:02:45  billh
 * Fixed problem with rotating smoothly; increment now always taken
 * as positive, and better error checking for when increment >= angle.
 * 
 * Revision 1.31  95/04/04  21:51:16  dalke
 * fixed usage of TextEvent
 * 
 * Revision 1.30  1995/03/31  16:03:04  billh
 * Changed CmdRotate: added ability to rock just once, so that a
 * smooth rotation can be done with a given increment.
 * New form: 'rot x by 50 1' rotates 50 deg about x axis, in 1-deg steps.
 *
 * Revision 1.29  95/03/24  18:52:26  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.28  1995/03/04  05:14:26  billh
 * Added geomtry labelling commands.
 *
 * Revision 1.27  1995/02/22  08:44:28  dalke
 * added external interface commands
 *
 * Revision 1.26  1994/12/07  07:48:33  billh
 * Temporarily took out text commands to change alpha, shininess, etc.
 * of colors.  Should be updated and put in UIText2.C
 *
 * Revision 1.25  94/12/06  08:22:44  billh
 * Added color commands.
 * 
 * Revision 1.24  94/11/24  07:27:03  dalke
 * Added "render format filename" and "render list"
 * 
 * Revision 1.23  1994/11/22  02:35:24  billh
 * added anim read,write,delete text commands.
 *
 * Revision 1.22  94/11/07  07:00:26  dalke
 * Added some commands to push/ pop Tools
 * 
 * Revision 1.21  1994/11/03  03:41:05  dalke
 * fixed modselect bug
 * fixed "scale by" bug
 * made delay a float
 *
 * Revision 1.20  1994/11/03  00:14:25  dalke
 * Added the 'wait' command
 *
 * Revision 1.19  1994/11/02  07:31:00  billh
 * Added commands to change display screen size and distance to screen.
 *
 * Revision 1.18  94/10/26  23:21:34  billh
 * Added routines to change the settings for graphics representations
 * (MOL_MODREP, MOL_MODREPITEM).
 * 
 * Revision 1.17  94/10/20  01:32:41  billh
 * Added animation commands.
 * 
 * Revision 1.16  1994/10/05  05:33:01  billh
 * arguments changed to allow to compile on HP's.
 *
 * Revision 1.15  1994/10/03  08:32:14  dalke
 * Added CmdTool Commands
 *
 * Revision 1.14  1994/10/03  01:40:25  dalke
 * Added a #include
 *
 * Revision 1.13  1994/10/01  11:01:00  billh
 * Added remote connection commands, put menu parsing commands in a
 * separate routine.
 *
 * Revision 1.12  94/09/30  19:11:34  dalke
 * took out references to forms when not compiled with VMDFORMS
 * 
 * Revision 1.11  1994/09/29  07:10:21  billh
 * Molecule commands updated to allow lists of indices.
 *
 * Revision 1.10  1994/09/26  18:52:36  billh
 * Updated molecule commands for adding and listing.
 *
 * Revision 1.9  94/09/23  00:59:37  billh
 * Added many more molecule commands.
 * 
 * Revision 1.8  1994/09/17  09:03:30  billh
 * Put in initial molecule commands (load and list).
 *
 * Revision 1.7  94/09/15  07:03:00  dalke
 * Made tracker text output commands into Commands
 * 
 * Revision 1.6  1994/09/14  04:10:59  billh
 * Removed some excess debugging text.
 *
 * Revision 1.5  1994/09/12  20:53:16  billh
 * Added 'view reset' command.
 *
 * Revision 1.4  94/09/07  07:49:38  dalke
 * added commands to manipulate the trackers via text
 * 
 * Revision 1.3  1994/09/05  22:50:38  billh
 * Added complete support for commands to position the axes and stage.
 *
 * Revision 1.2  1994/08/26  00:02:55  billh
 * Added 'rot stop' command and fixed problem with stereo mode command.
 *
 * Revision 1.1  94/08/24  03:10:37  billh
 * Initial revision
 * 
 ***************************************************************************/
