/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: UNCTracker.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 1995/05/12 00:10:53 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * A UNCTracker is a device for measuring the 3D spacial position of an input
 * device; it is a subclass of the Tracker class.
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "UNCTracker.h"

/**********************  private member functions  *********************/

// routines to create the source and output data structures

#define POL_H	"0"
#define POL_W	"0"

#define POL_XMIN	"0"
#define POL_XMAX	"0"
#define POL_YMIN	"0"
#define POL_YMAX	"0"
#define POL_ZMIN	"0"
#define POL_ZMAX	"0"


/**********************  public member functions  *********************/

// class constructor
// NOTE: this uses the UNC code for initializing the UNCTracker
UNCTracker::UNCTracker(char *name, int num_sensors) : Tracker(num_sensors) {
   register int i;

      readRetval = T_ERROR;

      // open and initialize the polhemus
      if(!name) {
        fprintf(stderr,"Opening tracker '%s' ...\n",getenv("TRACKER"));
        polIndex = t_open(T_ENV_TRACKER,0);
      } else {
        strncpy(polName,name,T_MAX_TRACKER_NAME_LENGTH-1);
        polName[T_MAX_TRACKER_NAME_LENGTH - 1] = '\0';
        fprintf(stderr,"Opening tracker '%s' ...\n",polName);
        polIndex = t_open(&polName,0);
      }

      if(polIndex == T_NULL_TRACKER) {
        fprintf(stderr,"Error opening the UNCTracker.\n");
        status = TRACKER_NOK;
      } else {
        status = TRACKER_OK;
        // indicate which sensors can be used
        for(i=0; i < sensors; i++) {
          if(t_enable(polIndex,i) != T_OK) {
            fprintf(stderr, "UNCTracker: Cannot enable sensor %d\n", i);
            sensorAvailable[i] = FALSE;
          } else {
            sensorAvailable[i] = TRUE;
          }
        }

        // set the format for data to be returned to 4x4 matrix
        if(t_data_format(polIndex,T_XFORM_MATRIX) != T_OK) {
          fprintf(stderr,"UNCTracker: cannot set data format to T_XFORM_MATRIX\n");
        }

        // set the coordinate frame to be the room's frame and orientation
        if(t_frame(polIndex,T_ROOM_FRAME) != T_OK) {
          fprintf(stderr,"UNCTracker: cannot set frame to T_ROOM_FRAME\n");
        }

        // get initial position and orientation
        update_sensors();

   }

}

// class destructor
// NOTE: this uses the UNC code for closing the UNCTracker
UNCTracker::~UNCTracker(void) {
   int i;
     if (polIndex<0) return;
      for(i=0; i < sensors; i++) {
         sensorAvailable[i] = FALSE;
         t_disable(polIndex,i);
      }
      t_close(polIndex);

}

// function to update the sensor positions
// NOTE: this uses the UNC code for reading the UNCTracker
// Returns TRUE if sensor data read and values returned, FALSE otherwise.
// If do_update is false, this will just return results from last read.
int UNCTracker::my_get_data(int sensor, float *x, float *y, float *z,
  float *orient, int do_update) {
   register int i,j,k, count;
   float *or = orient;

   if( polIndex == T_NULL_TRACKER)
     return FALSE;
  if (status != TRACKER_OK) {
     *x = *y = *z = 0;
     orient[0]=orient[4]=orient[8]=1;
     orient[1]=orient[2]=orient[3]=orient[5]=orient[6]=orient[7]=0;
     return FALSE;
    }

   // get the data from the sensor (UNC routine), if necessary
   if(readRetval != T_OK || do_update) {
     readRetval = t_read(polIndex, uncReport);
     if(readRetval != T_OK) {
       if(readRetval == T_ERROR) {
	 fprintf(stderr,"UNCTracker: Error reading sensor data.\n");
       } else {
         fprintf(stderr,"UNCTracker: Strange error during read.\n");
       }
       return FALSE;
     }
   }

   // return with no change if sensor is not available for reading
   if(!sensorAvailable[sensor])
     return FALSE;
   
   // now retreive this data
   for(i=0, count=0; i < sensor; i++) {
     if(sensorAvailable[i])
       count++;
   }

   *x = (float)(uncReport[count].matrix[3][0]);
   *y = (float)(uncReport[count].matrix[3][1]);
   *z = (float)(uncReport[count].matrix[3][2]);
   for(j=0; j < 3; j++)
     for(k=0; k < 3; k++)
       *(or++) = (float)(uncReport[count].matrix[j][k]);
   
   return TRUE;
}

/* REVISION HISTORY:********************************************************
 *
 * $Log: UNCTracker.C,v $
 * Revision 1.4  1995/05/12  00:10:53  billh
 * Moved log messages to end of file.
 *
 * Revision 1.3  95/03/24  18:52:33  billh
 * Added copyright notice to top of file; made sure all virtual routines
 * are defined in the .C file, not in the .h file.
 * 
 * Revision 1.2  1994/11/08  08:18:20  dalke
 * added a misssing return TRUE
 *
 * Revision 1.1  1994/09/03  11:18:18  dalke
 * Initial revision
 *
 * Revision 1.8  1994/04/20  23:08:32  billh
 * Fixed RCS header in files to include description; set most files to
 * include Global.h, and not ever main.h.  Global.h includes config.h and
 * utilities.h automatically, as well as Message.h for the output message
 * objects.
 *
 * Revision 1.7  1994/04/16  00:40:21  billh
 * Changed program name to vmd, added RCS header section.
 *
 ***************************************************************************/
