/*************************************************************************

 cave_ogl.h

 Authors:	Carolina Cruz-Neira, Dave Pape
 Version:	2.4.3
 Last revised:	30 March 1995

  This file contains the definitions required by CAVE applications.

**************************************************************************/

/*
*	Copyright 1994, Electronic Visualization Laboratory.
*	All rights reserved.
*
*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF EVL.
*	Unless otherwise agreed to in writing by EVL, this code is
*	to be used and accessed only by SIGGRAPH VROOM participants.
*	It is not public domain software.  Call Margaret Rawlings at
*	(312)-996-3002 for details.
*
*/

#ifndef	__CAVE_H__
#define	__CAVE_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <math.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>
#include <ulocks.h>


/* Wall IDs - for CAVEWall & CAVEConfig->ActiveWalls */
#define	CAVE_MAX_WALLS		16
#define  CAVE_FRONT_WALL  	1
#define  CAVE_FLOOR_WALL  	2
#define  CAVE_RIGHT_WALL  	3
#define  CAVE_LEFT_WALL   	4
#define  CAVE_CEILING_WALL	5
#define  CAVE_BACK_WALL		6
#define  CAVE_ARPAFLOOR_WALL  	7
#define  CAVE_DESK_WALL		8
#define  CAVE_SIMULATOR_WALL	9
#define  CAVE_SIMULATOR1_WALL	10
#define  CAVE_SIMULATOR2_WALL	11
#define  CAVE_DUAL_EYE_WALL	12
#define  CAVE_LEFT_EYE_WALL	13
#define  CAVE_RIGHT_EYE_WALL	14
#define  CAVE_FRONT_LEFT_WALL	15
#define  CAVE_FRONT_RIGHT_WALL	16
#define CAVE_MAX_WALL_ID	16


#define  ON                1
#define  OFF               0


/* Wand input device macros */
#define  CAVEBUTTON1    (CAVEController->button[0])
#define  CAVEBUTTON2    (CAVEController->button[1])
#define  CAVEBUTTON3    (CAVEController->button[2])
#define  CAVEBUTTON4    (CAVEController->button[3])

#define CAVE_JOYSTICK_X	(CAVEController->valuator[0])
#define CAVE_JOYSTICK_Y	(CAVEController->valuator[1])

#define CAVE_MAX_BUTTONS	8
#define CAVE_MAX_VALUATORS	4


/* Values for CAVEConfig->DisplayMode & CAVEConfig->NonCAVEDisplayMode */
#define CAVE_MONO		0
#define CAVE_STEREO_96		1
#define CAVE_STEREO_120		2
#define CAVE_SETMON		3
/* Additional values used only for CAVEConfig->NonCAVEDisplayMode */
#define CAVE_MONO_96		4
#define CAVE_MONO_60		5
#define CAVE_MONO_72		6


/* Values for CAVEConfig->WandType */
#define CAVE_NO_WAND		0
#define CAVE_MOUSE_WAND		1
#define CAVE_PC_WAND		2
#define CAVE_SIMULATOR_WAND	3
#define CAVE_LOGITECH_WAND	4
#define CAVE_CUSTOM_WAND	5


/* Values for CAVEConfig->TrackerDevice */
#define CAVE_NO_TRACKING	0
#define CAVE_POLHEMUS		1
#define CAVE_BIRDS		2
#define CAVE_LOGITECH		3
#define CAVE_MOUSETRACKER	4
#define CAVE_SIMTRACKER		5
#define CAVE_SPACEBALL		6
#define CAVE_BOOM		7


/* Values for CAVEConfig->Units */
#define CAVE_FEET		1
#define CAVE_METERS		2
#define CAVE_INCHES		3
#define CAVE_CENTIMETERS	4


/* Values for CAVEConfig->BirdsHemisphere */
#define CAVE_FRONT_HEMI		1
#define CAVE_LEFT_HEMI		2
#define CAVE_RIGHT_HEMI		3
#define CAVE_AFT_HEMI		4
#define CAVE_UPPER_HEMI		5
#define CAVE_LOWER_HEMI		6


/* Miscellaneous constants for utility functions */
typedef enum {
		CAVE_HEAD,
		CAVE_WAND,
		CAVE_LEFT_EYE,
		CAVE_RIGHT_EYE,
		CAVE_HEAD_FRONT,
		CAVE_HEAD_BACK,
		CAVE_HEAD_LEFT,
		CAVE_HEAD_RIGHT,
		CAVE_HEAD_UP,
		CAVE_HEAD_DOWN,
		CAVE_WAND_FRONT,
		CAVE_WAND_BACK,
		CAVE_WAND_LEFT,
		CAVE_WAND_RIGHT,
		CAVE_WAND_UP,
		CAVE_WAND_DOWN
		} CAVEID;


/* CAVE display window description (from WallDisplay configuration) */
typedef struct {
	int xoff,yoff;
	int xdim,ydim;
	int window;	/* 1 == "normal" window - framed, movable, etc. */
	char display_env[64];	/* definition of DISPLAY variable */
	} CAVE_WINDOW_ST;


/* Offsets for tracker devices */
typedef struct {
	float dx,dy,dz;
	} CAVE_OFFSET_ST;

typedef struct {
	float rx,ry,rz;
	} CAVE_ORIENTATION_ST;


/* Description of Immersadesk screen */
typedef struct
	{
	float lowerleft[3];	/* Position of lower left corner */
	float upperleft[3];	/* Position of upper left corner */
	float lowerright[3];	/* Position of lower right corner */
	} CAVE_PROJINFO_ST;

/* Configuration - contains data from config files */
typedef struct {
	int		ActiveWalls;
	int		DisplayWall[CAVE_MAX_WALLS];
	int		WallPipe[CAVE_MAX_WALL_ID+1];
	CAVE_WINDOW_ST	WallGeom[CAVE_MAX_WALL_ID+1];
	int		UseTracker;
	int		TrackerDevice;  /* CAVE_BIRDS / CAVE_LOGITECH / etc. */
	char		TrackerSerialPort[2][15];
	int		TrackerBaudRate;
	int		SyncTracker;
	CAVE_OFFSET_ST	TransmitterOffset;
	CAVE_ORIENTATION_ST	TransmitterOrientation;
	CAVE_OFFSET_ST	WandSensorOffset;
	float		WandSensorElevation;
	CAVE_OFFSET_ST	HeadSensorOffset;
	float		InterocularDistance;
	float		HalfInterocularDistance;
	int		UseCalibration;
	char		CalibrationFile[256];
	char		TrackerLogfile[256];
	float		CAVEWidth;
	float		CAVEHeight;
	float		OrigX,OrigY,OrigZ;
	int		Units;
	char		AudioServer[256];
	int		DisplayMode;
	char		DisplayModeSetmon[256];
	int		NonCAVEDisplayMode;
	char		NonCAVEDisplayModeSetmon[256];
	int		WandType;
	int		CPULock;
	int		HideCursor;
	int		Simulator;
	float		CAVEScale;
	KeySym		SimulatorControls[20];
	int		WandButtonsCount;
	int		WandButtons[CAVE_MAX_BUTTONS];
	int		WandValuatorsCount;
	int		WandValuators[CAVE_MAX_VALUATORS];
	int		WandValuatorsMin[CAVE_MAX_VALUATORS];
	int		WandValuatorsMax[CAVE_MAX_VALUATORS];
	float		SimViewDistance,SimViewHeight,SimViewWidth;
	int		SerialTracking;
	int		BirdsHemisphere;
	CAVE_PROJINFO_ST	ProjectionData[CAVE_MAX_WALL_ID+1];
	int		InStereo;	/* NB: Derived, not set in config file */ 
	}  CAVE_CONFIG_ST;


/* Display process synchronization data */
typedef	struct {
	unsigned int 	Initted;
	unsigned int	HostMaster;
	unsigned int	Quit;
	} CAVE_SYNC_ST;


/* Position & orientation data for trackers */
typedef struct {
	float 			x, y, z;
	float 			azi, elev, roll;
	} CAVE_SENSOR_ST;


/* Collection of buttons & valuators associated with wand */
typedef struct {
	int	num_buttons,num_valuators;
	int	button[CAVE_MAX_BUTTONS];
	float	valuator[CAVE_MAX_VALUATORS];
	} CAVE_CONTROLLER_ST;


/* Tracker logfile header */
#define CAVE_LOG_MAGIC  51902
typedef struct {
		unsigned long magic;
		char cave_version[80];
		char start_time[32];
		int num_sensors, num_buttons, num_valuators;
		char pad[128];  /* Pad to 256 bytes */
		} CAVE_LOG_HEADER_ST;


/* CAVE lock pointer */
typedef struct CAVE_LOCK_ST * CAVELOCK;



void	CAVEInit(void);
void	CAVEExit(void);
void	CAVEDisplay(void (*)(), int, ...);
void	CAVEInitApplication( void (*their_draw)(), int argc, ...);
void	*CAVEUserSharedMemory(int);
void	CAVEFrameFunction(void (*)(), int, ...);
void	CAVEStopApplication(void (*)(), int, ...);
void	CAVEGetEyePosition(int eye,float *x,float *y,float *z);
void	CAVEConfigure(int *argc,char **argv,char **appdefaults);
int	CAVEMasterDisplay(void);
void	CAVEDisplayBarrier(void);

int	CAVEButtonChange(int num);
void	CAVEGetVector(CAVEID vecname,float *vec);
void	CAVEGetPosition(CAVEID posname,float *pos);
void	CAVEGetOrientation(CAVEID oname,float *or);
double	CAVEGetTime(void);
void	CAVEWallTransform(void);
void	CAVEHeadTransform(void);
void	CAVEWandTransform(void);

void	CAVENavigationInit(float, float);
void	CAVESetNavigationParm(float, float);
void	CAVENavigate();
void	CAVECalculateNavigation();

CAVELOCK	CAVENewLock(void);
void	CAVEFreeLock(CAVELOCK lock);
void	CAVESetReadLock(CAVELOCK lock);
void	CAVESetWriteLock(CAVELOCK lock);
void	CAVEUnsetReadLock(CAVELOCK lock);
void	CAVEUnsetWriteLock(CAVELOCK lock);

void	CAVENavLoadIdentity(void);
void	CAVENavTranslate(float x,float y,float z);
void	CAVENavRot(float a,char axis);
void	CAVENavScale(float x,float y,float z);
void	CAVENavConvertWorldToCAVE(float *in,float *out);
void	CAVENavConvertCAVEToWorld(float *in,float *out);
void	CAVENavConvertVectorWorldToCAVE(float *in,float *out);
void	CAVENavConvertVectorCAVEToWorld(float *in,float *out);
void	CAVENavTransform(void);

void	CAVEResetTracker(void);


/* CAVE library global data;  data in shared memory is declared volatile */
extern	volatile float     	*CAVEFramesPerSecond;
extern	volatile float		*CAVETime;
extern	int                 	CAVEWall;
extern	int			CAVEPipe;
extern	volatile CAVE_CONFIG_ST	*CAVEConfig;
extern	volatile CAVE_SYNC_ST  	*CAVESync;     
extern	volatile CAVE_SENSOR_ST	*CAVESensor1, *CAVESensor2;
extern  float    		CAVENear, CAVEFar;
extern	int			CAVEEye;
extern	char 			*CAVEVersion;
extern	volatile CAVE_CONTROLLER_ST	*CAVEController;



/*************************************************************************/
/**                  SENSOR RELATED MACROS                              **/
/*************************************************************************/

#define  CAVEGetHead(xhead, yhead, zhead) \
   {                                   \
   xhead = CAVESensor1 -> x;              \
   yhead = CAVESensor1 -> y;              \
   zhead = CAVESensor1 -> z;               \
   }
#define  CAVEGetWand(xw,  yw,  zw) \
   { \
    xw = CAVESensor2 -> x; \
    yw = CAVESensor2 -> y; \
    zw = CAVESensor2 -> z; \
   }
    
#define  CAVEGetHeadOrientation(azimuth, elevation, wroll) \
   {                                   \
   azimuth = CAVESensor1 -> azi;              \
   elevation = CAVESensor1 -> elev;              \
   wroll = CAVESensor1 -> roll;               \
   }

#define  CAVEGetWandOrientation(azimuth, elevation, wroll) \
   { \
   azimuth = CAVESensor2 -> azi;              \
   elevation = CAVESensor2 -> elev;              \
   wroll = CAVESensor2 -> roll; \
   }


#define  CAVEWandOrientation  \
   { \
   glTranslatef( CAVESensor2 -> x,  CAVESensor2 -> y, CAVESensor2 -> z); \
   glRotatef(CAVESensor2->azi, 0.0f,1.0f,0.0f); \
   glRotatef(CAVESensor2->elev, 1.0f,0.0f,0.0f); \
   glRotatef(CAVESensor2->roll, 0.0f,0.0f,1.0f); \
   glTranslatef( -CAVESensor2 -> x,  -CAVESensor2 -> y, -CAVESensor2 -> z); \
   }
   
#define  CAVEHeadOrientation  \
   { \
   glTranslatef( CAVESensor1 -> x,  CAVESensor1 -> y, CAVESensor1 -> z); \
   glRotatef(CAVESensor1->azi, 0.0f,1.0f,0.0f); \
   glRotatef(CAVESensor1->elev, 1.0f,0.0f,0.0f); \
   glRotatef(CAVESensor1->roll, 0.0f,0.0f,1.0f); \
   glTranslatef( -CAVESensor1 -> x, -CAVESensor1 -> y, -CAVESensor1 -> z); \
   }


/* DTOR is single precision for speed */
#define DTOR(r) ((r)*0.01745329f)


/*************************************/
/* Wand unit direction vector macros */
/*************************************/

#define CAVEGetWandFront(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor2->elev)) * sinf(DTOR(CAVESensor2->azi));	\
 hy = sinf(DTOR(CAVESensor2->elev));					\
 hz = -cosf(DTOR(CAVESensor2->elev)) * cosf(DTOR(CAVESensor2->azi));	\
}

#define CAVEGetWandUp(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->roll)) +	\
	sinf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->elev)) *	\
	cosf(DTOR(CAVESensor2->roll));					\
 hy = cosf(DTOR(CAVESensor2->roll)) * cosf(DTOR(CAVESensor2->elev));	\
 hz = sinf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->roll)) +	\
	cosf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->elev)) *	\
	cosf(DTOR(CAVESensor2->roll));					\
}

#define CAVEGetWandLeft(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor2->azi)) * cosf(DTOR(CAVESensor2->roll)) -	\
	sinf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->elev)) *	\
	sinf(DTOR(CAVESensor2->roll));					\
 hy = -cosf(DTOR(CAVESensor2->elev)) * sinf(DTOR(CAVESensor2->roll));	\
 hz = sinf(DTOR(CAVESensor2->azi)) * cosf(DTOR(CAVESensor2->roll)) -	\
	cosf(DTOR(CAVESensor2->azi)) * sinf(DTOR(CAVESensor2->elev)) *	\
	sinf(DTOR(CAVESensor2->roll));					\
}

/*************************************/
/* Head unit direction vector macros */
/*************************************/

#define CAVEGetHeadFront(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor1->elev)) * sinf(DTOR(CAVESensor1->azi));	\
 hy = sinf(DTOR(CAVESensor1->elev));					\
 hz = -cosf(DTOR(CAVESensor1->elev)) * cosf(DTOR(CAVESensor1->azi));	\
}

#define CAVEGetHeadUp(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->roll)) +	\
	sinf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->elev)) *	\
	cosf(DTOR(CAVESensor1->roll));					\
 hy = cosf(DTOR(CAVESensor1->roll)) * cosf(DTOR(CAVESensor1->elev));	\
 hz = sinf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->roll)) +	\
	cosf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->elev)) *	\
	cosf(DTOR(CAVESensor1->roll));					\
}

#define CAVEGetHeadLeft(hx, hy, hz)    \
{   \
 hx = -cosf(DTOR(CAVESensor1->azi)) * cosf(DTOR(CAVESensor1->roll)) -	\
	sinf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->elev)) *	\
	sinf(DTOR(CAVESensor1->roll));					\
 hy = -cosf(DTOR(CAVESensor1->elev)) * sinf(DTOR(CAVESensor1->roll));	\
 hz = sinf(DTOR(CAVESensor1->azi)) * cosf(DTOR(CAVESensor1->roll)) -	\
	cosf(DTOR(CAVESensor1->azi)) * sinf(DTOR(CAVESensor1->elev)) *	\
	sinf(DTOR(CAVESensor1->roll));					\
}


#ifdef __cplusplus
}
#endif

#endif
