/* This may look like C code, but it is really -*- C++ -*- */
/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of the GNU C++ Library.  This library is free
software; you can redistribute it and/or modify it under the terms of
the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This library is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#ifndef _String_h
#ifdef __GNUG__
#pragma interface
#endif
#define _String_h 1

#include <iostream.h>
//#include <Regex.h>  -- APD
#include "Regex.h"

struct StrRep                     // internal String representations
{
  unsigned short    len;         // string length 
  unsigned short    sz;          // allocated space
  char              s[1];        // the string starts here 
                                 // (at least 1 char for trailing null)
                                 // allocated & expanded via non-public fcts
};

// primitive ops on StrReps -- nearly all String fns go through these.

StrRep*     Salloc(StrRep*, const char*, int, int);
StrRep*     Scopy(StrRep*, const StrRep*);
StrRep*     Scat(StrRep*, const char*, int, const char*, int);
StrRep*     Scat(StrRep*, const char*, int,const char*,int, const char*,int);
StrRep*     Sprepend(StrRep*, const char*, int);
StrRep*     Sreverse(const StrRep*, StrRep*);
StrRep*     Supcase(const StrRep*, StrRep*);
StrRep*     Sdowncase(const StrRep*, StrRep*);
StrRep*     Scapitalize(const StrRep*, StrRep*);

// These classes need to be defined in the order given

class GString;
class SubString;

class SubString
{
  friend class      GString;
protected:

  GString&           S;        // The GString I'm a substring of
  unsigned short    pos;      // starting position in S's rep
  unsigned short    len;      // length of substring

  void              assign(const StrRep*, const char*, int = -1);
                    SubString(GString& x, int p, int l);
                    SubString(const SubString& x);

public:

// Note there are no public constructors. SubStrings are always
// created via GString operations

                   ~SubString();

  SubString&        operator =  (const GString&     y);
  SubString&        operator =  (const SubString&  y);
  SubString&        operator =  (const char* t);
  SubString&        operator =  (char        c);

// return 1 if target appears anywhere in SubString; else 0

  int               contains(char        c) const;
  int               contains(const GString&     y) const;
  int               contains(const SubString&  y) const;
  int               contains(const char* t) const;
  int               contains(const Regex&       r) const;

// return 1 if target matches entire SubString

  int               matches(const Regex&  r) const;

// IO 

  friend ostream&   operator<<(ostream& s, const SubString& x);

// status

  unsigned int      length() const;
  int               empty() const;
  const char*       chars() const;

  int               OK() const; 

};


class GString
{
  friend class      SubString;

protected:
  StrRep*           rep;   // GStrings are pointers to their representations

// some helper functions

  int               search(int, int, const char*, int = -1) const;
  int               search(int, int, char) const;
  int               match(int, int, int, const char*, int = -1) const;
  int               _gsub(const char*, int, const char* ,int);
  int               _gsub(const Regex&, const char*, int);
  SubString         _substr(int, int);

public:

// constructors & assignment

                    GString();
                    GString(const GString& x);
                    GString(const SubString&  x);
                    GString(const char* t);
                    GString(const char* t, int len);
                    GString(char c);

                    ~GString();

  GString&           operator =  (const GString&     y);
  GString&           operator =  (const char* y);
  GString&           operator =  (char        c);
  GString&           operator =  (const SubString&  y);

// concatenation

  GString&           operator += (const GString&     y); 
  GString&           operator += (const SubString&  y);
  GString&           operator += (const char* t);
  GString&           operator += (char        c);

  void              prepend(const GString&     y); 
  void              prepend(const SubString&  y);
  void              prepend(const char* t);
  void              prepend(char        c);


// procedural versions:
// concatenate first 2 args, store result in last arg

  friend inline void     cat(const GString&, const GString&, GString&);
  friend inline void     cat(const GString&, const SubString&, GString&);
  friend inline void     cat(const GString&, const char*, GString&);
  friend inline void     cat(const GString&, char, GString&);

  friend inline void     cat(const SubString&, const GString&, GString&);
  friend inline void     cat(const SubString&, const SubString&, GString&);
  friend inline void     cat(const SubString&, const char*, GString&);
  friend inline void     cat(const SubString&, char, GString&);

  friend inline void     cat(const char*, const GString&, GString&);
  friend inline void     cat(const char*, const SubString&, GString&);
  friend inline void     cat(const char*, const char*, GString&);
  friend inline void     cat(const char*, char, GString&);

// double concatenation, by request. (yes, there are too many versions, 
// but if one is supported, then the others should be too...)
// Concatenate first 3 args, store in last arg

  friend inline void     cat(const GString&,const GString&, const GString&,GString&);
  friend inline void     cat(const GString&,const GString&,const SubString&,GString&);
  friend inline void     cat(const GString&,const GString&, const char*, GString&);
  friend inline void     cat(const GString&,const GString&, char, GString&);
  friend inline void     cat(const GString&,const SubString&,const GString&,GString&);
  inline friend void     cat(const GString&,const SubString&,const SubString&,GString&);
  friend inline void     cat(const GString&,const SubString&, const char*, GString&);
  friend inline void     cat(const GString&,const SubString&, char, GString&);
  friend inline void     cat(const GString&,const char*, const GString&,    GString&);
  friend inline void     cat(const GString&,const char*, const SubString&, GString&);
  friend inline void     cat(const GString&,const char*, const char*, GString&);
  friend inline void     cat(const GString&,const char*, char, GString&);

  friend inline void     cat(const char*, const GString&, const GString&,GString&);
  friend inline void     cat(const char*,const GString&,const SubString&,GString&);
  friend inline void     cat(const char*,const GString&, const char*, GString&);
  friend inline void     cat(const char*,const GString&, char, GString&);
  friend inline void     cat(const char*,const SubString&,const GString&,GString&);
  friend inline void     cat(const char*,const SubString&,const SubString&,GString&);
  friend inline void     cat(const char*,const SubString&, const char*, GString&);
  friend inline void     cat(const char*,const SubString&, char, GString&);
  friend inline void     cat(const char*,const char*, const GString&,    GString&);
  friend inline void     cat(const char*,const char*, const SubString&, GString&);
  friend inline void     cat(const char*,const char*, const char*, GString&);
  friend inline void     cat(const char*,const char*, char, GString&);


// searching & matching

// return position of target in Gstring or -1 for failure

  int               index(char        c, int startpos = 0) const;      
  int               index(const GString&     y, int startpos = 0) const;      
  int               index(const SubString&  y, int startpos = 0) const;      
  int               index(const char* t, int startpos = 0) const;  
  int               index(const Regex&      r, int startpos = 0) const;       

// return 1 if target appears anyhere in GString; else 0

  int               contains(char        c) const;
  int               contains(const GString&     y) const;
  int               contains(const SubString&  y) const;
  int               contains(const char* t) const;
  int               contains(const Regex&      r) const;

// return 1 if target appears anywhere after position pos 
// (or before, if pos is negative) in GString; else 0

  int               contains(char        c, int pos) const;
  int               contains(const GString&     y, int pos) const;
  int               contains(const SubString&  y, int pos) const;
  int               contains(const char* t, int pos) const;
  int               contains(const Regex&      r, int pos) const;

// return 1 if target appears at position pos in GString; else 0

  int               matches(char        c, int pos = 0) const;
  int               matches(const GString&     y, int pos = 0) const;
  int               matches(const SubString&  y, int pos = 0) const;
  int               matches(const char* t, int pos = 0) const;
  int               matches(const Regex&      r, int pos = 0) const;

//  return number of occurences of target in GString

  int               freq(char        c) const; 
  int               freq(const GString&     y) const;
  int               freq(const SubString&  y) const;
  int               freq(const char* t) const;

// SubString extraction

// Note that you can't take a substring of a const GString, since
// this leaves open the possiblility of indirectly modifying the
// GString through the SubString

  SubString         at(int         pos, int len);
  SubString         operator () (int         pos, int len); // synonym for at

  SubString         at(const GString&     x, int startpos = 0); 
  SubString         at(const SubString&  x, int startpos = 0); 
  SubString         at(const char* t, int startpos = 0);
  SubString         at(char        c, int startpos = 0);
  SubString         at(const Regex&      r, int startpos = 0); 

  SubString         before(int          pos);
  SubString         before(const GString&      x, int startpos = 0);
  SubString         before(const SubString&   x, int startpos = 0);
  SubString         before(const char*  t, int startpos = 0);
  SubString         before(char         c, int startpos = 0);
  SubString         before(const Regex&       r, int startpos = 0);

  SubString         through(int          pos);
  SubString         through(const GString&      x, int startpos = 0);
  SubString         through(const SubString&   x, int startpos = 0);
  SubString         through(const char*  t, int startpos = 0);
  SubString         through(char         c, int startpos = 0);
  SubString         through(const Regex&       r, int startpos = 0);

  SubString         from(int          pos);
  SubString         from(const GString&      x, int startpos = 0);
  SubString         from(const SubString&   x, int startpos = 0);
  SubString         from(const char*  t, int startpos = 0);
  SubString         from(char         c, int startpos = 0);
  SubString         from(const Regex&       r, int startpos = 0);

  SubString         after(int         pos);
  SubString         after(const GString&     x, int startpos = 0);
  SubString         after(const SubString&  x, int startpos = 0);
  SubString         after(const char* t, int startpos = 0);
  SubString         after(char        c, int startpos = 0);
  SubString         after(const Regex&      r, int startpos = 0);


// deletion

// delete len chars starting at pos
  void              del(int         pos, int len);

// delete the first occurrence of target after startpos

  void              del(const GString&     y, int startpos = 0);
  void              del(const SubString&  y, int startpos = 0);
  void              del(const char* t, int startpos = 0);
  void              del(char        c, int startpos = 0);
  void              del(const Regex&      r, int startpos = 0);

// global substitution: substitute all occurrences of pat with repl

  int               gsub(const GString&     pat, const GString&     repl);
  int               gsub(const SubString&  pat, const GString&     repl);
  int               gsub(const char* pat, const GString&     repl);
  int               gsub(const char* pat, const char* repl);
  int               gsub(const Regex&      pat, const GString&     repl);

// friends & utilities

// split Gstring into array res at separators; return number of elements

  friend int        split(const GString& x, GString res[], int maxn, 
                          const GString& sep);
  friend int        split(const GString& x, GString res[], int maxn, 
                          const Regex&  sep);

  friend GString     common_prefix(const GString& x, const GString& y, 
                                  int startpos = 0);
  friend GString     common_suffix(const GString& x, const GString& y, 
                                  int startpos = -1);
  friend GString     replicate(char        c, int n);
  friend GString     replicate(const GString&     y, int n);
  friend GString     join(GString src[], int n, const GString& sep);

// simple builtin transformations

  friend inline GString     reverse(const GString& x);
  friend inline GString     upcase(const GString& x);
  friend inline GString     downcase(const GString& x);
  friend inline GString     capitalize(const GString& x);

// in-place versions of above

  void              reverse();
  void              upcase();
  void              downcase();
  void              capitalize();

// element extraction

  char&             operator [] (int i);
  const char&       operator [] (int i) const;
  char              elem(int i) const;
  char              firstchar() const;
  char              lastchar() const;

// conversion

                    operator const char*() const;
  const char*       chars() const;


// IO

  friend inline ostream&   operator<<(ostream& s, const GString& x);
  friend ostream&   operator<<(ostream& s, const SubString& x);
  friend istream&   operator>>(istream& s, GString& x);

  friend int        readline(istream& s, GString& x, 
                             char terminator = '\n',
                             int discard_terminator = 1);

// status

  unsigned int      length() const;
  int               empty() const;

// preallocate some space for GString
  void              alloc(int newsize);

// report current allocation (not length!)

  int               allocation() const;


  void     error(const char* msg) const;

  int               OK() const;
};

typedef GString StrTmp; // for backward compatibility

// other externs

int        compare(const GString&    x, const GString&     y);
int        compare(const GString&    x, const SubString&  y);
int        compare(const GString&    x, const char* y);
int        compare(const SubString& x, const GString&     y);
int        compare(const SubString& x, const SubString&  y);
int        compare(const SubString& x, const char* y);
int        fcompare(const GString&   x, const GString&     y); // ignore case

extern StrRep  _nilStrRep;
extern GString _nilString;

// status reports, needed before defining other things

inline unsigned int GString::length() const {  return rep->len; }
inline int         GString::empty() const { return rep->len == 0; }
inline const char* GString::chars() const { return &(rep->s[0]); }
inline int         GString::allocation() const { return rep->sz; }

inline unsigned int SubString::length() const { return len; }
inline int         SubString::empty() const { return len == 0; }
inline const char* SubString::chars() const { return &(S.rep->s[pos]); }


// constructors

inline GString::GString() 
  : rep(&_nilStrRep) {}
inline GString::GString(const GString& x) 
  : rep(Scopy(0, x.rep)) {}
inline GString::GString(const char* t) 
  : rep(Salloc(0, t, -1, -1)) {}
inline GString::GString(const char* t, int tlen)
  : rep(Salloc(0, t, tlen, tlen)) {}
inline GString::GString(const SubString& y)
  : rep(Salloc(0, y.chars(), y.length(), y.length())) {}
inline GString::GString(char c) 
  : rep(Salloc(0, &c, 1, 1)) {}

// for some reason, CenterLine (from ObjectCenter) doesn't support
// inline destructors.  HOWEVER, if you don't compile String.C with
// CenterLine's compiler, then you'll need to put the following four
// lines of code somewhere in your code:
// #ifdef __CENTERLINE__
// GString::~GString() { if (rep != &_nilStrRep) delete rep; }
// SubString::~SubString() {}
// #endif

#ifndef __CENTERLINE__
inline GString::~GString() { if (rep != &_nilStrRep) delete rep; }
#endif

inline SubString::SubString(const SubString& x)
  :S(x.S), pos(x.pos), len(x.len) {}
inline SubString::SubString(GString& x, int first, int l)
  :S(x), pos(first), len(l) {}

// same here, for the destructors
#ifndef __CENTERLINE__
inline SubString::~SubString() {}
#endif


// assignment

inline GString& GString::operator =  (const GString& y)
{ 
  rep = Scopy(rep, y.rep);
  return *this;
}

inline GString& GString::operator=(const char* t)
{
  rep = Salloc(rep, t, -1, -1);
  return *this;
}

inline GString& GString::operator=(const SubString&  y)
{
  rep = Salloc(rep, y.chars(), y.length(), y.length());
  return *this;
}

inline GString& GString::operator=(char c)
{
  rep = Salloc(rep, &c, 1, 1);
  return *this;
}


inline SubString& SubString::operator = (const char* ys)
{
  assign(0, ys);
  return *this;
}

inline SubString& SubString::operator = (char ch)
{
  assign(0, &ch, 1);
  return *this;
}

inline SubString& SubString::operator = (const GString& y)
{
  assign(y.rep, y.chars(), y.length());
  return *this;
}

inline SubString& SubString::operator = (const SubString& y)
{
  assign(y.S.rep, y.chars(), y.length());
  return *this;
}

// Zillions of cats...

inline void cat(const GString& x, const GString& y, GString& r)
{
  r.rep = Scat(r.rep, x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const GString& x, const SubString& y, GString& r)
{
  r.rep = Scat(r.rep, x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const GString& x, const char* y, GString& r)
{
  r.rep = Scat(r.rep, x.chars(), x.length(), y, -1);
}

inline void cat(const GString& x, char y, GString& r)
{
  r.rep = Scat(r.rep, x.chars(), x.length(), &y, 1);
}

inline void cat(const SubString& x, const GString& y, GString& r)
{
  r.rep = Scat(r.rep, x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const SubString& x, const SubString& y, GString& r)
{
  r.rep = Scat(r.rep, x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const SubString& x, const char* y, GString& r)
{
  r.rep = Scat(r.rep, x.chars(), x.length(), y, -1);
}

inline void cat(const SubString& x, char y, GString& r)
{
  r.rep = Scat(r.rep, x.chars(), x.length(), &y, 1);
}

inline void cat(const char* x, const GString& y, GString& r)
{
  r.rep = Scat(r.rep, x, -1, y.chars(), y.length());
}

inline void cat(const char* x, const SubString& y, GString& r)
{
  r.rep = Scat(r.rep, x, -1, y.chars(), y.length());
}

inline void cat(const char* x, const char* y, GString& r)
{
  r.rep = Scat(r.rep, x, -1, y, -1);
}

inline void cat(const char* x, char y, GString& r)
{
  r.rep = Scat(r.rep, x, -1, &y, 1);
}

inline void cat(const GString& a, const GString& x, const GString& y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const GString& a, const GString& x, const SubString& y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const GString& a, const GString& x, const char* y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x.chars(), x.length(), y, -1);
}

inline void cat(const GString& a, const GString& x, char y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x.chars(), x.length(), &y, 1);
}

inline void cat(const GString& a, const SubString& x, const GString& y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const GString& a, const SubString& x, const SubString& y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const GString& a, const SubString& x, const char* y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x.chars(), x.length(), y, -1);
}

inline void cat(const GString& a, const SubString& x, char y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x.chars(), x.length(), &y, 1);
}

inline void cat(const GString& a, const char* x, const GString& y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x, -1, y.chars(), y.length());
}

inline void cat(const GString& a, const char* x, const SubString& y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x, -1, y.chars(), y.length());
}

inline void cat(const GString& a, const char* x, const char* y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x, -1, y, -1);
}

inline void cat(const GString& a, const char* x, char y, GString& r)
{
  r.rep = Scat(r.rep, a.chars(), a.length(), x, -1, &y, 1);
}


inline void cat(const char* a, const GString& x, const GString& y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const char* a, const GString& x, const SubString& y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const char* a, const GString& x, const char* y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x.chars(), x.length(), y, -1);
}

inline void cat(const char* a, const GString& x, char y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x.chars(), x.length(), &y, 1);
}

inline void cat(const char* a, const SubString& x, const GString& y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const char* a, const SubString& x, const SubString& y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x.chars(), x.length(), y.chars(), y.length());
}

inline void cat(const char* a, const SubString& x, const char* y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x.chars(), x.length(), y, -1);
}

inline void cat(const char* a, const SubString& x, char y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x.chars(), x.length(), &y, 1);
}

inline void cat(const char* a, const char* x, const GString& y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x, -1, y.chars(), y.length());
}

inline void cat(const char* a, const char* x, const SubString& y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x, -1, y.chars(), y.length());
}

inline void cat(const char* a, const char* x, const char* y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x, -1, y, -1);
}

inline void cat(const char* a, const char* x, char y, GString& r)
{
  r.rep = Scat(r.rep, a, -1, x, -1, &y, 1);
}


// operator versions

inline GString& GString::operator +=(const GString& y)
{
  cat(*this, y, *this);
  return *this;
}

inline GString& GString::operator +=(const SubString& y)
{
  cat(*this, y, *this);
  return *this;
}

inline GString& GString::operator += (const char* y)
{
  cat(*this, y, *this);
  return *this;
}

inline GString& GString:: operator +=(char y)
{
  cat(*this, y, *this);
  return *this;
}

// constructive concatenation

#if defined(__GNUG__) && !defined(_G_NO_NRV)

inline GString operator + (const GString& x, const GString& y) return r;
{
  cat(x, y, r);
}

inline GString operator + (const GString& x, const SubString& y) return r;
{
  cat(x, y, r);
}

inline GString operator + (const GString& x, const char* y) return r;
{
  cat(x, y, r);
}

inline GString operator + (const GString& x, char y) return r;
{
  cat(x, y, r);
}

inline GString operator + (const SubString& x, const GString& y) return r;
{
  cat(x, y, r);
}

inline GString operator + (const SubString& x, const SubString& y) return r;
{
  cat(x, y, r);
}

inline GString operator + (const SubString& x, const char* y) return r;
{
  cat(x, y, r);
}

inline GString operator + (const SubString& x, char y) return r;
{
  cat(x, y, r);
}

inline GString operator + (const char* x, const GString& y) return r;
{
  cat(x, y, r);
}

inline GString operator + (const char* x, const SubString& y) return r;
{
  cat(x, y, r);
}

inline GString reverse(const GString& x) return r;
{
  r.rep = Sreverse(x.rep, r.rep);
}

inline GString upcase(const GString& x) return r;
{
  r.rep = Supcase(x.rep, r.rep);
}

inline GString downcase(const GString& x) return r;
{
  r.rep = Sdowncase(x.rep, r.rep);
}

inline GString capitalize(const GString& x) return r;
{
  r.rep = Scapitalize(x.rep, r.rep);
}

#else /* NO_NRV */

inline GString operator + (const GString& x, const GString& y)
{
  GString r;  cat(x, y, r);  return r;
}

inline GString operator + (const GString& x, const SubString& y) 
{
  GString r; cat(x, y, r); return r;
}

inline GString operator + (const GString& x, const char* y) 
{
  GString r; cat(x, y, r); return r;
}

inline GString operator + (const GString& x, char y) 
{
  GString r; cat(x, y, r); return r;
}

inline GString operator + (const SubString& x, const GString& y) 
{
  GString r; cat(x, y, r); return r;
}

inline GString operator + (const SubString& x, const SubString& y) 
{
  GString r; cat(x, y, r); return r;
}

inline GString operator + (const SubString& x, const char* y) 
{
  GString r; cat(x, y, r); return r;
}

inline GString operator + (const SubString& x, char y) 
{
  GString r; cat(x, y, r); return r;
}

inline GString operator + (const char* x, const GString& y) 
{
  GString r; cat(x, y, r); return r;
}

inline GString operator + (const char* x, const SubString& y) 
{
  GString r; cat(x, y, r); return r;
}

inline GString reverse(const GString& x) 
{
  GString r; r.rep = Sreverse(x.rep, r.rep); return r;
}

inline GString upcase(const GString& x) 
{
  GString r; r.rep = Supcase(x.rep, r.rep); return r;
}

inline GString downcase(const GString& x) 
{
  GString r; r.rep = Sdowncase(x.rep, r.rep); return r;
}

inline GString capitalize(const GString& x) 
{
  GString r; r.rep = Scapitalize(x.rep, r.rep); return r;
}

#endif

// prepend

inline void GString::prepend(const GString& y)
{
  rep = Sprepend(rep, y.chars(), y.length());
}

inline void GString::prepend(const char* y)
{
  rep = Sprepend(rep, y, -1); 
}

inline void GString::prepend(char y)
{
  rep = Sprepend(rep, &y, 1); 
}

inline void GString::prepend(const SubString& y)
{
  rep = Sprepend(rep, y.chars(), y.length());
}

// misc transformations


inline void GString::reverse()
{
  rep = Sreverse(rep, rep);
}


inline void GString::upcase()
{
  rep = Supcase(rep, rep);
}


inline void GString::downcase()
{
  rep = Sdowncase(rep, rep);
}


inline void GString::capitalize()
{
  rep = Scapitalize(rep, rep);
}

// element extraction

inline char&  GString::operator [] (int i) 
{ 
  if (((unsigned)i) >= length()) error("invalid index");
  return rep->s[i];
}

inline const char&  GString::operator [] (int i) const
{ 
  if (((unsigned)i) >= length()) error("invalid index");
  return rep->s[i];
}

inline char  GString::elem (int i) const
{ 
  if (((unsigned)i) >= length()) error("invalid index");
  return rep->s[i];
}

inline char  GString::firstchar() const
{ 
  return elem(0);
}

inline char  GString::lastchar() const
{ 
  return elem(length() - 1);
}

// searching

inline int GString::index(char c, int startpos) const
{
  return search(startpos, length(), c);
}

inline int GString::index(const char* t, int startpos) const
{   
  return search(startpos, length(), t);
}

inline int GString::index(const GString& y, int startpos) const
{   
  return search(startpos, length(), y.chars(), y.length());
}

inline int GString::index(const SubString& y, int startpos) const
{   
  return search(startpos, length(), y.chars(), y.length());
}

inline int GString::index(const Regex& r, int startpos) const
{
  int unused;  return r.search(chars(), length(), unused, startpos);
}

inline int GString::contains(char c) const
{
  return search(0, length(), c) >= 0;
}

inline int GString::contains(const char* t) const
{   
  return search(0, length(), t) >= 0;
}

inline int GString::contains(const GString& y) const
{   
  return search(0, length(), y.chars(), y.length()) >= 0;
}

inline int GString::contains(const SubString& y) const
{   
  return search(0, length(), y.chars(), y.length()) >= 0;
}

inline int GString::contains(char c, int p) const
{
  return match(p, length(), 0, &c, 1) >= 0;
}

inline int GString::contains(const char* t, int p) const
{
  return match(p, length(), 0, t) >= 0;
}

inline int GString::contains(const GString& y, int p) const
{
  return match(p, length(), 0, y.chars(), y.length()) >= 0;
}

inline int GString::contains(const SubString& y, int p) const
{
  return match(p, length(), 0, y.chars(), y.length()) >= 0;
}

inline int GString::contains(const Regex& r) const
{
  int unused;  return r.search(chars(), length(), unused, 0) >= 0;
}

inline int GString::contains(const Regex& r, int p) const
{
  return r.match(chars(), length(), p) >= 0;
}


inline int GString::matches(const SubString& y, int p) const
{
  return match(p, length(), 1, y.chars(), y.length()) >= 0;
}

inline int GString::matches(const GString& y, int p) const
{
  return match(p, length(), 1, y.chars(), y.length()) >= 0;
}

inline int GString::matches(const char* t, int p) const
{
  return match(p, length(), 1, t) >= 0;
}

inline int GString::matches(char c, int p) const
{
  return match(p, length(), 1, &c, 1) >= 0;
}

inline int GString::matches(const Regex& r, int p) const
{
  int l = (p < 0)? -p : length() - p;
  return r.match(chars(), length(), p) == l;
}


inline int SubString::contains(const char* t) const
{   
  return S.search(pos, pos+len, t) >= 0;
}

inline int SubString::contains(const GString& y) const
{   
  return S.search(pos, pos+len, y.chars(), y.length()) >= 0;
}

inline int SubString::contains(const SubString&  y) const
{   
  return S.search(pos, pos+len, y.chars(), y.length()) >= 0;
}

inline int SubString::contains(char c) const
{
  return S.search(pos, pos+len, c) >= 0;
}

inline int SubString::contains(const Regex& r) const
{
  int unused;  return r.search(chars(), len, unused, 0) >= 0;
}

inline int SubString::matches(const Regex& r) const
{
  return r.match(chars(), len, 0) == len;
}


inline int GString::gsub(const GString& pat, const GString& r)
{
  return _gsub(pat.chars(), pat.length(), r.chars(), r.length());
}

inline int GString::gsub(const SubString&  pat, const GString& r)
{
  return _gsub(pat.chars(), pat.length(), r.chars(), r.length());
}

inline int GString::gsub(const Regex& pat, const GString& r)
{
  return _gsub(pat, r.chars(), r.length());
}

inline int GString::gsub(const char* pat, const GString& r)
{
  return _gsub(pat, -1, r.chars(), r.length());
}

inline int GString::gsub(const char* pat, const char* r)
{
  return _gsub(pat, -1, r, -1);
}



inline  ostream& operator<<(ostream& s, const GString& x)
{
   s << x.chars(); return s;
}

// a zillion comparison operators

inline int operator==(const GString& x, const GString& y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const GString& x, const GString& y)
{
  return compare(x, y) != 0; 
}

inline int operator>(const GString& x, const GString& y)
{
  return compare(x, y) > 0; 
}

inline int operator>=(const GString& x, const GString& y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(const GString& x, const GString& y)
{
  return compare(x, y) < 0; 
}

inline int operator<=(const GString& x, const GString& y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(const GString& x, const SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const GString& x, const SubString&  y)
{
  return compare(x, y) != 0; 
}

inline int operator>(const GString& x, const SubString&  y)      
{
  return compare(x, y) > 0; 
}

inline int operator>=(const GString& x, const SubString&  y)
{
  return compare(x, y) >= 0; 
}

inline int operator<(const GString& x, const SubString&  y) 
{
  return compare(x, y) < 0; 
}

inline int operator<=(const GString& x, const SubString&  y)
{
  return compare(x, y) <= 0; 
}

inline int operator==(const GString& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(const GString& x, const char* t) 
{
  return compare(x, t) != 0; 
}

inline int operator>(const GString& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(const GString& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(const GString& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(const GString& x, const char* t) 
{
  return compare(x, t) <= 0; 
}

inline int operator==(const SubString& x, const GString& y) 
{
  return compare(y, x) == 0; 
}

inline int operator!=(const SubString& x, const GString& y)
{
  return compare(y, x) != 0;
}

inline int operator>(const SubString& x, const GString& y)      
{
  return compare(y, x) < 0;
}

inline int operator>=(const SubString& x, const GString& y)     
{
  return compare(y, x) <= 0;
}

inline int operator<(const SubString& x, const GString& y)      
{
  return compare(y, x) > 0;
}

inline int operator<=(const SubString& x, const GString& y)     
{
  return compare(y, x) >= 0;
}

inline int operator==(const SubString& x, const SubString&  y) 
{
  return compare(x, y) == 0; 
}

inline int operator!=(const SubString& x, const SubString&  y)
{
  return compare(x, y) != 0;
}

inline int operator>(const SubString& x, const SubString&  y)      
{
  return compare(x, y) > 0;
}

inline int operator>=(const SubString& x, const SubString&  y)
{
  return compare(x, y) >= 0;
}

inline int operator<(const SubString& x, const SubString&  y) 
{
  return compare(x, y) < 0;
}

inline int operator<=(const SubString& x, const SubString&  y)
{
  return compare(x, y) <= 0;
}

inline int operator==(const SubString& x, const char* t) 
{
  return compare(x, t) == 0; 
}

inline int operator!=(const SubString& x, const char* t) 
{
  return compare(x, t) != 0;
}

inline int operator>(const SubString& x, const char* t)  
{
  return compare(x, t) > 0; 
}

inline int operator>=(const SubString& x, const char* t) 
{
  return compare(x, t) >= 0; 
}

inline int operator<(const SubString& x, const char* t)  
{
  return compare(x, t) < 0; 
}

inline int operator<=(const SubString& x, const char* t) 
{
  return compare(x, t) <= 0; 
}


// a helper needed by at, before, etc.

inline SubString GString::_substr(int first, int l)
{
  if (first < 0 || (unsigned)(first + l) > length() )
    return SubString(_nilString, 0, 0) ;
  else 
    return SubString(*this, first, l);
}

#endif
