/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: AlphaShape.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 1995/05/31 06:47:35 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Class to create and read the alpha shape from Edelsbrunner's group
 *
 ***************************************************************************/


#include <iostream.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <unistd.h>
#include <fstream.h>
#include "AlphaShape.h"
#include "alf.h"



// check if a file is readable
static int check_stat(const char *filename)
{
   struct stat buf;
   return (stat(filename, &buf) == 0);  // returns 1 if readable
}


// used for passing the class information about
// ugly, but it was the best I could think of..
static AlphaShape *alphaShapeP;
static int alpha_num_triangles;
static int alpha_num_edges;
static int alpha_num_vertices;

static void f1_hook(int t)
{
   int rank = alphaShapeP -> rank;
   int v = 0, ef = EdFacet (t, v);
   int tetra = alf_tetra_index (ef); /* tetra == 0 iff ef^+ outside hull */

   if ( (tetra > 0) && alf_is_interior (ALF_TETRA, rank, tetra))
   { /* get right orientation */
      v = 1;
      ef = Sym (ef);
   }
   Assert (   (alf_tetra_index (ef) == 0)
	      or (not alf_is_interior (ALF_TETRA, rank,
				       alf_tetra_index (ef))));
   trist_triangle (ef,
		   alphaShapeP->triangles[alpha_num_triangles]+0,
		   alphaShapeP->triangles[alpha_num_triangles]+1,
		   alphaShapeP->triangles[alpha_num_triangles]+2
		   );
   alpha_num_triangles++;
   
/*
  upfor(a, 1,3)
    print("%d ", afl->triangles[afl->num_triangles][a]);
  print("\n");
*/
}

static void f2_hook(int t)
{
   trist_triangle (EdFacet (t, 0),
		   alphaShapeP->triangles[alpha_num_triangles]+0,
		   alphaShapeP->triangles[alpha_num_triangles]+1,
		   alphaShapeP->triangles[alpha_num_triangles]+2
		   );
   alpha_num_triangles++;
}

static void e_hook(int ix)
{
   int ef = alf_edge_ef(ix);
   alphaShapeP -> edges[alpha_num_edges][0] = Org(ef);
   alphaShapeP -> edges[alpha_num_edges][1] = Dest(ef);
   alpha_num_edges++;
}

static void v_hook(int v)
{
   alphaShapeP -> vertices[alpha_num_vertices] = v;
   alpha_num_vertices++;
}

// set everything to 0 or NULL
void AlphaShape::init(void)
{
   is_okay = 0;
   filename = NULL;
   dirname = NULL;
   rank = 0;
   scale = 1.0;
   num_triangles = num_edges = num_vertices = 0;
   triangles = NULL;
   edges = NULL;
   vertices = NULL;
}

// clean up everything except the file/dir names and is_okay
void AlphaShape::free_all(void)
{
   if (triangles) {
      delete [] triangles;
      triangles = NULL;
   }
   if (edges) {
      delete [] edges;
      edges = NULL;
   }
   if (vertices) {
      delete [] vertices;
      vertices = NULL;
   }
   rank = 0;
   num_triangles = num_edges = num_vertices = 0;
}

// constructor which reads from a file and sets the radius
AlphaShape::AlphaShape(float radius, const char *new_filename)
{
   init();
   filename = new char[strlen( new_filename) + 1];
   strcpy(filename, new_filename);
   get_faces(radius);
}

// constructor which makes a temporary file, calls the appropriate
// programs to do the conversion, then read the resultant files and
// read it in for the appropriate weight
AlphaShape::AlphaShape(float radius, int num, float *x, float *y, float *z,
		       float *r, float new_scale)
{
   init();
   
   // first, find a good name for the temp directory
   char randprefix[20];
   {  // make a four digit random number
      int i = rand();
      sprintf(randprefix, "%d", i);
      randprefix[4]=0;
   }
   if (getenv("ALPHASHAPE_TMPDIR") ) {
      dirname = new char[strlen(getenv("ALPHASHAPE_TMPDIR")) + 100];
      sprintf(dirname, "%s/alpha_shape%d.%sXXXXXX", getuid(),
	      randprefix);
   } else {
      dirname = new char[strlen("/usr/tmp") + 100];
      sprintf(dirname, "/usr/tmp/alpha_shape%d.%sXXXXXX", getuid(),
	      randprefix);
   }
   mktemp(dirname); // this should be unique
   if (dirname[0] == 0) {
      cerr << "Cannot make new directory name for alpha shape:"
	   << strerror(errno) << endl;
      delete [] dirname;
      dirname = NULL;
      return;
   }
   // make the temp directory
   {
      mode_t tmpmask = umask(0);
      umask(tmpmask);
      if (mkdir(dirname, 0544 | ~tmpmask) != 0) {
	 cerr << "Cannot create temp directory " << dirname
	      << " for alpha shape: " << strerror(errno) << endl;
	 delete [] dirname;
	 return;
      };
   }
   // create the input file for that directory
   filename = new char[strlen(dirname) + strlen("/alpha_data") + 5];
   strcpy(filename, dirname);
   strcat(filename, "/alpha_data");

   // write everything to the data file in that dir
   {
      scale = new_scale;
      ofstream outfile(filename);
//      outfile << "#scale: " << scale << endl;
      for (int i=0; i<num; i++) {
	 outfile << int(x[i]*scale) << " " << int(y[i]*scale)
		 << " " << int(z[i]*scale) 
		 << " " << int(r[i]*scale) << endl;
      }
   }
   // call the system programs for conversion
   {
      char *s = new char[strlen(filename) + 80];
      strcpy(s, "delcx ");
      strcat(s, filename);
      system(s);           // `delcx filename`

      strcpy(s, "mkalf ");
      strcat(s, filename);
      system(s);           // `mkalf filename`
   }
   get_faces(radius);
}

// destructor
AlphaShape::~AlphaShape(void)
{
   free_all();
   if (filename) delete [] filename;
   if (dirname) { // then I created a temp directory
      char *s = new char[strlen(dirname) + strlen("/bin/rm") + 10];
      strcpy(s, "/bin/rm -rf ");
      strcat(s, dirname);
//      cout << s << endl;
      system(s);             // so get rid of it
      delete [] s;
      delete [] dirname;
   }
}

// read the data given the prefix for the file names
// Three files are involved, the coordinate file (x,y,z,r),
// the Delaunay triangulation file ( *.dt), and the
// alpha shape file ( *.alf).
void AlphaShape::get_faces(float radius)
{
   is_okay = 0;
   
   // the alpha shape code seems to core dump if the files don't
   // exist, so I'll check them by hand
   if (!check_stat(filename)) {
      cerr << "Cannot read file: " << filename << endl;
      return;
   }
   char *dt_path = STRDUP( dt_PATH(filename));
   if (!check_stat(dt_path)) {
      cerr << "Cannot read file: " << dt_path << endl;
      free(dt_path);
      return;
   }
      
   char *alf_path = STRDUP( alf_PATH(filename));
   if (!check_stat(alf_path)) {
      cerr << "Cannot read file: " << alf_path << endl;
      free(dt_path);
      free(alf_path);
      return;
   }
   Alf_adt alp = alf_load_all(filename, dt_path, alf_path);
   free(dt_path);
   free(alf_path);
   if (!alp) {
      is_okay = 0;
      fprintf(stderr, "No open!\n");
      return;
   }
   is_okay = 1;

   // The structure is loaded into global memory, now I need
   // to get the data
   
   Alf_info *alfi = alf_info();     // where does it get the info?
   triangles = new Tvect[alfi->dt_num.f]; // place to store it
   edges     = new Evect[alfi->dt_num.e];
   vertices  = new int[alfi->dt_num.v];

   // apply the scaling factor
   radius *= scale;
   rank = alf_rank2(radius);
   
//   printf("Rank = %d\n", rank);
   
   // read in the triangles, edges, and vertices using the given
   // function pointers
   alphaShapeP= this;
   alpha_num_triangles = alpha_num_edges = alpha_num_vertices = 0;
   alf_scan_alpha_f1(rank, f1_hook);
   alf_scan_alpha_f2(rank, f2_hook);
   alf_scan_alpha_e2(rank, e_hook);
   alf_scan_alpha_v2(rank, v_hook);
   num_triangles = alpha_num_triangles;
   num_edges = alpha_num_edges;
   num_vertices = alpha_num_vertices;

   alf_kill(alp);
}





