/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Animation.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.6 $	$Date: 1995/05/11 21:16:05 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The Animation class, which stores a list of pointers to Timestep objects
 * that contain 3D coordinates and other data that varies with time.
 *
 ***************************************************************************/

#include "Animation.h"
#include "Timestep.h"
#include "Inform.h"
#include "utilities.h"


// strings describing animation styles
char *animationStyleName[Animation::TOTAL_STYLES] = {
	"Once", "Loop", "Rock" };


// constructor
Animation::Animation(void) : tsList(INITIAL_FRAME_COUNT, 4) {

  MSGDEBUG(1,"Creating new Animation list ..." << sendmsg);
    
  lastTime = time_of_day();	// get the current time for determining speed
  currFrame = (-1);		// no frames yet
  appendFrame = (-1);		// appending at end; this does not matter
  moveTo = (-1);		// we do not want to jump to a frame
  speed(1.0);			// set initial speed to maximum
  skip(1);			// don't skip any frames yet
  append_end();			// new frames added to end of list
  anim_dir(PAUSE);		// not yet animating
  anim_style(LOOP);		// loop back to beginning when end reached
  currTime = 0.0;		// at beginning of animation
  frameChanged = needSetFrameChanged = TRUE; // data has changed; now exists
}


// destructor; remove all frames
Animation::~Animation(void) {
  MSGDEBUG(1,"Deleting Animation list with " << num() << " frames ...");
  MSGDEBUG(1,sendmsg);
  delete_animation();
}
 

// delete all the frames in the animation
void Animation::delete_animation(void) {
  
  MSGDEBUG(2,"Animation: Deleting all frames" << sendmsg);

  for(int i=(num() - 1); i >= 0; i--)
    delete_frame(i);
}


// delete the Nth frame, with numbering starting at 0
void Animation::delete_frame(int n) {
  if(n >= 0 && n < num()) {
  
    MSGDEBUG(2,"Animation: Deleting frame " << n << sendmsg);

    delete tsList[n];		// delete object stored at Nth position
    tsList.remove(n);		// delete Nth position

    if(num() == 0) {
      change_current_frame(-1);
      moveTo = (-1);
    } else {
      if(moveTo > n)
        moveTo--;
      if(currFrame > n)
        change_current_frame(currFrame - 1);
    }
  }
}


// append a new frame; return (-1) if error, new frame number if OK
// Uses the current append method to determine position of new frame
int Animation::append_frame(Timestep *ts) {
  int newfr;

  // initialize the frame
  ts->init();

  MSGDEBUG(2,"Animation: Appending new frame ");

  // now append at the proper position
  if(appendPos == START) {
    newfr = tsList.insert_before(-1, ts);  // put new item at beginning of list
    MSGDEBUG(2,"at start");

  } else if(appendPos == END) {
    newfr = tsList.append(ts);		   // just append at end
    MSGDEBUG(2,"at end");

  } else if(appendPos == AFTER) {
    newfr = tsList.insert_after(appendFrame, ts);
    MSGDEBUG(2,"after frame" << appendFrame);
  
  } else if(appendPos == BEFORE) {
    newfr = tsList.insert_before(appendFrame, ts);
    MSGDEBUG(2,"before frame" << appendFrame);

  } else {
    newfr = tsList.append(ts);		   // just append at end
    MSGDEBUG(2,"at end (due to unknown append pos)");
  }

  if(currFrame < 0)
    // must set current frame because this frame is the first one
    change_current_frame(0);

  else if(currFrame >= newfr)
    // inserted frame before current one
    change_current_frame(currFrame + 1);

  MSGDEBUG(2,"; Now " << num() << " frames." << sendmsg);

  // move to new position ONLY if currently paused
  if(animDir == PAUSE)
    change_current_frame(newfr);

  return newfr;
}


// update the animation list based on current mode; return if curr frame change
int Animation::anim_update(void) {
  double currTime, dt;
  int i;

  // if a jump has been requested, move to that frame and just return
  if(moveTo >= 0 && moveTo < num()) {
    if(moveTo != currFrame)
      change_current_frame(moveTo);
    moveTo = (-1);
    anim_dir(PAUSE);		// pause after jumping
    lastTime = time_of_day();	// reset timing

  } else if (animDir == PAUSE || num() <= 0) {
    // if just paused, do nothing
    animDir = PAUSE;

  } else if(num() > 0) {

    // check the clock to see if the current frame should be updated
    if(Speed  < 1.0) {
      currTime = time_of_day();
      dt = currTime - lastTime;
      if(dt <= (SPEED_FACTOR - Speed)) {
        frameChanged = needSetFrameChanged;
        needSetFrameChanged = FALSE;
        return frameChanged;
      } else {
        lastTime = currTime;
      }
    }

    // skip the current frame ahead the proper amount
    for(i=0; i < frameSkip; i++) {
      if (animDir == REVERSE || animDir == REVERSE1) {
        if (currFrame <= 0) {
          if(animStyle == LOOP || animDir == REVERSE1) {
	    change_current_frame(num() - 1);
          } else if(animStyle == ROCK) {
            animDir = FORWARD;
          } else if(animStyle == ONCE) {
            animDir = PAUSE;
          }
        } else {
          change_current_frame(currFrame - 1);
        }

      } else if (animDir == FORWARD || animDir == FORWARD1) {
        if (currFrame >= (num() - 1)) {
          if(animStyle == LOOP || animDir == FORWARD1) {
	    change_current_frame(0);
          } else if(animStyle == ROCK) {
            animDir = REVERSE;
          } else if(animStyle == ONCE) {
            animDir = PAUSE;
          }
        } else {
          change_current_frame(currFrame + 1);
        }
      }
    }

    if(animDir == FORWARD1 || animDir == REVERSE1)
      animDir = PAUSE;
  }
  
  // indicate whether the current frame has changed
  frameChanged = needSetFrameChanged;
  needSetFrameChanged = FALSE;
  return frameChanged;
}

