/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Atom.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.8 $	$Date: 1995/06/29 23:48:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Structure which holds data for one atom.
 *
 ***************************************************************************/
#ifndef ATOM_H
#define ATOM_H

#include <string.h>
#include "Inform.h"
#include "utilities.h"

// maximum length of name strings
#ifndef NAMESTRLEN
#define NAMESTRLEN	6
#endif

// coordinates to store for each atom
#define ATOMCOORDS	3

// maximum number of bonds allowed to other atoms
#define MAXATOMBONDS	8

// extra data values to store for each atom, and their indices
#define ATOMEXTRA	5
#define ATOMCHARGE	0
#define ATOMMASS	1
#define ATOMBETA	2
#define ATOMOCCUP	3
#define ATOMRAD		4

class Atom {

public:

  enum BackboneType { NORMAL, PROTEINBACK, NUCLEICBACK };
  
  // used to tell me if this atom is part of some larger, and complete,
  // structure
  enum ResidueType { NOTHING, PROTEIN, NUCLEIC, WATERS};

  int index;			// the index of this atom
  int bonds;			// how many bonds this atom has
  int uniq_resid;		// a unique resid, since there can be more
  				//   than one residue with the same resid
  int fragment;			// a fragment is a set of connect residues
  				//   which are not connected to any other
  				//   residues
  int bondTo[MAXATOMBONDS];	// atoms to which this atom is bonded
  BackboneType bondType[MAXATOMBONDS];	// what kind of bond it is
  BackboneType atomType;		// is this atom part of the backbone?
  ResidueType residueType;  // is this part of a larger component?
     // for instance, is this CG atom in an amino acid of some sort?

  float pos[ATOMCOORDS];	// 'default' position of this atom
  float extra[ATOMEXTRA];	// 'default' extra data for this atom

  char namestr[NAMESTRLEN+1];
  int   nameindex;		// string and index for atom name

  char typestr[NAMESTRLEN+1];
  int   typeindex;		// string and index for atom type

  char resnamestr[NAMESTRLEN+1];
  int   resnameindex;		// string and index for atom residue name

  char residstr[NAMESTRLEN+1];
  int   residindex;		// string and index for atom residue id

  char chainstr[1];
  int chainindex;                // chain identifier

  char segnamestr[NAMESTRLEN+1];
  int   segnameindex;		// string and index for atom segment name

public:
  // constructor
  Atom(int n, float *defpos, float *defextra,
  	char *name, char *type, char *res, char *resid, 
        char *chain, char *segname) {
    int i;
    for(i=0; i < ATOMCOORDS; i++)	pos[i] = defpos[i];
    for(i=0; i < ATOMEXTRA; i++)	extra[i] = defextra[i];
    index = n;
    uniq_resid = 0; // don't know yes, found in BaseMolecule
    bonds = 0;
    strncpy(namestr, name, NAMESTRLEN);		namestr[NAMESTRLEN] = '\0';
    strncpy(typestr, type, NAMESTRLEN);		typestr[NAMESTRLEN] = '\0';
    strncpy(resnamestr, res, NAMESTRLEN);	resnamestr[NAMESTRLEN] = '\0';
    strncpy(residstr, resid, NAMESTRLEN);	residstr[NAMESTRLEN] = '\0';
    strncpy(chainstr, chain, NAMESTRLEN);	chainstr[NAMESTRLEN] = '\0';
    strncpy(segnamestr, segname, NAMESTRLEN);	segnamestr[NAMESTRLEN] = '\0';
    nameindex = typeindex = resnameindex = residindex = segnameindex = (-1);
    for (i=0; i<MAXATOMBONDS; i++) {
      bondTo[i] = -1;
      bondType[i] = NORMAL;
    }
    atomType = NORMAL;
    residueType = NOTHING;
  }
  
  //
  // atom characteristics
  //
  
  float radius(void) { return extra[ATOMRAD]; }
  float mass(void) { return extra[ATOMMASS]; }
  float charge(void) { return extra[ATOMCHARGE]; }
  float beta(void) { return extra[ATOMBETA]; }
  float occup(void) { return extra[ATOMOCCUP]; }

  // add a bond into the atom.  Note that each bond will be stored twice, so
  // make sure when using bonds to define whether the 'official' bond is from
  // low # atom -> high #, or vice versa
  void add_bond(int a, BackboneType type) {
    if(bonds >= MAXATOMBONDS) {
      msgErr << "Atom " << index << ": Exceeded maximum number of bonds (";
      msgErr << bonds << ")." << sendmsg;
    } else {
      bondTo[bonds] = a;
      bondType[bonds] = type;
      if(type == PROTEINBACK || type == NUCLEICBACK)
        atomType = type;
      bonds++;
    }
  }

  // return TRUE if this atom is bonded to the specified atom.  Returns FALSE
  // otherwise.
  int bonded(int a) {
    int i, retval = FALSE;
    for(i=0; i < bonds; i++)
      if(bondTo[i] == a) {
        retval = TRUE;
	break;
      }
    return retval;
  }

};

#endif

