/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: AtomColor.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.17 $      $Date: 1995/06/29 23:48:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Parse and maintain the data for how a molecule should be colored.
 *
 ***************************************************************************/

#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "AtomColor.h"
#include "DrawMolecule.h"
#include "MoleculeList.h"
#include "ColorList.h"
#include "Inform.h"
#include "utilities.h"
#include "config.h"

// a string array with text descriptions of coloring methods
char *AtomColorName[AtomColor::TOTAL] = {"Name", "Type", "ResName", "ResType",
	"ResID", "Chain", "SegName", "Molecule", "ColorID", "Beta", "Occupancy",
	"Mass", "Charge", "Pos", "Index", "Backbone" };


//////////////////////////  constructor and destructor
// constructor; parse string and see if OK
AtomColor::AtomColor(MoleculeList *mlist, ColorList *clist) {

  MSGDEBUG(2,"Creating new AtomColor object ..." << sendmsg);

  // initialize variables
  molList = mlist;
  colList = clist;
  color = NULL;
  transpar = FALSE;
  colIndex = VISCLRS;
  nAtoms = 0;
  mol = NULL;
  colorMethod  = DEFAULT_ATOMCOLOR;
  strcpy(cmdStr, AtomColorName[DEFAULT_ATOMCOLOR]);
}


// copy constructor
AtomColor::AtomColor(AtomColor& ar) {
  strcpy(cmdStr, ar.cmdStr);
  colorMethod = ar.colorMethod;
  colIndex = ar.colIndex;
  colList = ar.colList;
  molList = ar.molList;
  mol = ar.mol;
  transpar = ar.transpar;
  if(ar.color) {
    nAtoms = ar.nAtoms;
    color = new int[nAtoms];
    for(int i=0; i < nAtoms; i++)
      color[i] = ar.color[i];
  } else {
    nAtoms = 0;
    color = NULL;
  }
}


// destructor; free up space
AtomColor::~AtomColor(void) {
  if(color)
    delete [] color;
}


//////////////////////////  private routines

// parse the given command, and store results.  Return success.
int AtomColor::parse_cmd(char *newcmd) {
  int argc, ok = TRUE;
  char *argv[128], *cmdStrTok = NULL;

  ColorMethod newMethod = colorMethod;
  int newIndex = colIndex, newTrans = FALSE;
  
  // make sure the new command is not too long
  if(newcmd && strlen(newcmd) > MAX_ATOMCOLOR_CMD) {
    msgErr << "Atom coloring method string is too long (over ";
    msgErr << MAX_ATOMCOLOR_CMD << " characters)." << sendmsg;
    return FALSE;
  }

  // tokenize the command
  if(!newcmd || !(cmdStrTok = str_tokenize(newcmd, &argc, argv))) {
    // no command; keep current settings
    return TRUE;
  }

  // now parse the command
  if(!strupncmp(argv[0], "default", CMDLEN)) {
    newMethod = DEFAULT_ATOMCOLOR;
  } else if(!strupncmp(argv[0], AtomColorName[NAME], CMDLEN)) {
    newMethod = NAME;
  } else if(!strupncmp(argv[0], AtomColorName[TYPE], CMDLEN)) {
    newMethod = TYPE;
  } else if(!strupncmp(argv[0], AtomColorName[RESNAME], CMDLEN)) {
    newMethod = RESNAME;
  } else if(!strupncmp(argv[0], AtomColorName[RESTYPE], CMDLEN)) {
    newMethod = RESTYPE;
  } else if(!strupncmp(argv[0], AtomColorName[RESID], CMDLEN)) {
    newMethod = RESID;
  } else if(!strupncmp(argv[0], AtomColorName[CHAIN], CMDLEN)) {
    newMethod = CHAIN;
  } else if(!strupncmp(argv[0], AtomColorName[SEGNAME], CMDLEN)) {
    newMethod = SEGNAME;
  } else if(!strupncmp(argv[0], AtomColorName[MOLECULE], CMDLEN)) {
    newMethod = MOLECULE;
  } else if(!strupncmp(argv[0], AtomColorName[COLORID], CMDLEN) && argc > 1) {
    newMethod = COLORID;
    newIndex = atoi(argv[1]);
  } else if(!strupncmp(argv[0], AtomColorName[BETA], CMDLEN)) {
    newMethod = BETA;
  } else if(!strupncmp(argv[0], AtomColorName[OCCUP], CMDLEN)) {
    newMethod = OCCUP;
  } else if(!strupncmp(argv[0], AtomColorName[MASS], CMDLEN)) {
    newMethod = MASS;
  } else if(!strupncmp(argv[0], AtomColorName[CHARGE], CMDLEN)) {
    newMethod = CHARGE;
  } else if(!strupncmp(argv[0], AtomColorName[POS], CMDLEN)) {
    newMethod = POS;
//  } else if(!strupncmp(argv[0], AtomColorName[VEL], CMDLEN)) {
//    newMethod = VEL;
  } else if(!strupncmp(argv[0], AtomColorName[INDEX], CMDLEN)) {
    newMethod = INDEX;
  } else if(!strupncmp(argv[0], AtomColorName[BACKBONE], CMDLEN)) {
    newMethod = BACKBONE;
  } else {
    // unknown representation
    ok = FALSE;
  }

  // check the command was not too long
  ok = ok && ((newMethod == COLORID && argc < 4) || 
  	    (newMethod != COLORID && argc < 3));

  // check if last word is 'transparent'
  if(ok && ((newMethod == COLORID && argc == 3) || 
  	    (newMethod != COLORID && argc == 2))) {
    if(!strupncmp(argv[argc - 1], "transparent", CMDLEN))
      newTrans = TRUE;
    else
      ok = FALSE;
  }

  // print error message if necessary
  if(!ok) {
    msgErr << "Incorrect atom color method command '" << newcmd << "'";
    msgErr << sendmsg;
  } else {
    // command was successful, save new settings
    colorMethod = newMethod;
    colIndex = newIndex;
    transpar = newTrans;
    strcpy(cmdStr, newcmd);
    
    if(colIndex < 0)
      colIndex = 0;
    else if(colIndex >= VISCLRS)
      colIndex = VISCLRS - 1;
  }

  // delete parsing space
  delete [] cmdStrTok;
  
  return ok;
}


//////////////////////////  public routines

// equal operator, to change the current settings.  Does NOT change the
// current molecule, molecule list, or color list.
AtomColor& AtomColor::operator=(const AtomColor &ar) {

  // copy values
  if(cmdStr != ar.cmdStr)
    strcpy(cmdStr, ar.cmdStr);
  colorMethod = ar.colorMethod;
  colIndex = ar.colIndex;
  transpar = ar.transpar;
  
  // update current colors based on new settings
  find(mol);  
    
  return *this;
}


// return whether the given colorlist and color category index are the
// settings for the current coloring method
int AtomColor::current_color_use(ColorList *collist, int ccat) {
  if(!molList || !colList)
    return FALSE;

  if(collist == colList) {
    if((colorMethod==MOLECULE&& ccat==molList->colorCatIndex[MLCAT_MOLECULES])
      || (colorMethod==NAME&& ccat==molList->colorCatIndex[MLCAT_NAMES])
      || (colorMethod==RESNAME&& ccat==molList->colorCatIndex[MLCAT_RESNAMES])
      || (colorMethod==RESTYPE&& ccat==molList->colorCatIndex[MLCAT_RESTYPES])
      || (colorMethod==CHAIN&& ccat==molList->colorCatIndex[MLCAT_CHAINS])
      || (colorMethod==SEGNAME&& ccat==molList->colorCatIndex[MLCAT_SEGNAMES])
      || (colorMethod==BACKBONE&& ccat==molList->colorCatIndex[MLCAT_SPECIAL])
      ) {
        return TRUE;
    }
  }
  
  return FALSE;
}


// macro used to set transparent/regular color in 'find' routine
#define CHECK_TRANSP(t,x)	(t ? TRANSLUCENT(x) : REGCOLOR(x))
#define CHECK_MAPTRANSP(t,x)	(t ? MAPTRANSLUCENT(x) : MAPCOLOR(x))

// find the color index for the atoms of the given molecule.  Return success.
int AtomColor::find(DrawMolecule *m) {
  int i, dindex, tsindex;
  float minval, maxval, val;

  // make sure things are OK
  if(!m || !molList || !colList)
    return FALSE;
    
  // save new molecule, and remove old storage if necessary  
  if(color && mol && nAtoms != m->nAtoms) {
    delete [] color;
    color = NULL;
  }
    
  // allocate new storage
  mol = m;
  nAtoms = m->nAtoms;
  if(!color)
    color = new int[nAtoms];

  // check for special cases
  if(colorMethod == MOLECULE) {
    dindex =(colList->color_category(molList->colorCatIndex[MLCAT_MOLECULES]))
    		-> data(mol->name);
    dindex = CHECK_TRANSP(transpar, dindex);
    for(i=0; i < nAtoms; color[i++] = dindex);

  } else if(colorMethod == COLORID) {
    dindex = CHECK_TRANSP(transpar, colIndex);
    for(i=0; i < nAtoms; i++) {
      color[i] = dindex;
    }

  } else if(colorMethod == BACKBONE) {
    int regc = CHECK_TRANSP(transpar,
    	(colList->color_category(molList->colorCatIndex[MLCAT_SPECIAL]))
    		-> data("Nonback"));
    int proc = CHECK_TRANSP(transpar,
    	(colList->color_category(molList->colorCatIndex[MLCAT_SPECIAL]))
    		-> data("Proback"));
    int dnac = CHECK_TRANSP(transpar,
    	(colList->color_category(molList->colorCatIndex[MLCAT_SPECIAL]))
    		-> data("Nucback"));
    for(i=0; i < nAtoms; i++) {
      Atom *a = mol->atom(i);
      int c = regc;
      if(a->atomType != Atom::NORMAL) {  // is this a backbone atom?
        for(int j=0; j < a->bonds; j++) {
	  if(a->bondType[j] == Atom::PROTEINBACK) {
	    c = proc;
	    break;
	  } else if(a->bondType[j] == Atom::NUCLEICBACK) {
	    c = dnac;
	    break;
	  }
	}
      }
      color[i] = c;
    }

  } else if(colorMethod == INDEX) {
    // scan molecule to find min and max resid ... may not be necessary in
    // the future if this is precomputed by the molecule
    if(nAtoms > 0) {
      int minres, maxres;
      maxres = minres = (mol->atom(0))->residindex;
      for(i=0; i < nAtoms; i++) {
        dindex =  (mol->atom(i))->residindex;
        if(dindex < minres)
          minres = dindex;
        else if(dindex > maxres);
          maxres = dindex;
      }
    
      // now color atoms using the colormap, scaling from minres to maxres
      float scalefac = (float)(MAPCLRS-1) / (float)(maxres - minres + 1);
      for(i=0; i < nAtoms; i++) {
        dindex = (int)( 0.5 + scalefac * 
      	  (float)((mol->atom(i))->residindex - minres));
        color[i] = CHECK_MAPTRANSP(transpar, dindex);
      }
    
      MSGDEBUG(2, "AtomColor: Coloring by INDEX, from " << minres << " to ");
      MSGDEBUG(2, maxres << sendmsg);
      MSGDEBUG(2, "           Scaling by " << scalefac << ", last index = ");
      MSGDEBUG(2, dindex << sendmsg);
    }
    
  } else if(colorMethod == RESID) {
    for(i=0; i < nAtoms; i++) {
      dindex = (mol->atom(i))->residindex % VISCLRS;
      color[i] = CHECK_TRANSP(transpar, dindex);
    }
    
  } else if(colorMethod >= BETA && colorMethod <= POS) {
    // find min/max values, and where to get data, for current timestep.
    // If no timestep, just color by molecule
    
    // first find current timestep
    if(!mol->is_current()) {
      // no timestep is current; set coloring by molecule and exit
      ColorMethod oldMethod = colorMethod;
      colorMethod = MOLECULE;
      find(mol);
      colorMethod = oldMethod;
      return TRUE;
    }
    Timestep *ts = mol->current();
    
    // now find min/max values, and where to get data
    if(colorMethod == BETA)
      tsindex = ATOMBETA;
    else if(colorMethod == OCCUP)
      tsindex = ATOMOCCUP;
    else if(colorMethod == MASS)
      tsindex = ATOMMASS;
    else if(colorMethod == CHARGE)
      tsindex = ATOMCHARGE;
    if(colorMethod == POS) { //|| colorMethod == VEL) {
      maxval = ts->maxrad;
      minval = 0.0;
    } else {
      minval = ts->mindata[tsindex];
      maxval = ts->maxdata[tsindex];
    }

    MSGDEBUG(2,"AtomColor: Finding colors by: '");
    MSGDEBUG(2,AtomColorName[colorMethod] << "' ..." << sendmsg);
    MSGDEBUG(2,"           min: " << minval << ", max: " << maxval << sendmsg);

    // now assign color based on scaling from min to max
    if(maxval == minval) {
      for(i=0; i < nAtoms; i++)
        color[i] = CHECK_MAPTRANSP(transpar, MAPCLRS/2);
    } else {
      float scalefac = (float)(MAPCLRS-1) / (maxval - minval);
      MSGDEBUG(2,"         scalefac = " << scalefac << sendmsg);
      for(i=0; i < nAtoms; i++) {
        if(colorMethod == POS ) { // || colorMethod == VEL) {
	  float *dpos = &(ts->pos[3*i]);
	  float x = *dpos - ts->COV[0];
	  float y = *(dpos + 1) - ts->COV[1];
	  float z = *(dpos + 2) - ts->COV[2];
	  val = sqrtf(x*x + y*y + z*z);
	} else {
	  val = ts->data[ATOMEXTRA * i + tsindex];
	}
	
        dindex = (int)(scalefac * (val - minval));
        color[i] = CHECK_MAPTRANSP(transpar, dindex);
	MSGDEBUG(2,"         Atom " << i << ": dindex= " << dindex << sendmsg);
      }
    }
    
  } else if(colorMethod == NAME) {
    for(i=0; i < nAtoms; i++) {
      dindex = (colList->color_category(molList->colorCatIndex[MLCAT_NAMES]))
		-> data((mol->atomNames).data((mol->atom(i))->nameindex));
      color[i] = CHECK_TRANSP(transpar, dindex);
    }
    
  } else if(colorMethod == TYPE) {
    for(i=0; i < nAtoms; i++) {
      dindex = (colList->color_category(molList->colorCatIndex[MLCAT_TYPES]))
		-> data((mol->atomTypes).data((mol->atom(i))->typeindex));
      color[i] = CHECK_TRANSP(transpar, dindex);
    }
    
  } else if(colorMethod == RESNAME) {
    for(i=0; i < nAtoms; i++) {
      dindex =(colList->color_category(molList->colorCatIndex[MLCAT_RESNAMES]))
		-> data((mol->resNames).data((mol->atom(i))->resnameindex));
      color[i] = CHECK_TRANSP(transpar, dindex);
    }
    
  } else if(colorMethod == RESTYPE) {
    for(i=0; i < nAtoms; i++) {
      dindex =(colList->color_category(molList->colorCatIndex[MLCAT_RESTYPES]))
		-> data((molList->resTypes).data((mol->atom(i))->resnamestr));
      color[i] = CHECK_TRANSP(transpar, dindex);
    }
    
  } else if(colorMethod == CHAIN) {
    for(i=0; i < nAtoms; i++) {
      dindex =(colList->color_category(molList->colorCatIndex[MLCAT_CHAINS]))
		-> data((mol->chainNames).data((mol->atom(i))->chainindex));
      color[i] = CHECK_TRANSP(transpar, dindex);
    }

  } else if(colorMethod == SEGNAME) {
    for(i=0; i < nAtoms; i++) {
      dindex =(colList->color_category(molList->colorCatIndex[MLCAT_SEGNAMES]))
		-> data((mol->segNames).data((mol->atom(i))->segnameindex));
      color[i] = CHECK_TRANSP(transpar, dindex);
    }

  } else {
    msgErr << "Unknown coloring method " << colorMethod << " in AtomColor.";
    msgErr << sendmsg;
    return FALSE;
  }
    
  return TRUE;
}

