/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: AtomRep.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.14 $      $Date: 1995/05/25 08:19:08 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Parse and maintain the data for how a molecule should be represented.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include "AtomRep.h"
#include "Inform.h"
#include "utilities.h"
#include "config.h"
#ifdef VMDALPHA_SHAPE
#include "AlphaShape.h"
#endif

// a string array with text descriptions of representation methods
char *AtomRepName[AtomRep::TOTAL] = { "Lines", "Bonds", "CPK", "Points",
     "VDW", "Dotted", "Licorice", "Ribbons", "Tube",
#ifdef VMDALPHA_SHAPE
     "Alpha",
#endif
				      "Off" };


//////////////////////////  constructor and destructor
// constructor; parse string and see if OK
AtomRep::AtomRep(void) {
  
  MSGDEBUG(2,"Creating new AtomRep object ..." << sendmsg);

  // initialize variables
  // remember to change the values in GraphicsFormsObj if you
  // change these defaults!!
  lineThickness = 3;
  sphereRad = 1.0;
  bondRad = 0.3;
  sphereRes = 6;
  bondRes = 8;
  repMethod  = DEFAULT_ATOMREP;
  option_changed = FALSE;
#ifdef VMDALPHA_SHAPE
  alpha = NULL;
  alpha_count = NULL;
#endif
  
  strcpy(cmdStr, AtomRepName[DEFAULT_ATOMREP]);
}


// copy constructor
AtomRep::AtomRep(AtomRep& ar) {
  strcpy(cmdStr, ar.cmdStr);
  repMethod = ar.repMethod;
  option_changed = FALSE;
#ifdef VMDALPHA_SHAPE
//  msgInfo << "Copy constructor" << sendmsg;
  alpha = ar.alpha;
  alpha_count = ar.alpha_count;
  if (alpha_count) {   // increase the reference count
//     msgInfo << " count is " << *alpha_count << sendmsg;
     (*alpha_count)++;
  } else {
//     msgInfo << " no counter " << sendmsg;
  }
     
#endif
  sphereRad = ar.sphereRad;
  bondRad = ar.bondRad;
  sphereRes = ar.sphereRes;
  bondRes = ar.bondRes;
  lineThickness = ar.lineThickness;
}

#ifdef VMDALPHA_SHAPE
void AtomRep::delete_alpha(void)
{
   if (alpha) {                 // alpha can be used by several selections
      if (*alpha_count <= 1) {  // so I check the reference count to see if
	 delete alpha;          // anyone else is using this.
	 delete alpha_count;
      } else {
	 (*alpha_count)--;      // there is another selection with this shape
	                        // so just reduce the count
      }
      alpha = NULL;
      alpha_count = NULL;
   }
}
void AtomRep::set_alpha(AlphaShape *newalpha)
{
   if (alpha) {
      delete_alpha();
   }
   alpha = newalpha;
   alpha_count = new int;
   *alpha_count = 1;
}
AtomRep::~AtomRep(void) {
   delete_alpha();
}
#else
AtomRep::~AtomRep(void) {
}
#endif

//////////////////////////  private routines

// parse the given command, and store results.  Return success.
int AtomRep::parse_cmd(char *newcmd) {
  int argc, ok = TRUE;
  char *argv[128], *cmdStrTok = NULL;

  RepMethod newMethod = repMethod;
  int newThick = lineThickness;
  int newBondRes = bondRes, newSphRes = sphereRes;
  float newSphRad = sphereRad, newBondRad = bondRad;
  
  // make sure the new command is not too long
  if(newcmd && strlen(newcmd) > MAX_ATOMREP_CMD) {
    msgErr << "Atom representation string is too long (over ";
    msgErr << MAX_ATOMREP_CMD << " characters)." << sendmsg;
    return FALSE;
  }

  // tokenize the command
  if(!newcmd || !(cmdStrTok = str_tokenize(newcmd, &argc, argv))) {
    // no command; keep current settings
    return TRUE;
  }

  // now parse the command
  if(!strupncmp(argv[0], "default", CMDLEN) && argc == 1) {
    newMethod = DEFAULT_ATOMREP;
  } else if(!strupncmp(argv[0], AtomRepName[LINES], CMDLEN) && argc < 3) {
    newMethod = LINES;
    if(argc > 1)
      newThick = atoi(argv[1]);
  } else if(!strupncmp(argv[0], AtomRepName[BONDS], CMDLEN) && argc < 4) {
    newMethod = BONDS;
    if(argc > 1)
      newBondRad = atof(argv[1]);
    if(argc > 2)
      newBondRes = atoi(argv[2]);
  } else if(!strupncmp(argv[0], AtomRepName[CPK], CMDLEN) && argc < 6) {
    newMethod = CPK;
    if(argc > 1)
      newSphRad = atof(argv[1]);
    if(argc > 2)
      newBondRad = atof(argv[2]);
    if(argc > 3)
      newSphRes = atoi(argv[3]);
    if(argc > 4)
      newBondRes = atoi(argv[4]);
  } else if(!strupncmp(argv[0], AtomRepName[POINTS], CMDLEN) && argc < 2) {
    newMethod = POINTS;
  } else if(!strupncmp(argv[0], AtomRepName[VDW], CMDLEN) && argc < 4) {
    newMethod = VDW;
    if(argc > 1)
      newSphRad = atof(argv[1]);
    if(argc > 2)
      newSphRes = atoi(argv[2]);
  } else if(!strupncmp(argv[0], AtomRepName[DOTTED], CMDLEN) && argc < 4) {
    newMethod = DOTTED;
    if(argc > 1)
      newSphRad = atof(argv[1]);
    if(argc > 2)
      newSphRes = atoi(argv[2]);
  } else if(!strupncmp(argv[0], AtomRepName[LICORICE], CMDLEN) && argc < 5) {
    newMethod = LICORICE;
    if(argc > 1)
      newBondRad = atof(argv[1]);
    if(argc > 2)
      newSphRes = atoi(argv[2]);
    if(argc > 3)
      newBondRes = atoi(argv[3]);
  } else if(!strupncmp(argv[0], AtomRepName[RIBBONS], CMDLEN) && argc < 5) {
     newMethod = RIBBONS;
     if (argc > 1)
       newBondRad = atof(argv[1]);
     if (argc > 2)
       newBondRes = atoi(argv[2]);
     if (argc > 3)
       newThick = atoi(argv[3]);

  } else if(!strupncmp(argv[0], AtomRepName[TUBE], CMDLEN) && argc < 4) {
     newMethod = TUBE;
     if (argc > 1)
       newBondRad = atof(argv[1]);
     if (argc > 2)
       newBondRes = atoi(argv[2]);
#ifdef VMDALPHA_SHAPE
  } else if(!strupncmp(argv[0], AtomRepName[ALPHASHAPE], CMDLEN) && argc < 3) {
     newMethod = ALPHASHAPE;
     if (argc > 1)
       newSphRad = atof(argv[1]);
#endif
  } else if(!strupncmp(argv[0], AtomRepName[REPOFF], CMDLEN) && argc == 1) {
    newMethod = REPOFF;
  } else {
    // unknown representation
    ok = FALSE;
  }
  
  // check if error
  if(!ok) {
    msgErr << "Incorrect atom representation command '" << newcmd << "'";
    msgErr << sendmsg;
  } else {
    // command was successful, save new settings
     repMethod = newMethod;
     if ( sphereRad != newSphRad ||
	  bondRad != newBondRad  ||
	  sphereRes != newSphRes ||
	  bondRes != newBondRes  ||
	  lineThickness != newThick ) {
	option_changed = TRUE;
     }
     sphereRad = newSphRad;
     bondRad = newBondRad;
     sphereRes = newSphRes;
     bondRes = newBondRes;
     lineThickness = newThick;
     strcpy(cmdStr, newcmd);
  }

  // delete parsing space
  delete [] cmdStrTok;
  
  return ok;
}


//////////////////////////  public routines

// equal operator, to change the current settings.
AtomRep& AtomRep::operator=(const AtomRep &ar) {

  // copy values
//   msgInfo << "At the = operator" << sendmsg;
//   msgInfo << "alpha " << (alpha != NULL) << sendmsg;
//   msgInfo << "ar.alpha " << (ar.alpha != NULL) << sendmsg;
   
  strcpy(cmdStr, ar.cmdStr);
  repMethod = ar.repMethod;
  sphereRad = ar.sphereRad;
  bondRad = ar.bondRad;
  sphereRes = ar.sphereRes;
  bondRes = ar.bondRes;
  option_changed = FALSE;

  lineThickness = ar.lineThickness;
  
  return *this;
}

